/*
 *  linux/fs/tcfs/inode.c
 *
 *  Copyright (C) 1992  Rick Sladkey
 *
 *  tcfs inode and superblock handling functions
 *
 *  Modularised by Alan Cox <Alan.Cox@linux.org>, while hacking some
 *  experimental TCFS changes. Modularisation taken straight from SYS5 fs.
 *
 *  Change to tcfs_read_super() to permit TCFS mounts to multi-homed hosts.
 *  J.S.Peatfield@damtp.cam.ac.uk
 *
 */

#define __NO_VERSION__
#include <linux/config.h>
#include <linux/module.h>

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/locks.h>
#include <linux/unistd.h>
#include <linux/sunrpc/clnt.h>
#include <linux/sunrpc/stats.h>
#include <linux/lockd/bind.h>

#include <linux/tcfs_fs.h>
#include <linux/tcfs/hash.h>
#include <linux/tcfs/tcfs_library.h>

#include <asm/system.h>
#include <asm/uaccess.h>

extern int tcfs_debug;

#define CONFIG_TCFS_SNAPSHOT 1
#define TCFSDBG_FACILITY		TCFSDBG_VFS
#define TCFS_PARANOIA 1

#if 0
#undef dprintk
#undef dfprintk
#define dprintk		printk
#define dfprintk(fac, args...)	printk(## args)
#endif

extern int init_procfs (void);
extern int cleanup_procfs (void);

static struct inode * __tcfs_fhget(struct super_block *, struct tcfs_fattr *,
				struct tcfs_fh *);
static void tcfs_zap_caches(struct inode *);
static void tcfs_invalidate_inode(struct inode *);

static void tcfs_read_inode(struct inode *);
static void tcfs_put_inode(struct inode *);
static void tcfs_delete_inode(struct inode *);
static int  tcfs_notify_change(struct dentry *, struct iattr *);
static void tcfs_put_super(struct super_block *);
static void tcfs_umount_begin(struct super_block *);
static int  tcfs_statfs(struct super_block *, struct statfs *, int);

static struct super_operations tcfs_sops = { 
	tcfs_read_inode,		/* read inode */
	NULL,			/* write inode */
	tcfs_put_inode,		/* put inode */
	tcfs_delete_inode,	/* delete inode */
	tcfs_notify_change,	/* notify change */
	tcfs_put_super,		/* put superblock */
	NULL,			/* write superblock */
	tcfs_statfs,		/* stat filesystem */
	NULL,			/* no remount */
	NULL,			/* no clear inode */
	tcfs_umount_begin	/* umount attempt begin */
};

struct rpc_stat			tcfs_rpcstat = { &tcfs_program };

/*
 * The "read_inode" function doesn't actually do anything:
 * the real data is filled in later in tcfs_fhget. Here we
 * just mark the cache times invalid, and zero out i_mode
 * (the latter makes "tcfs_refresh_inode" do the right thing
 * wrt pipe inodes)
 */
static void
tcfs_read_inode(struct inode * inode)
{
	inode->i_blksize = inode->i_sb->s_blocksize;
	inode->i_mode = 0;
	inode->i_rdev = 0;
	inode->i_op = NULL;

	memset (&(inode->u.tcfs_i.tcfsheader), 0,
			sizeof (struct tcfs_inode_header));
	inode->u.tcfs_i.cached=0;

	TCFS_CACHEINV(inode);
	TCFS_ATTRTIMEO(inode) = TCFS_MINATTRTIMEO(inode);
}

static void
tcfs_put_inode(struct inode * inode)
{
	dprintk("TCFS: put_inode(%x/%ld)\n", inode->i_dev, inode->i_ino);
	/*
	 * We want to get rid of unused inodes ...
	 */
	if (inode->i_count == 1)
		inode->i_nlink = 0;
}

static void
tcfs_delete_inode(struct inode * inode)
{
	int failed;

	dprintk("TCFS: delete_inode(%x/%ld)\n", inode->i_dev, inode->i_ino);
	/*
	 * Flush out any pending write requests ...
	 */
	if (TCFS_WRITEBACK(inode) != NULL) {
		unsigned long timeout = jiffies + 5*HZ;
#ifdef TCFS_DEBUG_VERBOSE
printk("tcfs_delete_inode: inode %ld has pending RPC requests\n", inode->i_ino);
#endif
		tcfs_inval(inode);
		while (TCFS_WRITEBACK(inode) != NULL &&
		       time_before(jiffies, timeout)) {
			current->state = TASK_INTERRUPTIBLE;
			schedule_timeout(HZ/10);
		}
		current->state = TASK_RUNNING;
		if (TCFS_WRITEBACK(inode) != NULL)
			printk("TCFS: Arghhh, stuck RPC requests!\n");
	}

	failed = tcfs_check_failed_request(inode);
	if (failed)
		printk("TCFS: inode %ld had %d failed requests\n",
			inode->i_ino, failed);
	clear_inode(inode);
}

void
tcfs_put_super(struct super_block *sb)
{
	struct tcfs_server *server = &sb->u.tcfs_sb.s_server;
	struct rpc_clnt	*rpc;

	if ((rpc = server->client) != NULL)
		rpc_shutdown_client(rpc);

	if (!(server->flags & TCFS_MOUNT_NONLM))
		lockd_down();	/* release rpc.lockd */
	rpciod_down();		/* release rpciod */
	/*
	 * Invalidate the dircache for this superblock.
	 */
	tcfs_invalidate_dircache_sb(sb);

	kfree(server->hostname);

	MOD_DEC_USE_COUNT;
}

void
tcfs_umount_begin(struct super_block *sb)
{
	struct tcfs_server *server = &sb->u.tcfs_sb.s_server;
	struct rpc_clnt	*rpc;

	/* -EIO all pending I/O */
	if ((rpc = server->client) != NULL)
		rpc_killall_tasks(rpc);
}

/*
 * Compute and set TCFS server blocksize
 */
/* TCFS TODO:
	puo' darsi che sia utile ricalcolare il blocksize in funzione
	anche dell'hash di TCFS. Verificare.
*/
static unsigned int
tcfs_block_size(unsigned int bsize, unsigned char *nrbitsp)
{
	if (bsize < 1024)
		bsize = TCFS_DEF_FILE_IO_BUFFER_SIZE;
	else if (bsize >= TCFS_MAX_FILE_IO_BUFFER_SIZE)
		bsize = TCFS_MAX_FILE_IO_BUFFER_SIZE;

	/* make sure blocksize is a power of two */
	if ((bsize & (bsize - 1)) || nrbitsp) {
		unsigned int	nrbits;

		for (nrbits = 31; nrbits && !(bsize & (1 << nrbits)); nrbits--)
			;
		bsize = 1 << nrbits;
		if (nrbitsp)
			*nrbitsp = nrbits;
		if (bsize < TCFS_DEF_FILE_IO_BUFFER_SIZE)
			bsize = TCFS_DEF_FILE_IO_BUFFER_SIZE;
	}

	return bsize;
}

/*
 * The way this works is that the mount process passes a structure
 * in the data argument which contains the server's IP address
 * and the root file handle obtained from the server's mount
 * daemon. We stash these away in the private superblock fields.
 */
struct super_block *
tcfs_read_super(struct super_block *sb, void *raw_data, int silent)
{
	struct tcfs_mount_data	*data = (struct tcfs_mount_data *) raw_data;
	struct tcfs_server	*server;
	struct rpc_xprt		*xprt;
	struct rpc_clnt		*clnt;
	struct tcfs_fh		*root_fh;
	struct inode		*root_inode;
	unsigned int		authflavor;
	int			tcp;
	struct sockaddr_in	srvaddr;
	struct rpc_timeout	timeparms;
	struct tcfs_fattr	fattr;

	MOD_INC_USE_COUNT;
	if (!data)
		goto out_miss_args;

	if (data->version != TCFS_MOUNT_VERSION) {
		printk("tcfs warning: mount version %s than kernel\n",
			data->version < TCFS_MOUNT_VERSION ? "older" : "newer");
		if (data->version < 2)
			data->namlen = 0;
		if (data->version < 3)
			data->bsize  = 0;
	}

	/* We now require that the mount process passes the remote address */
	memcpy(&srvaddr, &data->addr, sizeof(srvaddr));
	if (srvaddr.sin_addr.s_addr == INADDR_ANY)
		goto out_no_remote;

	lock_super(sb);

	sb->s_flags |= MS_ODD_RENAME; /* This should go away */

	sb->s_magic      = TCFS_SUPER_MAGIC;
	sb->s_op         = &tcfs_sops;
	sb->s_blocksize  = tcfs_block_size(data->bsize, &sb->s_blocksize_bits);
	sb->u.tcfs_sb.s_root = data->root;
	server           = &sb->u.tcfs_sb.s_server;
	server->rsize    = tcfs_block_size(data->rsize, NULL);
	server->wsize    = tcfs_block_size(data->wsize, NULL);
	server->flags    = data->flags;

	if (data->flags & TCFS_MOUNT_NOAC) {
		data->acregmin = data->acregmax = 0;
		data->acdirmin = data->acdirmax = 0;
	}
	server->acregmin = data->acregmin*HZ;
	server->acregmax = data->acregmax*HZ;
	server->acdirmin = data->acdirmin*HZ;
	server->acdirmax = data->acdirmax*HZ;

	server->hostname = kmalloc(strlen(data->hostname) + 1, GFP_KERNEL);
	if (!server->hostname)
		goto out_unlock;
	strcpy(server->hostname, data->hostname);

	/* Which protocol do we use? */
	tcp   = (data->flags & TCFS_MOUNT_TCP);

	/* Initialize timeout values */
	timeparms.to_initval = data->timeo * HZ / 10;
	timeparms.to_retries = data->retrans;
	timeparms.to_maxval  = tcp? RPC_MAX_TCP_TIMEOUT : RPC_MAX_UDP_TIMEOUT;
	timeparms.to_exponential = 1;

	/* Now create transport and client */
	xprt = xprt_create_proto(tcp? IPPROTO_TCP : IPPROTO_UDP,
						&srvaddr, &timeparms);
	if (xprt == NULL)
		goto out_no_xprt;

	/* Choose authentication flavor */
	authflavor = RPC_AUTH_UNIX;
	if (data->flags & TCFS_MOUNT_SECURE)
		authflavor = RPC_AUTH_DES;
	else if (data->flags & TCFS_MOUNT_KERBEROS)
		authflavor = RPC_AUTH_KRB;

	clnt = rpc_create_client(xprt, server->hostname, &tcfs_program,
						TCFS_VERSION, authflavor);
	if (clnt == NULL)
		goto out_no_client;

	clnt->cl_intr     = (data->flags & TCFS_MOUNT_INTR)? 1 : 0;
	clnt->cl_softrtry = (data->flags & TCFS_MOUNT_SOFT)? 1 : 0;
	clnt->cl_chatty   = 1;
	server->client    = clnt;

	/* Fire up rpciod if not yet running */
	if (rpciod_up() != 0)
		goto out_no_iod;

	/*
	 * Keep the super block locked while we try to get 
	 * the root fh attributes.
	 */
	root_fh = kmalloc(sizeof(struct tcfs_fh), GFP_KERNEL);
	if (!root_fh)
		goto out_no_fh;
	*root_fh = data->root;

	if (tcfs_proc_getattr(server, root_fh, &fattr) != 0)
		goto out_no_fattr;

	root_inode = __tcfs_fhget(sb, &fattr, NULL);
	if (!root_inode)
		goto out_no_root;
	sb->s_root = d_alloc_root(root_inode, NULL);
	if (!sb->s_root)
		goto out_no_root;
	sb->s_root->d_op = &tcfs_dentry_operations;
	sb->s_root->d_fsdata = root_fh;
	root_inode->u.tcfs_i.fhandle=(struct tcfs_fh *)kmalloc (sizeof(struct tcfs_fh), GFP_KERNEL);
	*(root_inode->u.tcfs_i.fhandle)=*(struct tcfs_fh *)root_fh;

	/* We're airborne */
	unlock_super(sb);

	/* Check whether to start the lockd process */
	if (!(server->flags & TCFS_MOUNT_NONLM))
		lockd_up();
	return sb;

	/* Yargs. It didn't work out. */
out_no_root:
	printk("tcfs_read_super: get root inode failed\n");
	iput(root_inode);
	goto out_free_fh;

out_no_fattr:
	printk("tcfs_read_super: get root fattr failed\n");
out_free_fh:
	kfree(root_fh);
out_no_fh:
	rpciod_down();
	goto out_shutdown;

out_no_iod:
	printk(KERN_WARNING "TCFS: couldn't start rpciod!\n");
out_shutdown:
	rpc_shutdown_client(server->client);
	goto out_free_host;

out_no_client:
	printk(KERN_WARNING "TCFS: cannot create RPC client.\n");
	xprt_destroy(xprt);
	goto out_free_host;

out_no_xprt:
	printk(KERN_WARNING "TCFS: cannot create RPC transport.\n");

out_free_host:
	kfree(server->hostname);
out_unlock:
	unlock_super(sb);
	goto out_fail;

out_no_remote:
	printk("TCFS: mount program didn't pass remote address!\n");
	goto out_fail;

out_miss_args:
	printk("tcfs_read_super: missing data argument\n");

out_fail:
	sb->s_dev = 0;
	MOD_DEC_USE_COUNT;
	return NULL;
}

static int
tcfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
	int error;
	struct tcfs_fsinfo res;
	struct statfs tmp;

	error = tcfs_proc_statfs(&sb->u.tcfs_sb.s_server, &sb->u.tcfs_sb.s_root,
		&res);
	if (error) {
		printk("tcfs_statfs: statfs error = %d\n", -error);
		res.bsize = res.blocks = res.bfree = res.bavail = 0;
	}
	tmp.f_type = TCFS_SUPER_MAGIC;
	tmp.f_bsize = res.bsize;
	tmp.f_blocks = res.blocks;
	tmp.f_bfree = res.bfree;
	tmp.f_bavail = res.bavail;
	tmp.f_files = 0;
	tmp.f_ffree = 0;
	tmp.f_namelen = NAME_MAX;
	return copy_to_user(buf, &tmp, bufsiz) ? -EFAULT : 0;
}

/*
 * Free all unused dentries in an inode's alias list.
 *
 * Subtle note: we have to be very careful not to cause
 * any IO operations with the stale dentries, as this
 * could cause file corruption. But since the dentry
 * count is 0 and all pending IO for a dentry has been
 * flushed when the count went to 0, we're safe here.
 * Also returns the number of unhashed dentries
 */
static int
tcfs_free_dentries(struct inode *inode)
{
	struct list_head *tmp, *head = &inode->i_dentry;
	int unhashed;

restart:
	tmp = head;
	unhashed = 0;
	while ((tmp = tmp->next) != head) {
		struct dentry *dentry = list_entry(tmp, struct dentry, d_alias);
		if (!list_empty(&dentry->d_subdirs))
			shrink_dcache_parent(dentry);
		dprintk("tcfs_free_dentries: found %s/%s, d_count=%d, hashed=%d\n",
			dentry->d_parent->d_name.name, dentry->d_name.name,
			dentry->d_count, !list_empty(&dentry->d_hash));
		if (!dentry->d_count) {
			dget(dentry);
			d_drop(dentry);
			dput(dentry);
			goto restart;
		}
		if (list_empty(&dentry->d_hash))
			unhashed++;
	}
	return unhashed;
}

/*
 * Invalidate the local caches
 */
static void
tcfs_zap_caches(struct inode *inode)
{
	TCFS_ATTRTIMEO(inode) = TCFS_MINATTRTIMEO(inode);
	TCFS_CACHEINV(inode);

	if (S_ISDIR(inode->i_mode))
		tcfs_invalidate_dircache(inode);
	else
		invalidate_inode_pages(inode);
}

/*
 * Invalidate, but do not unhash, the inode
 */
static void
tcfs_invalidate_inode(struct inode *inode)
{
	umode_t save_mode = inode->i_mode;

	make_bad_inode(inode);
	inode->i_mode = save_mode;
	tcfs_inval(inode);
	tcfs_zap_caches(inode);
}

/*
 * Fill in inode information from the fattr.
 */
static void
tcfs_fill_inode(struct inode *inode, struct tcfs_fattr *fattr)
{
	/*
	 * Check whether the mode has been set, as we only want to
	 * do this once. (We don't allow inodes to change types.)
	 */
	if (inode->i_mode == 0) {
		inode->i_mode = fattr->mode;
		if (S_ISREG(inode->i_mode))
			inode->i_op = &tcfs_file_inode_operations;
		else if (S_ISDIR(inode->i_mode))
			inode->i_op = &tcfs_dir_inode_operations;
		else if (S_ISLNK(inode->i_mode))
			inode->i_op = &tcfs_symlink_inode_operations;
		else if (S_ISCHR(inode->i_mode)) {
			inode->i_op = &chrdev_inode_operations;
			inode->i_rdev = to_kdev_t(fattr->rdev);
		} else if (S_ISBLK(inode->i_mode)) {
			inode->i_op = &blkdev_inode_operations;
			inode->i_rdev = to_kdev_t(fattr->rdev);
		} else if (S_ISFIFO(inode->i_mode))
			init_fifo(inode);
		else
			inode->i_op = NULL;
		/*
		 * Preset the size and mtime, as there's no need
		 * to invalidate the caches.
		 */ 
		inode->i_size  = fattr->size;
		inode->i_mtime = fattr->mtime.seconds;
		TCFS_OLDMTIME(inode) = fattr->mtime.seconds;
	}
	tcfs_refresh_inode(inode, fattr);
}

/*
 * This is our own version of iget that looks up inodes by file handle
 * instead of inode number.  We use this technique instead of using
 * the vfs read_inode function because there is no way to pass the
 * file handle or current attributes into the read_inode function.
 *
 * We provide a special check for NetApp .snapshot directories to avoid
 * inode aliasing problems. All snapshot inodes are anonymous (unhashed).
 */
struct inode *
tcfs_fhget(struct dentry *dentry, struct tcfs_fh *fhandle,
				 struct tcfs_fattr *fattr)
{
	struct super_block *sb = dentry->d_sb;

	dprintk("TCFS: tcfs_fhget(%s/%s fileid=%d)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name,
		fattr->fileid);

	/* Install the file handle in the dentry */
	*((struct tcfs_fh *) dentry->d_fsdata) = *fhandle;

	return __tcfs_fhget(sb, fattr, fhandle);
}

/*
 * Look up the inode by super block and fattr->fileid.
 *
 * Note carefully the special handling of busy inodes (i_count > 1).
 * With the kernel 2.1.xx dcache all inodes except hard links must
 * have i_count == 1 after iget(). Otherwise, it indicates that the
 * server has reused a fileid (i_ino) and we have a stale inode.
 */
static struct inode *
__tcfs_fhget(struct super_block *sb, struct tcfs_fattr *fattr, struct tcfs_fh *fh)
{
	struct inode *inode;
	int max_count, stale_inode, unhashed = 0;

retry:
	inode = iget(sb, fattr->fileid);
	if (!inode)
		goto out_no_inode;
	/* N.B. This should be impossible ... */
	if (inode->i_ino != fattr->fileid)
		goto out_bad_id;

	/*
	 * Check for busy inodes, and attempt to get rid of any
	 * unused local references. If successful, we release the
	 * inode and try again.
	 *
	 * Note that the busy test uses the values in the fattr,
	 * as the inode may have become a different object.
	 * (We can probably handle modes changes here, too.)
	 */
	stale_inode = inode->i_mode &&
		      ((fattr->mode ^ inode->i_mode) & S_IFMT);
	stale_inode |= inode->i_count && inode->i_count == unhashed;
	max_count = S_ISDIR(fattr->mode) ? 1 : fattr->nlink;
	if (stale_inode || inode->i_count > max_count + unhashed) {
		dprintk("__tcfs_fhget: inode %ld busy, i_count=%d, i_nlink=%d\n",
			inode->i_ino, inode->i_count, inode->i_nlink);
		unhashed = tcfs_free_dentries(inode);
		if (stale_inode || inode->i_count > max_count + unhashed) {
			printk("__tcfs_fhget: inode %ld still busy, i_count=%d\n",
				inode->i_ino, inode->i_count);
			if (!list_empty(&inode->i_dentry)) {
				struct dentry *dentry;
				dentry = list_entry(inode->i_dentry.next,
						 struct dentry, d_alias);
				printk("__tcfs_fhget: killing %s/%s filehandle\n",
					dentry->d_parent->d_name.name,
					dentry->d_name.name);
				memset(dentry->d_fsdata, 0,
					sizeof(struct tcfs_fh));
			}
			remove_inode_hash(inode);
			tcfs_invalidate_inode(inode);
			unhashed = 0;
		}
		iput(inode);
		goto retry;
	}

	/* TCFS hack. Ora posso copiare il file handler per la TCFS_FH. */
	if (fh) {
		if (!inode->u.tcfs_i.fhandle) {
			inode->u.tcfs_i.fhandle=(struct tcfs_fh *)kmalloc(sizeof(struct tcfs_fh), GFP_KERNEL);
			if (!inode->u.tcfs_i.fhandle)
				return inode;
		}

		*inode->u.tcfs_i.fhandle = *fh;
	}

	tcfs_fill_inode(inode, fattr);

	dprintk("TCFS: __tcfs_fhget(%x/%ld ct=%d)\n",
		inode->i_dev, inode->i_ino, inode->i_count);

out:
	return inode;

out_no_inode:
	printk("__tcfs_fhget: iget failed\n");
	goto out;
out_bad_id:
	printk("__tcfs_fhget: unexpected inode from iget\n");
	goto out;
}

int
tcfs_notify_change(struct dentry *dentry, struct iattr *attr)
{
	struct inode *inode = dentry->d_inode;
	int error;
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	void *ks=NULL;
	struct tcfs_header tcfsheader;

	dprintk ("TCFS notify_change\n");
	ks=tcfs_get_key (inode);

	if (ks && ks!=TCFS_NO_SECURE) {
		tcfs_get_header (inode, &tcfsheader, ks);
	}
	/*
	 * Make sure the inode is up-to-date.
	 */
	error = tcfs_revalidate(dentry);
	if (error) {
#ifdef TCFS_PARANOIA
printk("tcfs_notify_change: revalidate failed, error=%d\n", error);
#endif
		goto out;
	}

	sattr.mode = (u32) -1;
	if (attr->ia_valid & ATTR_MODE) 
		sattr.mode = attr->ia_mode;

	sattr.uid = (u32) -1;
	if (attr->ia_valid & ATTR_UID)
		sattr.uid = attr->ia_uid;

	sattr.gid = (u32) -1;
	if (attr->ia_valid & ATTR_GID)
		sattr.gid = attr->ia_gid;

	sattr.size = (u32) -1;
	if ((attr->ia_valid & ATTR_SIZE) && S_ISREG(inode->i_mode))
		sattr.size = attr->ia_size;

	sattr.mtime.seconds = sattr.mtime.useconds = (u32) -1;
	if (attr->ia_valid & ATTR_MTIME) {
		sattr.mtime.seconds = attr->ia_mtime;
		sattr.mtime.useconds = 0;
	}

	sattr.atime.seconds = sattr.atime.useconds = (u32) -1;
	if (attr->ia_valid & ATTR_ATIME) {
		sattr.atime.seconds = attr->ia_atime;
		sattr.atime.useconds = 0;
	}

	error = tcfs_wb_all(inode);
	if (error)
		goto out;

	error = tcfs_proc_setattr(TCFS_DSERVER(dentry), TCFS_FH(dentry),
				&sattr, &fattr);
	if (error)
		goto out;
	/*
	 * If we changed the size or mtime, update the inode
	 * now to avoid invalidating the page cache.
	 */
	if (sattr.size != (u32) -1) {
		if (sattr.size != fattr.size)
			printk("tcfs_notify_change: sattr=%d, fattr=%d??\n",
				sattr.size, fattr.size);
		inode->i_size  = sattr.size;
		inode->i_mtime = fattr.mtime.seconds;
	}

	/*
	TCFS TODO:
		controllare il 3o IF
	*/
	if (ks && ks!=TCFS_NO_SECURE) {
		if (attr->ia_valid & ATTR_SIZE) {
			if (S_ISREG(inode->i_mode)) {
				tcfsheader.size=S_ISREG(inode->i_mode) ?
					sattr.size :
					inode->i_size;
			}
		}

		tcfs_put_header (NULL, dentry, &tcfsheader, ks);
	}

	if (sattr.mtime.seconds != (u32) -1)
		inode->i_mtime = fattr.mtime.seconds;
	error = tcfs_refresh_inode(inode, &fattr);
out:
	return error;
}

/*
 * Externally visible revalidation function
 */
int
tcfs_revalidate(struct dentry *dentry)
{
	return tcfs_revalidate_inode(TCFS_DSERVER(dentry), dentry);
}

/*
 * These are probably going to contain hooks for
 * allocating and releasing RPC credentials for
 * the file. I'll have to think about Tronds patch
 * a bit more..
 */
int tcfs_open(struct inode *inode, struct file *filp)
{
	return 0;
}

int tcfs_release(struct inode *inode, struct file *filp)
{
	return 0;
}

/*
 * This function is called whenever some part of TCFS notices that
 * the cached attributes have to be refreshed.
 */
int
_tcfs_revalidate_inode(struct tcfs_server *server, struct dentry *dentry)
{
	struct inode	*inode = dentry->d_inode;
	int		 status = 0;
	struct tcfs_fattr fattr;

	dfprintk(PAGECACHE, "TCFS: revalidating %s/%s, ino=%ld\n",
		dentry->d_parent->d_name.name, dentry->d_name.name,
		inode->i_ino);
	status = tcfs_proc_getattr(server, TCFS_FH(dentry), &fattr);
	
	if (status) {
		int error;
		u32 *fh;
		struct tcfs_fh fhandle;
			dfprintk(PAGECACHE, "tcfs_revalidate_inode: %s/%s getattr failed, ino=%ld, error=%d\n",
				dentry->d_parent->d_name.name,
				dentry->d_name.name, inode->i_ino, status);
			if (status != -ESTALE)
				goto out;
			/*
			 * A "stale filehandle" error ... show the current fh
			 * and find out what the filehandle should be.
			 */
			fh = (u32 *) TCFS_FH(dentry);
			dfprintk(PAGECACHE, "TCFS: bad fh %08x%08x%08x%08x%08x%08x%08x%08x\n",
				fh[0],fh[1],fh[2],fh[3],fh[4],fh[5],fh[6],fh[7]);
			error = tcfs_proc_lookup(server, TCFS_FH(dentry->d_parent), 
						dentry->d_name.name, &fhandle, &fattr);

			if (error) {
				dfprintk(PAGECACHE, "TCFS: lookup failed, error=%d\n", error);
				goto out;
			}
			fh = (u32 *) &fhandle;
			dfprintk(PAGECACHE, "            %08x%08x%08x%08x%08x%08x%08x%08x\n",
				fh[0],fh[1],fh[2],fh[3],fh[4],fh[5],fh[6],fh[7]);
			goto out;
		}

		if (dentry->d_fsdata) {
			if (!inode->u.tcfs_i.fhandle) {
				inode->u.tcfs_i.fhandle=(struct tcfs_fh *)kmalloc (sizeof(struct tcfs_fh), GFP_KERNEL);
				if (!inode->u.tcfs_i.fhandle)
					goto out;
			}
			*inode->u.tcfs_i.fhandle = *(struct tcfs_fh *)dentry->d_fsdata;
		}

		status = tcfs_refresh_inode(inode, &fattr);
		if (status) {
			dfprintk(PAGECACHE, "tcfs_revalidate_inode: %s/%s refresh failed, ino=%ld, error=%d\n",
				dentry->d_parent->d_name.name,
				dentry->d_name.name, inode->i_ino, status);
			goto out;
		}
	dfprintk(PAGECACHE, "TCFS: %s/%s revalidation complete\n",
		dentry->d_parent->d_name.name, dentry->d_name.name);
	
out:
	return status;
}

/*
 * Many tcfs protocol calls return the new file attributes after
 * an operation.  Here we update the inode to reflect the state
 * of the server's inode.
 *
 * This is a bit tricky because we have to make sure all dirty pages
 * have been sent off to the server before calling invalidate_inode_pages.
 * To make sure no other process adds more write requests while we try
 * our best to flush them, we make them sleep during the attribute refresh.
 *
 * A very similar scenario holds for the dir cache.
 */
int
tcfs_refresh_inode(struct inode *inode, struct tcfs_fattr *fattr)
{
	int invalid = 0;
	int error = -EIO;
	void *ks=NULL;
	
	dfprintk(VFS, "TCFS: refresh_inode(%x/%ld ct=%d)\n",
		 inode->i_dev, inode->i_ino, inode->i_count);

	if (!inode || !fattr) {
		goto out;
	}
	if (inode->i_ino != fattr->fileid) {
		printk("tcfs_refresh_inode: mismatch, ino=%ld, fattr=%d\n",
			inode->i_ino, fattr->fileid);
		goto out;
	}

	/*
	 * Make sure the inode's type hasn't changed.
	 */
	if ((inode->i_mode & S_IFMT) != (fattr->mode & S_IFMT))
		goto out_changed;

	inode->i_mode = fattr->mode;
	inode->i_nlink = fattr->nlink;
	inode->i_uid = fattr->uid;
	inode->i_gid = fattr->gid;

	inode->i_blocks = fattr->blocks;
	inode->i_atime = fattr->atime.seconds;
	inode->i_ctime = fattr->ctime.seconds;

	ks=tcfs_get_key (inode);

	/*
	 * Update the read time so we don't revalidate too often.
	 */
	TCFS_READTIME(inode) = jiffies;
	error = 0;

	/*
	 * If we have pending write-back entries, we don't want
	 * to look at the size or the mtime the server sends us
	 * too closely, as we're in the middle of modifying them.
	 */
	if (TCFS_WRITEBACK(inode))
		goto out;

	if (inode->i_size != fattr->size) {
#ifdef TCFS_DEBUG_VERBOSE
printk("TCFS: size change on %x/%ld\n", inode->i_dev, inode->i_ino);
#endif
		inode->i_size = fattr->size;
		invalid = 1;
	}

	if (ks && ks!=TCFS_NO_SECURE) {
		struct tcfs_header tcfsheader;

		tcfs_get_header (inode, &tcfsheader, ks);
		inode->i_size=inode->u.tcfs_i.tcfsheader.size;
	} else if (!ks) {
		inode->i_size=0;
	}

	if (inode->i_mtime != fattr->mtime.seconds) {
#ifdef TCFS_DEBUG_VERBOSE
printk("TCFS: mtime change on %x/%ld\n", inode->i_dev, inode->i_ino);
#endif
		inode->i_mtime = fattr->mtime.seconds;
		invalid = 1;
	}

	if (invalid)
		goto out_invalid;

	/* Update attrtimeo value */
	if (fattr->mtime.seconds == TCFS_OLDMTIME(inode)) {
		if ((TCFS_ATTRTIMEO(inode) <<= 1) > TCFS_MAXATTRTIMEO(inode))
			TCFS_ATTRTIMEO(inode) = TCFS_MAXATTRTIMEO(inode);
	}
	TCFS_OLDMTIME(inode) = fattr->mtime.seconds;

out:
	return error;

out_changed:
	/*
	 * Big trouble! The inode has become a different object.
	 */
#ifdef TCFS_PARANOIA
printk("tcfs_refresh_inode: inode %ld mode changed, %07o to %07o\n",
inode->i_ino, inode->i_mode, fattr->mode);
#endif
	/*
	 * No need to worry about unhashing the dentry, as the
	 * lookup validation will know that the inode is bad.
	 */
	tcfs_invalidate_inode(inode);
	goto out;

out_invalid:
#ifdef TCFS_DEBUG_VERBOSE
	printk("tcfs_refresh_inode: invalidating %ld pages\n", inode->i_nrpages);
	#endif
		tcfs_zap_caches(inode);
		goto out;
	}

	/*
	 * File system information
	 */
	static struct file_system_type tcfs_fs_type = {
		"tcfs",
		0 /* FS_NO_DCACHE - this doesn't work right now*/,
		tcfs_read_super,
		NULL
	};

	/*
	 * Initialize TCFS
	 */
	int
	init_tcfs_fs(void)
	{
	#if 0
	#ifdef CONFIG_PROC_FS
		rpc_register_sysctl();
		rpc_proc_init();
		rpc_proc_register(&tcfs_rpcstat);
	#endif
	#endif

		init_hash ();

	#ifdef CONFIG_PROC_FS
		init_procfs ();
	#endif

		return register_filesystem(&tcfs_fs_type);
	}

	#ifdef MODULE
	/*
	   I'm not the real author (i.e. Ermelindo Mauriello <ermmau@tcfs.unisa.it>),
	   but I did more of the work.
	*/
	MODULE_AUTHOR ("Aniello Del Sorbo <anidel@tcfs.unisa.it>");

	void
	cleanup_tcfs_fs (void)
	{
		unregister_filesystem (&tcfs_fs_type);
		tcfs_free_dircache ();

	#ifdef CONFIG_PROC_FS
		cleanup_procfs ();
	#endif

		cleanup_hash ();
	}
#endif
