/*
 *  linux/fs/tcfs/dir.c
 *
 *  Copyright (C) 1992  Rick Sladkey
 *
 *  tcfs directory handling functions
 *
 * 10 Apr 1996	Added silly rename for unlink	--okir
 * 28 Sep 1996	Improved directory cache --okir
 * 23 Aug 1997  Claus Heine claus@momo.math.rwth-aachen.de 
 *              Re-implemented silly rename for unlink, newly implemented
 *              silly rename for tcfs_rename() following the suggestions
 *              of Olaf Kirch (okir) found in this file.
 *              Following Linus comments on my original hack, this version
 *              depends only on the dcache stuff and doesn't touch the inode
 *              layer (iput() and friends).
 * xx Mar 2000  Added TCFS code. --anidel
 */

#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/stat.h>
#include <linux/fcntl.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/sunrpc/types.h>

#include <asm/segment.h>	/* for fs functions */

#include <linux/tcfs_fs.h>
#include <linux/tcfs.h>
#include <linux/tcfs/tcfs_library.h>
#include <linux/tcfs/hash.h>
#include <linux/tcfs/cipherinterface.h>

extern int tcfs_debug;
#define TCFS_PARANOIA 1
#define TCFS_DEBUG_VERBOSE 1
#define TCFSDBG_FACILITY	TCFSDBG_ALL

#if 0
#undef dprintk
#undef dfprintk
#define dprintk		printk
#define dfprintk(fac, args...)	printk(## args)
#endif
/*
 * Head for a dircache entry. Currently still very simple; when
 * the cache grows larger, we will need a LRU list.
 */
struct tcfs_dirent {
	dev_t			dev;		/* device number */
	ino_t			ino;		/* inode number */
	u32			cookie;		/* cookie of first entry */
	unsigned short		valid  : 1,	/* data is valid */
				locked : 1;	/* entry locked */
	unsigned int		size;		/* # of entries */
	unsigned long		age;		/* last used */
	unsigned long		mtime;		/* last attr stamp */
	struct wait_queue *	wait;
	__u32 *			entry;		/* three __u32's per entry */
};

extern int tcfs_ioctl (struct inode *, struct file *, unsigned int,
			unsigned long);
static int tcfs_safe_remove(struct dentry *);

static ssize_t tcfs_dir_read(struct file *, char *, size_t, loff_t *);
static int tcfs_readdir(struct file *, void *, filldir_t);
static struct dentry *tcfs_lookup(struct inode *, struct dentry *);
static int tcfs_create(struct inode *, struct dentry *, int);
static int tcfs_mkdir(struct inode *, struct dentry *, int);
static int tcfs_rmdir(struct inode *, struct dentry *);
static int tcfs_unlink(struct inode *, struct dentry *);
static int tcfs_symlink(struct inode *, struct dentry *, const char *);
static int tcfs_link(struct dentry *, struct inode *, struct dentry *);
static int tcfs_mknod(struct inode *, struct dentry *, int, int);
static int tcfs_rename(struct inode *, struct dentry *,
		      struct inode *, struct dentry *);

static struct file_operations tcfs_dir_operations = {
	NULL,			/* lseek - default */
	tcfs_dir_read,		/* read - bad */
	NULL,			/* write - bad */
	tcfs_readdir,		/* readdir */
	NULL,			/* select - default */
	tcfs_ioctl,		/* ioctl */
	NULL,			/* mmap */
	tcfs_open,		/* open */
	NULL,			/* flush */
	tcfs_release,		/* release */
	NULL			/* fsync */
};

struct inode_operations tcfs_dir_inode_operations = {
	&tcfs_dir_operations,	/* default directory file-ops */
	tcfs_create,		/* create */
	tcfs_lookup,		/* lookup */
	tcfs_link,		/* link */
	tcfs_unlink,		/* unlink */
	tcfs_symlink,		/* symlink */
	tcfs_mkdir,		/* mkdir */
	tcfs_rmdir,		/* rmdir */
	tcfs_mknod,		/* mknod */
	tcfs_rename,		/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL,			/* permission */
	NULL,			/* smap */
	NULL,			/* updatepage */
	tcfs_revalidate,		/* revalidate */
};

static ssize_t
tcfs_dir_read(struct file *filp, char *buf, size_t count, loff_t *ppos)
{
	return -EISDIR;
}

static struct tcfs_dirent	dircache[TCFS_MAX_DIRCACHE];

/*
 * We need to do caching of directory entries to prevent an
 * incredible amount of RPC traffic.  Only the most recent open
 * directory is cached.  This seems sufficient for most purposes.
 * Technically, we ought to flush the cache on close but this is
 * not a problem in practice.
 *
 * XXX: Do proper directory caching by stuffing data into the
 * page cache (may require some fiddling for rsize < PAGE_SIZE).
 */

static int tcfs_readdir(struct file *filp, void *dirent, filldir_t filldir)
{
	struct dentry 		*dentry = filp->f_dentry;
	struct inode 		*inode = dentry->d_inode;
	static struct wait_queue *readdir_wait = NULL;
	struct wait_queue	**waitp = NULL;
	struct tcfs_dirent	*cache, *free;
	unsigned long		age, dead;
	u32			cookie;
	int			ismydir, result;
	int			i, j, index = 0;
	__u32			*entry;
	char			*name, *start;
	void			*ks=NULL;

	dfprintk(VFS, "TCFS: tcfs_readdir(%s/%s)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name);

	result = tcfs_revalidate_inode(TCFS_DSERVER(dentry), dentry);

	if (result < 0)
		goto out;

	/*
	 * Try to find the entry in the cache
	 */
again:
	if (waitp) {
		interruptible_sleep_on(waitp);
		if (signal_pending(current)) {
			return -ERESTARTSYS;
		}
		waitp = NULL;
	}

	cookie = filp->f_pos;
	entry  = NULL;
	free   = NULL;
	age    = ~(unsigned long) 0;
	dead   = jiffies - TCFS_ATTRTIMEO(inode);

	for (i = 0, cache = dircache; i < TCFS_MAX_DIRCACHE; i++, cache++) {
		/*
		dprintk("TCFS: dircache[%d] valid %d locked %d\n",
					i, cache->valid, cache->locked);
		 */
		ismydir = (cache->dev == inode->i_dev
				&& cache->ino == inode->i_ino);
		if (cache->locked) {
			if (!ismydir || cache->cookie != cookie)
				continue;
			dfprintk(DIRCACHE, "TCFS: waiting on dircache entry\n");
			waitp = &cache->wait;
			goto again;
		}

		if (ismydir && cache->mtime != inode->i_mtime)
			cache->valid = 0;

		if (!cache->valid || cache->age < dead) {
			free = cache;
			age  = 0;
		} else if (cache->age < age) {
			free = cache;
			age  = cache->age;
		}

		if (!ismydir || !cache->valid)
			continue;

		if (cache->cookie == cookie && cache->size > 0) {
			entry = cache->entry + (index = 0);
			cache->locked = 1;
			break;
		}
		for (j = 0; j < cache->size; j++) {
			__u32 *this_ent = cache->entry + j*3;

			if (*(this_ent+1) != cookie)
				continue;
			if (j < cache->size - 1) {
				index = j + 1;
				entry = this_ent + 3;
			} else if (*(this_ent+2) & (1 << 15)) {
				/* eof */
				return 0;
			}
			break;
		}
		if (entry) {
			dfprintk(DIRCACHE, "TCFS: found dircache entry %d\n",
						(int)(cache - dircache));
			cache->locked = 1;
			break;
		}
	}

	/*
	 * Okay, entry not present in cache, or locked and inaccessible.
	 * Set up the cache entry and attempt a READDIR call.
	 */
	if (entry == NULL) {
		if ((cache = free) == NULL) {
			dfprintk(DIRCACHE, "TCFS: dircache contention\n");
			waitp = &readdir_wait;
			goto again;
		}
		dfprintk(DIRCACHE, "TCFS: using free dircache entry %d\n",
				(int)(free - dircache));
		cache->cookie = cookie;
		cache->locked = 1;
		cache->valid  = 0;
		cache->dev    = inode->i_dev;
		cache->ino    = inode->i_ino;
		if (!cache->entry) {
			result = -ENOMEM;
			cache->entry = (__u32 *) get_free_page(GFP_KERNEL);
			if (!cache->entry)
				goto done;
		}

		result = tcfs_proc_readdir(TCFS_SERVER(inode), TCFS_FH(dentry),
					cookie, PAGE_SIZE, cache->entry);

		if (result <= 0)
			goto done;
		cache->size  = result;
		cache->valid = 1;
		entry = cache->entry + (index = 0);
	}
	cache->mtime = inode->i_mtime;
	cache->age = jiffies;

	if (!(ks=tcfs_get_key (inode))) {
		result=-EACCES;
		goto done;
	}

	/*
	 * Yowza! We have a cache entry...
	 */
	start = (char *) cache->entry;
	while (index < cache->size) {
		__u32	fileid  = *entry++;
		__u32	nextpos = *entry++; /* cookie */
		__u32	length  = *entry++;
		int	k	= length&0x7FFF;
		char	tcfsname[k+1];

		/*
		 * Unpack the eof flag, offset, and length
		 */
		result = length & (1 << 15); /* eof flag */
		name = start + ((length >> 16) & 0xFFFF);

		if (ks!=TCFS_NO_SECURE) {
			/* The directory is secure or shared */
			/* int k=length&0x7FFF;
			char tcfsname[k+1];*/

			memcpy (tcfsname, name, k);
			tcfsname[k]='\0';

			if (!strcmp (tcfsname, ".tcfs")) {
				cookie = nextpos;
				index++;
				continue;
			}

			if (strcmp (tcfsname, ".") && strcmp (tcfsname, "..")) {
				k=tcfsdecode(tcfsname, k+1);
				tcfs_decrypt (tcfsname,(k&0xFFF8)+8,ks,NULL);
				name=tcfsname;
			}

			length=k;
		} else {
			length &= 0x7FFF;
		}

		/*
		dprintk("TCFS: filldir(%p, %.*s, %d, %d, %x, eof %x)\n", entry,
				(int) length, name, length,
				(unsigned int) filp->f_pos,
				fileid, result);
		*/

		if (filldir(dirent, name, length, cookie, fileid) < 0)
			break;
		cookie = nextpos;
		index++;
	}
	filp->f_pos = cookie;
	result = 0;

	/* XXX: May want to kick async readdir-ahead here. Not too hard
	 * to do. */

done:
	dfprintk(DIRCACHE, "TCFS: tcfs_readdir complete\n");
	cache->locked = 0;
	wake_up(&cache->wait);
	wake_up(&readdir_wait);

out:
	dprintk ("TCFS: tcfs_readdir complete\n");
	return result;
}

/*
 * Invalidate dircache entries for an inode.
 */
void
tcfs_invalidate_dircache(struct inode *inode)
{
	struct tcfs_dirent *cache = dircache;
	dev_t		dev = inode->i_dev;
	ino_t		ino = inode->i_ino;
	int		i;

	dfprintk(DIRCACHE, "TCFS: invalidate dircache for %x/%ld\n", dev, (long)ino);
	for (i = TCFS_MAX_DIRCACHE; i--; cache++) {
		if (cache->ino != ino)
			continue;
		if (cache->dev != dev)
			continue;
		if (cache->locked) {
			printk("TCFS: cache locked for %s/%ld\n",
				kdevname(dev), (long) ino);
			continue;
		}
		cache->valid = 0;	/* brute force */
	}
}

/*
 * Invalidate the dircache for a super block (or all caches),
 * and release the cache memory.
 */
void
tcfs_invalidate_dircache_sb(struct super_block *sb)
{
	struct tcfs_dirent *cache = dircache;
	int		i;

	for (i = TCFS_MAX_DIRCACHE; i--; cache++) {
		if (sb && sb->s_dev != cache->dev)
			continue;
		if (cache->locked) {
			printk("TCFS: cache locked at umount %s\n",
				(cache->entry ? "(lost a page!)" : ""));
			continue;
		}
		cache->valid = 0;	/* brute force */
		if (cache->entry) {
			free_page((unsigned long) cache->entry);
			cache->entry = NULL;
		}
	}
}

/*
 * Free directory cache memory
 * Called from cleanup_module
 */
void
tcfs_free_dircache(void)
{
	dfprintk(DIRCACHE, "TCFS: freeing dircache\n");
	tcfs_invalidate_dircache_sb(NULL);
}

/*
 * Whenever an TCFS operation succeeds, we know that the dentry
 * is valid, so we update the revalidation timestamp.
 */
static inline void tcfs_renew_times(struct dentry * dentry)
{
	dentry->d_time = jiffies;
}

static inline int tcfs_dentry_force_reval(struct dentry *dentry, int flags)
{
	struct inode *inode = dentry->d_inode;
	unsigned long timeout = TCFS_ATTRTIMEO(inode);

	/*
	 * If it's the last lookup in a series, we use a stricter
	 * cache consistency check by looking at the parent mtime.
	 *
	 * If it's been modified in the last hour, be really strict.
	 * (This still means that we can avoid doing unnecessary
	 * work on directories like /usr/share/bin etc which basically
	 * never change).
	 */
	if (!(flags & LOOKUP_CONTINUE)) {
		long diff = CURRENT_TIME - dentry->d_parent->d_inode->i_mtime;

		if (diff < 15*60)
			timeout = 0;
	}
	
	return time_after(jiffies,dentry->d_time + timeout);
}

/*
 * We judge how long we want to trust negative
 * dentries by looking at the parent inode mtime.
 *
 * If mtime is close to present time, we revalidate
 * more often.
 */
static inline int tcfs_neg_need_reval(struct dentry *dentry)
{
	unsigned long timeout = 30 * HZ;
	long diff = CURRENT_TIME - dentry->d_parent->d_inode->i_mtime;

	if (diff < 5*60)
		timeout = 1 * HZ;

	return time_after(jiffies, dentry->d_time + timeout);
}

/*
 * This is called every time the dcache has a lookup hit,
 * and we should check whether we can really trust that
 * lookup.
 *
 * NOTE! The hit can be a negative hit too, don't assume
 * we have an inode!
 *
 * If the dentry is older than the revalidation interval, 
 * we do a new lookup and verify that the dentry is still
 * correct.
 */
static int tcfs_lookup_revalidate(struct dentry * dentry, int flags)
{
	struct dentry * parent = dentry->d_parent;
	struct inode * inode = dentry->d_inode;
	int error;
	struct tcfs_fh fhandle;
	struct tcfs_fattr fattr;
	void *ks=NULL;
	char *cryptedname=NULL;

	/*
	 * If we don't have an inode, let's look at the parent
	 * directory mtime to get a hint about how often we
	 * should validate things..
	 */
	if (!inode) {
		if (tcfs_neg_need_reval(dentry))
			goto out_bad;
		goto out_valid;
	}

	if (is_bad_inode(inode)) {
		dfprintk(VFS, "tcfs_lookup_validate: %s/%s has dud inode\n",
			parent->d_name.name, dentry->d_name.name);
		goto out_bad;
	}

	if (IS_ROOT(dentry))
		goto out_valid;

	if (!tcfs_dentry_force_reval(dentry, flags))
		goto out_valid;

	if (!(ks=tcfs_get_key (inode))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (dentry->d_name.name, ".") &&
			strcmp (dentry->d_name.name, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k & 0xFFF8)+8, ks, NULL);
				tcfsencode (cryptedname,
						(k & 0xFFF8)+8);
		}
	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	/*
	 * Do a new lookup and check the dentry attributes.
	 */
	error = tcfs_proc_lookup(TCFS_DSERVER(parent), TCFS_FH(parent),
				cryptedname, &fhandle, &fattr);

	/* Anidel TODO: leggere gli attributi TCFS della dir se e' sicura.
			e' qui che dobbiamo leggere il file .tcfs */

	if (error)
		goto out_bad;

	/* Inode number matches? */
	if (fattr.fileid != inode->i_ino)
		goto out_bad;

	/* Filehandle matches? */
	if (memcmp(dentry->d_fsdata, &fhandle, sizeof(struct tcfs_fh))) {
		if (!list_empty(&dentry->d_subdirs))
			shrink_dcache_parent(dentry);
		if (dentry->d_count < 2)
			goto out_bad;
	}

	/* Ok, remeber that we successfully checked it.. */
	tcfs_renew_times(dentry);
	tcfs_refresh_inode(inode, &fattr);

out_valid:
	if (cryptedname) {
		kfree (cryptedname);
	}

	return 1;
out_bad:
	if (dentry->d_parent->d_inode)
		tcfs_invalidate_dircache(dentry->d_parent->d_inode);
	if (inode && S_ISDIR(inode->i_mode))
		tcfs_invalidate_dircache(inode);

	if (cryptedname) {
		kfree (cryptedname);
	}

	return 0;
}

/*
 * This is called from dput() when d_count is going to 0.
 * We use it to clean up silly-renamed files.
 */
static void tcfs_dentry_delete(struct dentry *dentry)
{
	dfprintk(VFS, "TCFS: dentry_delete(%s/%s, %x)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name,
		dentry->d_flags);

	if (dentry->d_flags & DCACHE_NFSFS_RENAMED) {
		int error;
		
		dentry->d_flags &= ~DCACHE_NFSFS_RENAMED;
		/* Unhash it first */
		d_drop(dentry);
		error = tcfs_safe_remove(dentry);
		if (error)
			printk("TCFS: can't silly-delete %s/%s, error=%d\n",
				dentry->d_parent->d_name.name,
				dentry->d_name.name, error);
	}

}

/*
 * Called when the dentry is being freed to release private memory.
 */
static void tcfs_dentry_release(struct dentry *dentry)
{
	if (dentry->d_fsdata)
		kfree(dentry->d_fsdata);
}

struct dentry_operations tcfs_dentry_operations = {
	tcfs_lookup_revalidate,	/* d_revalidate(struct dentry *, int) */
	NULL,			/* d_hash */
	NULL,			/* d_compare */
	tcfs_dentry_delete,	/* d_delete(struct dentry *) */
	tcfs_dentry_release,	/* d_release(struct dentry *) */
	NULL			/* d_iput */
};

#ifdef TCFS_PARANOIA
/*
 * Display all dentries holding the specified inode.
 */
static void show_dentry(struct list_head * dlist)
{
	struct list_head *tmp = dlist;

	while ((tmp = tmp->next) != dlist) {
		struct dentry * dentry = list_entry(tmp, struct dentry, d_alias);
		const char * unhashed = "";

		if (list_empty(&dentry->d_hash))
			unhashed = "(unhashed)";

		printk("show_dentry: %s/%s, d_count=%d%s\n",
			dentry->d_parent->d_name.name,
			dentry->d_name.name, dentry->d_count,
			unhashed);
	}
}
#endif

static struct dentry *tcfs_lookup(struct inode *dir, struct dentry * dentry)
{
	struct inode *inode;
	int error;
	struct tcfs_fh fhandle;
	struct tcfs_fattr fattr;
	void *ks=NULL;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: lookup(%s/%s)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name);

	error = -ENAMETOOLONG;
	if (dentry->d_name.len > TCFS_MAXNAMLEN)
		goto out;

	error = -ENOMEM;
	if (!dentry->d_fsdata) {
		dentry->d_fsdata = kmalloc(sizeof(struct tcfs_fh), GFP_KERNEL);
		if (!dentry->d_fsdata)
			goto out;
		if (dentry->d_inode)
			if (!dentry->d_inode->u.tcfs_i.fhandle) {
				dentry->d_inode->u.tcfs_i.fhandle=(struct tcfs_fh *)kmalloc (sizeof(struct tcfs_fh), GFP_KERNEL);
				if (!dentry->d_inode->u.tcfs_i.fhandle)
					goto out;
			}
	}

	dentry->d_op = &tcfs_dentry_operations;

	if (!(ks=tcfs_get_key(dir))) {
		error = -EACCES;
		goto out;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
			}
	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_lookup(TCFS_SERVER(dir), TCFS_FH(dentry->d_parent), 
				cryptedname, &fhandle, &fattr);

	inode = NULL;

	if (error == -ENOENT) {
		goto no_entry;
	}

	if (!error) {
		error = -EACCES;
		inode = tcfs_fhget(dentry, &fhandle, &fattr);

		if (inode) {

	    no_entry:
			d_add(dentry, inode);
			tcfs_renew_times(dentry);
			error = 0;
		}
	}

out:

	if (cryptedname) {
		kfree (cryptedname);
	}

	return ERR_PTR(error);
}

/*
 * Code common to create, mkdir, and mknod.
 */
int tcfs_instantiate(struct dentry *dentry, struct tcfs_fh *fhandle,
				struct tcfs_fattr *fattr)
{
	struct inode *inode;
	int error = -EACCES;

	inode = tcfs_fhget(dentry, fhandle, fattr);

	if (inode) {
		if (!inode->u.tcfs_i.fhandle) {
			inode->u.tcfs_i.fhandle=(struct tcfs_fh *)kmalloc(
				sizeof (struct tcfs_fh),
				GFP_KERNEL);
			if (!inode->u.tcfs_i.fhandle)
				return -ENOMEM;
		}

		if (fhandle) {
			*inode->u.tcfs_i.fhandle = *fhandle;
		}

		inode->u.tcfs_i.cached=0;

		d_instantiate(dentry, inode);
		tcfs_renew_times(dentry);
		error = 0;
	}
	return error;
}

/*
 * Following a failed create operation, we drop the dentry rather
 * than retain a negative dentry. This avoids a problem in the event
 * that the operation succeeded on the server, but an error in the
 * reply path made it appear to have failed.
 */
static int tcfs_create(struct inode *dir, struct dentry *dentry, int mode)
{
	int error;
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	void *ks=NULL;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: create(%x/%ld, %s\n",
		dir->i_dev, dir->i_ino, dentry->d_name.name);

	sattr.mode = mode;
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;

	/*
	 * Invalidate the dir cache before the operation to avoid a race.
	 */
	tcfs_invalidate_dircache(dir);

	if (!(ks=tcfs_get_key (dir))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
			}
	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_create(TCFS_SERVER(dir), TCFS_FH(dentry->d_parent),
			cryptedname, &sattr, &fhandle, &fattr);
	/* Anidel.
	Ora ho, per la prima volta, il file handler (fhandle) che dovrei
 	installare in inode->u.tcfs_i.fhandle (per la macro TCFS_FH che
	usiamo in tcfs_get_key. Il problema e' proprio l'inode.
	L'inode me lo da, in tcfs_instantiate, una chiamata alla
	tcfs_fhget. La tcfs_fhget, pero', chiama indirettamente la
	tcfs_refresh_inode, che a sua volta chiama la tcfs_get_key.
	Questa deve leggere l'header TCFS, ed ha bisogno del file
	handler, che ancora non ho installato nell'inode.
	Al server TCFS arriva quindi uno stale file handler.
	*/

	/* Anidel TODO: a questo punto si deve creare l'header TCFS e
	                aggiornare gli hash in .tcfs
	*/

	if (!error)
		error = tcfs_instantiate(dentry, &fhandle, &fattr);

	if (error)
		d_drop(dentry);

	if (ks!=TCFS_NO_SECURE && !error) {
		struct tcfs_header tcfsheader;

		memset (&tcfsheader, 0, sizeof (struct tcfs_header));

		strncpy (tcfsheader.start, "TCFS", 4);
		strcpy (tcfsheader.engine, "default");
		tcfsheader.size=fattr.size;
		tcfsheader.flags = dir->u.tcfs_i.tcfsheader.flags;
		tcfs_put_header (NULL, dentry, &tcfsheader, ks);
	}

	if (cryptedname) {
		kfree (cryptedname);
	}

	return error;
}

/*
 * See comments for tcfs_proc_create regarding failed operations.
 */
static int tcfs_mknod(struct inode *dir, struct dentry *dentry, int mode, int rdev)
{
	int error;
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	void *ks;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: mknod(%x/%ld, %s\n",
		dir->i_dev, dir->i_ino, dentry->d_name.name);

	sattr.mode = mode;
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	if (S_ISCHR(mode) || S_ISBLK(mode))
		sattr.size = rdev; /* get out your barf bag */
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;

	tcfs_invalidate_dircache(dir);

	if (!(ks=tcfs_get_key (dir))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
			}
	
	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_create(TCFS_SERVER(dir), TCFS_FH(dentry->d_parent),
				cryptedname, &sattr, &fhandle, &fattr);
	if (!error)
		error = tcfs_instantiate(dentry, &fhandle, &fattr);
	if (error)
		d_drop(dentry);
	
	if (cryptedname) {
		kfree (cryptedname);
	}

	return error;
}

/*
 * See comments for tcfs_proc_create regarding failed operations.
 */
static int tcfs_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	int error;
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	void *ks;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: mkdir(%x/%ld, %s\n",
		dir->i_dev, dir->i_ino, dentry->d_name.name);

	sattr.mode = mode | S_IFDIR;
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;

	/*
	 * Always drop the dentry, we can't always depend on
	 * the fattr returned by the server (AIX seems to be
	 * broken). We're better off doing another lookup than
	 * depending on potentially bogus information.
	 */
	d_drop(dentry);
	tcfs_invalidate_dircache(dir);

	if (!(ks=tcfs_get_key (dir))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
			}
	
	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_mkdir(TCFS_DSERVER(dentry), TCFS_FH(dentry->d_parent),
				cryptedname, &sattr, &fhandle, &fattr);

	if (cryptedname) {
		kfree (cryptedname);
	}

	if (ks!=TCFS_NO_SECURE) {
		struct tcfs_header	tcfsheader;

		tcfs_invalidate_dircache(dir);
		error=tcfs_get_header (dir, &tcfsheader, ks);
		if (error<0)
			return error;

		tcfs_invalidate_dircache (dir);

		error=tcfs_instantiate (dentry, &fhandle, &fattr);
		if (error)
			return error;

		strncpy (tcfsheader.start, "TCFS", 4);
		tcfsheader.size=dentry->d_inode->i_size;
		tcfsheader.flags = dir->u.tcfs_i.tcfsheader.flags;

		error=tcfs_put_header(NULL, dentry, &tcfsheader, ks);
		if (error)
			return error;
	}

	return error;
}

static int tcfs_rmdir(struct inode *dir, struct dentry *dentry)
{
	int error;
	void *ks;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: rmdir(%x/%ld, %s\n",
		dir->i_dev, dir->i_ino, dentry->d_name.name);

	tcfs_invalidate_dircache(dir);

	if (!(ks=tcfs_get_key (dentry->d_inode))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		error = tcfs_proc_remove(TCFS_SERVER(dir), TCFS_FH(dentry),
				".tcfs");

		tcfs_invalidate_dircache (dir);
		/* ATTENZIONE ATTENZIONE:
		   com'e' ora la tcfs_rmdir e' sbagliata.
		   Se si esegue 'rmdir dir' e 'dir' e' una directory
		   cifrata contenente files, il .tcfs viene cancellato,
		   ma la directory no!!!!
		   Soluzione possibile: se la proc_rmdir da errore ricreare
		   il .tcfs */
	}

	if (!(ks=tcfs_get_key (dir))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".")
			&& strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
			}
	
	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_rmdir(TCFS_SERVER(dir), TCFS_FH(dentry->d_parent),
				cryptedname);

	/* Update i_nlink and invalidate dentry. */
	if (!error) {
		d_drop(dentry);
		if (dentry->d_inode->i_nlink)
			dentry->d_inode->i_nlink --;
	}

	if (cryptedname) {
		kfree (cryptedname);
	}

	return error;
}


/*  Note: we copy the code from lookup_dentry() here, only: we have to
 *  omit the directory lock. We are already the owner of the lock when
 *  we reach here. And "down(&dir->i_sem)" would make us sleep forever
 *  ('cause WE have the lock)
 * 
 *  VERY IMPORTANT: calculate the hash for this dentry!!!!!!!!
 *  Otherwise the cached lookup DEFINITELY WILL fail. And a new dentry
 *  is created. Without the DCACHE_NFSFS_RENAMED flag. And with d_count
 *  == 1. And trouble.
 *
 *  Concerning my choice of the temp name: it is just nice to have
 *  i_ino part of the temp name, as this offers another check whether
 *  somebody attempts to remove the "silly renamed" dentry itself.
 *  Which is something that I consider evil. Your opinion may vary.
 *  BUT:
 *  Now that I compute the hash value right, it should be possible to simply
 *  check for the DCACHE_NFSFS_RENAMED flag in dentry->d_flag instead of
 *  doing the string compare.
 *  WHICH MEANS:
 *  This offers the opportunity to shorten the temp name. Currently, I use
 *  the hex representation of i_ino + an event counter. This sums up to
 *  as much as 36 characters for a 64 bit machine, and needs 20 chars on 
 *  a 32 bit machine.
 *  QUINTESSENCE
 *  The use of i_ino is simply cosmetic. All we need is a unique temp
 *  file name for the .tcfs files. The event counter seemed to be adequate.
 *  And as we retry in case such a file already exists, we are guaranteed
 *  to succeed.
 */

static
struct dentry *tcfs_silly_lookup(struct dentry *parent, char *silly, int slen)
{
	struct qstr    sqstr;
	struct dentry *sdentry;
	struct dentry *res;

	sqstr.name = silly;
	sqstr.len  = slen;
	sqstr.hash = full_name_hash(silly, slen);
	sdentry = d_lookup(parent, &sqstr);
	if (!sdentry) {
		sdentry = d_alloc(parent, &sqstr);
		if (sdentry == NULL)
			return ERR_PTR(-ENOMEM);
		res = tcfs_lookup(parent->d_inode, sdentry);
		if (res) {
			dput(sdentry);
			return res;
		}
	}
	return sdentry;
}

static int tcfs_sillyrename(struct inode *dir, struct dentry *dentry)
{
	static unsigned int sillycounter = 0;
	const int      i_inosize  = sizeof(dir->i_ino)*2;
	const int      countersize = sizeof(sillycounter)*2;
	int      slen       = strlen(".tcfs") + i_inosize + countersize;
	char           silly[slen+1];
	struct dentry *sdentry;
	int            error = -EIO;
	void	       *ks=NULL;
	char	       *cryptedname=NULL, *cryptednamesilly=NULL;

	dfprintk(VFS, "TCFS: silly-rename(%s/%s, ct=%d)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name, 
		dentry->d_count);

	/*
	 * Note that a silly-renamed file can be deleted once it's
	 * no longer in use -- it's just an ordinary file now.
	 */
	if (dentry->d_count == 1) {
		dentry->d_flags &= ~DCACHE_NFSFS_RENAMED;
		goto out;  /* No need to silly rename. */
	}

#ifdef TCFS_PARANOIA
if (!dentry->d_inode)
printk("TCFS: silly-renaming %s/%s, negative dentry??\n",
dentry->d_parent->d_name.name, dentry->d_name.name);
#endif
	/*
	 * We don't allow a dentry to be silly-renamed twice.
	 */
	error = -EBUSY;
	if (dentry->d_flags & DCACHE_NFSFS_RENAMED)
		goto out;

	sprintf(silly, ".tcfs%*.*lx",
		i_inosize, i_inosize, dentry->d_inode->i_ino);
	
	sdentry = NULL;
	do {
		char *suffix = silly + slen - countersize;

		dput(sdentry);
		sillycounter++;
		sprintf(suffix, "%*.*x", countersize, countersize, sillycounter);

		if (!(ks=tcfs_get_key(dir))) {
			error=-EACCES;
			goto out;
		}

		if (ks!=TCFS_NO_SECURE) {
			int len=strlen (silly);

			cryptednamesilly=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
			memset (cryptednamesilly, 0, TCFS_MAXNAMLEN);
			memcpy (cryptednamesilly, silly, len);
			cryptednamesilly[len]=0;

			if (strcmp (cryptednamesilly, ".") &&
				strcmp (cryptednamesilly, "..")) {
					int k=len+1;
					tcfs_encrypt (cryptednamesilly,
							(k&0xFFF8)+8,ks,NULL);
					tcfsencode(cryptednamesilly,(k&0xFFF8)+8);
			}
	
		} else {
			cryptednamesilly=kmalloc (strlen(silly)+1, GFP_KERNEL);
			memcpy (cryptednamesilly, silly, strlen(silly));
			cryptednamesilly[strlen(silly)]=0;
		}

		slen=strlen(cryptednamesilly);

		dfprintk(VFS, "trying to rename %s to %s\n",
			 dentry->d_name.name, silly);
		
		sdentry = tcfs_silly_lookup(dentry->d_parent, cryptednamesilly, slen);
		/*
		 * N.B. Better to return EBUSY here ... it could be
		 * dangerous to delete the file while it's in use.
		 */
		if (IS_ERR(sdentry))
			goto out;
	} while(sdentry->d_inode != NULL); /* need negative lookup */

	tcfs_invalidate_dircache(dir);

/*
	if (!(ks=tcfs_get_key(dir))) {
		return -EACCES;
	}
*/

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
		}

	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_rename(TCFS_SERVER(dir),
				TCFS_FH(dentry->d_parent), cryptedname,
				TCFS_FH(dentry->d_parent), cryptednamesilly);
	if (!error) {
		tcfs_renew_times(dentry);
		d_move(dentry, sdentry);
		dentry->d_flags |= DCACHE_NFSFS_RENAMED;
 		/* If we return 0 we don't unlink */
	}
	dput(sdentry);
out:

	if (cryptedname) {
		kfree (cryptedname);
	}

	if (cryptednamesilly) {
		kfree (cryptednamesilly);
	}

	return error;
}

/*
 * Remove a file after making sure there are no pending writes,
 * and after checking that the file has only one user. 
 *
 * We update inode->i_nlink and free the inode prior to the operation
 * to avoid possible races if the server reuses the inode.
 */
static int tcfs_safe_remove(struct dentry *dentry)
{
	struct inode *dir = dentry->d_parent->d_inode;
	struct inode *inode = dentry->d_inode;
	int error, rehash = 0;
	void *ks=NULL;
	char *cryptedname=NULL;
		
	dfprintk(VFS, "TCFS: safe_remove(%s/%s, %ld)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name,
		inode->i_ino);

	/* N.B. not needed now that d_delete is done in advance? */
	error = -EBUSY;
	if (!inode) {
#ifdef TCFS_PARANOIA
printk("tcfs_safe_remove: %s/%s already negative??\n",
dentry->d_parent->d_name.name, dentry->d_name.name);
#endif
	}

	if (dentry->d_count > 1) {
#ifdef TCFS_PARANOIA
printk("tcfs_safe_remove: %s/%s busy, d_count=%d\n",
dentry->d_parent->d_name.name, dentry->d_name.name, dentry->d_count);
#endif
		goto out;
	}
	/*
	 * Unhash the dentry while we remove the file ...
	 */
	if (!list_empty(&dentry->d_hash)) {
		d_drop(dentry);
		rehash = 1;
	}
	/*
	 * Update i_nlink and free the inode before unlinking.
	 */
	if (inode) {
		if (inode->i_nlink)
			inode->i_nlink --;
		d_delete(dentry);
	}
	tcfs_invalidate_dircache(dir);
	
	if (!(ks=tcfs_get_key(dir))) {
		error=-EACCES;
		goto out;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
		}

	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_remove(TCFS_SERVER(dir), TCFS_FH(dentry->d_parent),
				cryptedname);
	/*
	 * Rehash the negative dentry if the operation succeeded.
	 */
	if (!error && rehash)
		d_add(dentry, NULL);
out:
	if (cryptedname) {
		kfree (cryptedname);
	}

	return error;
}

/*  We do silly rename. In case sillyrename() returns -EBUSY, the inode
 *  belongs to an active ".tcfs..." file and we return -EBUSY.
 *
 *  If sillyrename() returns 0, we do nothing, otherwise we unlink.
 */
static int tcfs_unlink(struct inode *dir, struct dentry *dentry)
{
	int error;

	dfprintk(VFS, "TCFS: unlink(%x/%ld, %s)\n",
		dir->i_dev, dir->i_ino, dentry->d_name.name);

	error = tcfs_sillyrename(dir, dentry);
	if (error && error != -EBUSY) {
		error = tcfs_safe_remove(dentry);
		if (!error) {
			tcfs_renew_times(dentry);
		}
	}
	return error;
}

static int
tcfs_symlink(struct inode *dir, struct dentry *dentry, const char *symname)
{
	struct tcfs_sattr sattr;
	int error;
	void *ks=NULL;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: symlink(%x/%ld, %s, %s)\n",
		dir->i_dev, dir->i_ino, dentry->d_name.name, symname);

	error = -ENAMETOOLONG;
	if (strlen(symname) > TCFS_MAXPATHLEN)
		goto out;

#ifdef TCFS_PARANOIA
if (dentry->d_inode)
printk("tcfs_proc_symlink: %s/%s not negative!\n",
dentry->d_parent->d_name.name, dentry->d_name.name);
#endif
	/*
	 * Fill in the sattr for the call.
 	 * Note: SunOS 4.1.2 crashes if the mode isn't initialized!
	 */
	sattr.mode = S_IFLNK | S_IRWXUGO;
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;

	/*
	 * Drop the dentry in advance to force a new lookup.
	 * Since tcfs_proc_symlink doesn't return a fattr, we
	 * can't instantiate the new inode.
	 */
	d_drop(dentry);
	tcfs_invalidate_dircache(dir);
	
	if (!(ks=tcfs_get_key(dir))) {
		error=-EACCES;
		goto out;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
		}

	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_symlink(TCFS_SERVER(dir), TCFS_FH(dentry->d_parent),
				cryptedname, symname, &sattr);
	if (!error) {
		tcfs_renew_times(dentry->d_parent);
	} else if (error == -EEXIST) {
		printk("tcfs_proc_symlink: %s/%s already exists??\n",
			dentry->d_parent->d_name.name, dentry->d_name.name);
	}

out:
	if (cryptedname) {
		kfree (cryptedname);
	}

	return error;
}

static int 
tcfs_link(struct dentry *old_dentry, struct inode *dir, struct dentry *dentry)
{
	struct inode *inode = old_dentry->d_inode;
	int error;
	void *ks=NULL;
	char *cryptedname=NULL;

	dfprintk(VFS, "TCFS: link(%s/%s -> %s/%s)\n",
		old_dentry->d_parent->d_name.name, old_dentry->d_name.name,
		dentry->d_parent->d_name.name, dentry->d_name.name);

	/*
	 * Drop the dentry in advance to force a new lookup.
	 * Since tcfs_proc_link doesn't return a file handle,
	 * we can't use the existing dentry.
	 */
	d_drop(dentry);
	tcfs_invalidate_dircache(dir);
	
	if (!(ks=tcfs_get_key(dir))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (cryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;

		if (strcmp (cryptedname, ".") &&
			strcmp (cryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (cryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(cryptedname,(k&0xFFF8)+8);
		}

	} else {
		int len=dentry->d_name.len;

		cryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (cryptedname, dentry->d_name.name, len);
		cryptedname[len]=0;
	}

	error = tcfs_proc_link(TCFS_DSERVER(old_dentry), TCFS_FH(old_dentry),
				TCFS_FH(dentry->d_parent), cryptedname);
	if (!error) {
 		/*
		 * Update the link count immediately, as some apps
		 * (e.g. pine) test this after making a link.
		 */
		inode->i_nlink++;
	}

	if (cryptedname) {
		kfree (cryptedname);
	}

	return error;
}

/*
 * RENAME
 * FIXME: Some tcfsds, like the Linux user space tcfsd, may generate a
 * different file handle for the same inode after a rename (e.g. when
 * moving to a different directory). A fail-safe method to do so would
 * be to look up old_dir/old_name, create a link to new_dir/new_name and
 * rename the old file using the sillyrename stuff. This way, the original
 * file in old_dir will go away when the last process iput()s the inode.
 *
 * FIXED.
 * 
 * It actually works quite well. One needs to have the possibility for
 * at least one ".tcfs..." file in each directory the file ever gets
 * moved or linked to which happens automagically with the new
 * implementation that only depends on the dcache stuff instead of
 * using the inode layer
 *
 * Unfortunately, things are a little more complicated than indicated
 * above. For a cross-directory move, we want to make sure we can get
 * rid of the old inode after the operation.  This means there must be
 * no pending writes (if it's a file), and the use count must be 1.
 * If these conditions are met, we can drop the dentries before doing
 * the rename.
 */
static int tcfs_rename(struct inode *old_dir, struct dentry *old_dentry,
		      struct inode *new_dir, struct dentry *new_dentry)
{
	struct inode *old_inode = old_dentry->d_inode;
	struct inode *new_inode = new_dentry->d_inode;
	struct dentry *dentry = NULL;
	int error, rehash = 0;
	void *ks=NULL;
	char *oldcryptedname=NULL, *newcryptedname=NULL;

	dfprintk(VFS, "TCFS: rename(%s/%s -> %s/%s, ct=%d)\n",
		old_dentry->d_parent->d_name.name, old_dentry->d_name.name,
		new_dentry->d_parent->d_name.name, new_dentry->d_name.name,
		new_dentry->d_count);

	/*
	 * First check whether the target is busy ... we can't
	 * safely do _any_ rename if the target is in use.
	 *
	 * For files, make a copy of the dentry and then do a 
	 * silly-rename. If the silly-rename succeeds, the
	 * copied dentry is hashed and becomes the new target.
	 *
	 * With directories check is done in VFS.
	 */
	error = -EBUSY;
	if (new_dentry->d_count > 1 && new_inode) {
		int err;
		/* copy the target dentry's name */
		dentry = d_alloc(new_dentry->d_parent,
				 &new_dentry->d_name);
		if (!dentry)
			goto out;

		/* silly-rename the existing target ... */
		err = tcfs_sillyrename(new_dir, new_dentry);
		if (!err) {
			new_dentry = dentry;
			new_inode = NULL;
			/* hash the replacement target */
			d_add(new_dentry, NULL);
		}

		/* dentry still busy? */
		if (new_dentry->d_count > 1) {
#ifdef TCFS_PARANOIA
printk("tcfs_rename: target %s/%s busy, d_count=%d\n",
new_dentry->d_parent->d_name.name,new_dentry->d_name.name,new_dentry->d_count);
#endif
			goto out;
		}
	}

	/*
	 * ... prune child dentries and writebacks if needed.
	 */
	if (old_dentry->d_count > 1) {
		tcfs_wb_all(old_inode);
		shrink_dcache_parent(old_dentry);
	}

	if (new_dentry->d_count > 1 && new_inode) {
#ifdef TCFS_PARANOIA
printk("tcfs_rename: new dentry %s/%s busy, d_count=%d\n",
new_dentry->d_parent->d_name.name,new_dentry->d_name.name,new_dentry->d_count);
#endif
		goto out;
	}

	/*
	 * To prevent any new references to the target during the rename,
	 * we unhash the dentry and free the inode in advance.
	 */
	if (!list_empty(&new_dentry->d_hash)) {
		d_drop(new_dentry);
		rehash = 1;
	}
	if (new_inode)
		d_delete(new_dentry);

	tcfs_invalidate_dircache(new_dir);
	
	if (!(ks=tcfs_get_key(new_dir))) {
		error=-EACCES;
		goto out;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=new_dentry->d_name.len;

		newcryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (newcryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (newcryptedname, new_dentry->d_name.name, len);
		newcryptedname[len]=0;

		if (strcmp (newcryptedname, ".") &&
			strcmp (newcryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (newcryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(newcryptedname,(k&0xFFF8)+8);
		}

	} else {
		int len=new_dentry->d_name.len;

		newcryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (newcryptedname, new_dentry->d_name.name, len);
		newcryptedname[len]=0;
	}

	tcfs_invalidate_dircache(old_dir);
	
	if (!(ks=tcfs_get_key(old_dir))) {
		return -EACCES;
	}

	if (ks!=TCFS_NO_SECURE) {
		int len=old_dentry->d_name.len;

		oldcryptedname=kmalloc (TCFS_MAXNAMLEN, GFP_KERNEL);
		memset (oldcryptedname, 0, TCFS_MAXNAMLEN);
		memcpy (oldcryptedname, old_dentry->d_name.name, len);
		oldcryptedname[len]=0;

		if (strcmp (oldcryptedname, ".") &&
			strcmp (oldcryptedname, "..")) {
				int k=len+1;
				tcfs_encrypt (oldcryptedname,
						(k&0xFFF8)+8,ks,NULL);
				tcfsencode(oldcryptedname,(k&0xFFF8)+8);
		}

	} else {
		int len=old_dentry->d_name.len;

		oldcryptedname=kmalloc (len+1, GFP_KERNEL);
		memcpy (oldcryptedname, old_dentry->d_name.name, len);
		oldcryptedname[len]=0;
	}

	error = tcfs_proc_rename(TCFS_DSERVER(old_dentry),
			TCFS_FH(old_dentry->d_parent), oldcryptedname,
			TCFS_FH(new_dentry->d_parent), newcryptedname);

	/* Update the dcache if needed */
	if (rehash)
		d_add(new_dentry, NULL);
	if (!error && !S_ISDIR(old_inode->i_mode))
		d_move(old_dentry, new_dentry);

out:
	/* new dentry created? */
	if (dentry)
		dput(dentry);
	
	if (newcryptedname) {
		kfree (newcryptedname);
	}

	if (oldcryptedname) {
		kfree (oldcryptedname);
	}

	return error;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 * End:
 */
