/* Virtual server daemon declarations.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#ifndef VSD_H
#define VSD_H

#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Client connection structure.  */
struct connection
{
  /* Name of the virtual server that the client has just logged into.  */
  char *virtual_server;
  /* User name that the client has logged in as.  */
  char *user_name;
  /* Pathname to the root of the virtual server.  If the command is a
     ADMIN_HOST then server_root will be NULL.  */
  char *server_root;

  /* Timeouts for authentication commands and then general commands.
     Measured in seconds.  */
  int login_timeout, command_timeout;

  /* /etc/vsd.conf.  */
  struct vsd_vs_map *map;
  /* Entry specific to the virtual server.  */
  struct vsd_vs *vs;

  /* Socket i/o is abstracted.  See io.h.  */
  void *io;
};

#define USER 1
#define GROUP 0

#define NAME_DONT_CHECK_LEN 1

extern void output (struct connection *vc, const char *fmt, ...);

/* Declarations in vsd.c.  */

/* Server states.  */
#define STATE_AUTHORISATION 0
#define STATE_TRANSACTION 1
#define STATE_UPDATE 2
#define STATE_LOGOUT 3

extern int vsd_log_level;

extern char *tcp_serverhost (void);
extern int run_command (struct connection *vc,
			const char *chroot_path, const char *username,
			const char *program_name, const char *line,
			const char *redir_stdin, const char *redir_stdout);

/* Declarations in proftpd.c.  */
extern int module_proftpd_check_chrt_priv (struct connection *vc,
					   const char *username);
extern int module_proftpd_add_chrt_priv (struct connection *vc,
					 const char *username);

/* Declarations in user.c.  */
extern void userlist (struct connection *vc, int argc, char *argv[]);
extern int useradd (struct connection *vc, int argc, char *argv[]);
extern int userdel (struct connection *vc, int argc, char *argv[]);
extern void usermod (struct connection *vc, int argc, char *argv[]);
extern int groupadd (struct connection *vc, int argc, char *argv[]);
extern int groupdel (struct connection *vc, int argc, char *argv[]);
extern void chuserpass (struct connection *vc, int argc, char *argv[]);
extern int modify_user_rights (struct connection *vc, int argc, char *argv[]);
extern int set_user_rights (struct connection *vc, int argc, char *argv[]);
extern int list_user_rights (struct connection *vc, int argc, char *argv[]);

/* Declarations in util.c */
extern const char *get_option (const char *option, const char *cmdline);

/* io.c */
extern int io_initialise (struct connection *vc);
extern int io_finalise (struct connection *vc);
extern struct io_requests *io_parse_input (struct connection *vc);
extern void io_generate_output (struct connection *vc);
extern void io_collect_output (struct connection *vc, const char *command);
extern void io_buffer_store (struct connection *vc, const char *fmt, ...);
extern void io_buffer_free (struct connection *vc);

#ifdef __cplusplus
}
#endif

#endif
