/* Virtual server administrator.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#else
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <stdarg.h>
#include <time.h>
#include <netdb.h>
#include <grp.h>
#include <pwd.h>
#include <fcntl.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#define SYSLOG_NAMES
#include <sys/syslog.h>
#include <sys/time.h>
#include <sys/resource.h>

#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#include "io.h"
#include "vsd.h"
#include "module.h"
#include "libvsd.h"

/* Autologout timer */
#define LOGINTIMEOUT 60*3
#define TIMEOUT 60*10

int vsd_log_level;

/* Function prototypes */

/* getdelim doesn't seem to be declared in stdio.h.  */
extern ssize_t getdelim (char **lineptr, size_t *n, int delim, FILE *stream);
static void process_requests (struct connection *vc);
static void free_connection (struct connection *vc);

/* General signal handling.  On reception of a signal, our main goal is
   to output an appropriate message on stdout that the client application
   can read.  It helps the client application terminate quickly rather
   than waiting for a read timeout.  */
static void signalhandler (int sig)
{
  alarm (0);
  if (sig == SIGALRM)
    output (NULL, "-ERR Autologout; idle for too long\n");
  if (sig == SIGTERM || sig == SIGKILL)
    output (NULL, "-ERR Killed\n");
  if (sig == SIGSEGV)
    output (NULL, "-ERR Unexpectedly terminated\n");
  _exit (1);
}

/* Find out our hostname and store it in `buf'.  */
static void find_hostname (char *buf, size_t buflen)
{
  struct hostent *he;
  struct sockaddr_in sin;
  int sin_len = sizeof (struct sockaddr_in);

  if (getsockname (0, (struct sockaddr *) &sin, (void *) &sin_len))
    {
      gethostname (buf, buflen);
      he = gethostbyname (buf);
      if (he)
	strcpy (buf, he->h_name);
    }
  else
    {
      he = gethostbyaddr ((char *) &sin.sin_addr,
			  sizeof (struct in_addr), sin.sin_family);
      if (he)
	strcpy (buf, he->h_name);
      else
	sprintf (buf, "[%s]", inet_ntoa (sin.sin_addr));
    }
}


/* Translate `name' from the codetables given in the syslog.h header
   file.  */
static int facility_decode (CODE *codetab, char *name)
{
  CODE *c;

  if (isdigit (*name))
    return atoi (name);

  for (c = codetab; c->c_name; c++)
    if (strcasecmp (name, c->c_name) == 0)
      return c->c_val;

  return -1;
}

#if 0
/* FreeBSD's useful malloc debugging.  */
extern char *malloc_options;
#endif

/* Main program */
int 
main (int argc, char *argv[])
{
  char *s;
  struct sigaction nact, oact;
  struct connection *vsd_connection;
  int ch, facility;

#if 0
  /* FreeBSD's useful malloc debugging.  Set all warnings and abort
     upon error.  */
  malloc_options = "AX";
#endif

  /* If the program name doesn't exist, then create our own for the
     syslog daemon.  */
  s = argv[0];
  if (s == NULL)
    s = "vsd";

  /* The different log levels:
       3. Log translated protocol requests.
       4. Log full protocol exchange.  */
  vsd_log_level = 1;
  facility = LOG_DAEMON;
  while ((ch = getopt (argc, argv, "p:l:")) != EOF)
    switch (ch)
      {
      case 'p': /* Facility to which we wish to send logs.  */
	facility = facility_decode (facilitynames, optarg);
	if (facility < 0)
	  {
	    fprintf (stderr, "vsd: unknown facility name: %s\n", optarg);
	    return 1;
	  }
	facility &= LOG_FACMASK;
	break;
      case 'l': /* VSD log level.  */
	vsd_log_level = atoi (optarg);
	if (vsd_log_level < 0)
	  vsd_log_level = 0;
	if (vsd_log_level > 4)
	  vsd_log_level = 4;
	break;
      }
  openlog (s, LOG_PID, facility);

#if 0
  /* Check we are running as root.  */
  if (getuid () != 0)
    {
      fprintf (stderr, "%s: must be run as root\n", s);
      syslog (LOG_ERR, "must be run as root");
      return 1;
    }
#endif

  vsd_connection = (struct connection *) malloc (sizeof (struct connection));
  memset (vsd_connection, 0, sizeof (struct connection));

  /* Set a 10 second timeout on command input.  */
  vsd_connection->command_timeout = 30;

  {
    /* This should probably be moved somewhere else.  */
    struct io *io;

    vsd_connection->io = (void *) malloc (sizeof (struct io));
    io = (struct io *) vsd_connection->io;
    memset (io, 0, sizeof (struct io));

    io->in_fd = fileno (stdin);
    io->out_fd = fileno (stdout);
  }

  /* We trap these signals so that we can make a clean exit, if there
     is ever an error.  */
  sigemptyset (&nact.sa_mask);
  nact.sa_handler = signalhandler;
  nact.sa_flags = SA_RESTART;
  sigaction (SIGALRM, &nact, &oact);
  sigaction (SIGTERM, &nact, &oact);
  sigaction (SIGKILL, &nact, &oact);
  sigaction (SIGSEGV, &nact, &oact);

  if (io_initialise (vsd_connection))
    {
      fprintf (stderr, "-ERR aborting - see syslog\n");
      return 1;
    }

  /* XXX FIXME: There is a potential race condition here.  Currently we
     read the contents of /etc/vsd.conf here and refer to it throughout
     the rest of the application.  This isn't such a big deal in itself
     until we decide to update the map with new data, such as changing
     the server quota or adding a new account.  Some method of locking
     /etc/vsd.conf and then refreshing the memory structure needs to
     be written in libvsd/map.c and then implemented throughout VSD.  */
  vsd_connection->map = vsd_map_read ();
  if (vsd_connection->map == NULL)
    output (vsd_connection, "-ERR Fatal error: server map read failure: %m\n");
  else
    {
      char host[256];

      find_hostname (host, sizeof (host) - 1);

      output (vsd_connection,
	      "+OK %s Virtual Server Administrator %s v%s ready\n",
	      PACKAGE, host, VERSION);
      /* Retrieve incoming data, parse and store in a io_request structure.  */
      io_parse_input (vsd_connection);

      /* Process the requests in the io_request structure and create an
	 io_output structure containing the replies.  */
      process_requests (vsd_connection);
      /* Send the data back to the client.  */
      io_generate_output (vsd_connection);

      output (vsd_connection, "+OK %s signing off\n", PACKAGE);
    }

  /* Clean up.  */
  io_finalise (vsd_connection);
  free_connection (vsd_connection);

  return 0;
}

/* Free memory allocated within the connection structure.  */
static void free_connection (struct connection *vc)
{
  if (vc->virtual_server)
    free (vc->virtual_server);
  if (vc->user_name)
    free (vc->user_name);
  if (vc->server_root)
    free (vc->server_root);
  io_buffer_free (vc);
  if (vc->io)
    free (vc->io);
}

/* Invokes the appropriate backend VSD module to process requests
   sent by the client.  Replies are collected into a temporary
   buffer, ready for output to the client.  */
static void process_requests (struct connection *vc)
{
  int req, cmd, level, status;
  struct io *io = (struct io *) vc->io;

  for (req = 0; req < io->requests_count; req++)
    {
      vc->virtual_server = io->requests[req].vs;
      vc->vs = vsd_getmapent (vc->map, vc->virtual_server, NULL);

#ifdef WITH_OPENSSL
      /* The CN field in the client certificate must match the
	 virtual server name that the client is trying to do
	 operations on.  */
      if (strcmp (io->certificate_vs, "<root>")
        && strcmp (io->certificate_vs, io->requests[req].vs))
	{
	  io_buffer_store (vc, "-ERR Permission denied: certificate does not match");
	  io_collect_output (vc, io->requests[req].actionv[0].argv[0]);
	  vsd_log_error (VSD_LOG_ERROR,
			 "certificate VS (%s) and request VS (%s) do not match",
			 io->certificate_vs, io->requests[req].vs);
	  continue;
	}
#endif

      /* Some commands are not appropriate for a specific virtual server.
	 These are entered as a VS called `<root>'.  */
      vc->server_root = vsd_map_server_root (vc->map, vc->virtual_server);
      level = (vc->server_root) ? ADMIN_VS : ADMIN_HOST;
      for (cmd = 0; cmd < io->requests[req].actionc; cmd++)
	{
	  int ac = io->requests[req].actionv[cmd].argc;
	  char **av = io->requests[req].actionv[cmd].argv;

	  if (vsd_log_level >= 3)
	    {
	      /* Log basic protocol commands to a file or syslog.  */
	      if (ac == 1)
		vsd_log_error (VSD_LOG_PROTOCOL, "req: vs=%s cmd=%s",
			       vc->virtual_server, av[0]);
	      else if (ac == 2)
		vsd_log_error (VSD_LOG_PROTOCOL, "req: vs=%s cmd=%s (%s)",
			       vc->virtual_server, av[0], av[1]);
	      else if (ac == 3)
		vsd_log_error (VSD_LOG_PROTOCOL, "req: vs=%s cmd=%s (%s, %s)",
			       vc->virtual_server, av[0], av[1], av[2]);
	      else if (ac == 4)
		vsd_log_error (VSD_LOG_PROTOCOL, "req: vs=%s cmd=%s (%s, %s, %s)",
			       vc->virtual_server, av[0], av[1], av[2], av[3]);
	    }

	  if (vc->vs == NULL && strcmp (vc->virtual_server, "<root>"))
	    io_buffer_store (vc, "-ERR Virtual server does not exist");
	  else
	    {
	      struct command_tag *tag;
	      /* Check command syntax.  */
	      status = module_check_cmd (level,
					 io->requests[req].actionv[cmd].argv[0],
					 io->requests[req].actionv[cmd].argc - 1,
					 io->requests[req].actionv[cmd].argv + 1,
					 &tag);
	      if (status == -1) /* No such command */
		io_buffer_store (vc, "No such command");
	      if (status == 1) /* Syntax error.  */
		io_buffer_store (vc, "-ERR Syntax: %s %s",
				 io->requests[req].actionv[cmd].argv[0],
				 tag->vsd_syntax);
	      else if (status == 0) /* Command exists.  Execute it.  */
		module_transaction (vc, level,
				    io->requests[req].actionv[cmd].argv[0],
				    io->requests[req].actionv[cmd].argc - 1,
				    io->requests[req].actionv[cmd].argv + 1,
				    io->requests[req].actionv[cmd].data);
	    }
	  io_collect_output (vc, io->requests[req].actionv[cmd].argv[0]);
	}
    }
}

/* Fork and execute `program_name' as a child process as the user `username'
   and wait until completion. Optionally `parameters' can contain a space
   separated string of command line arguments for the process.  The child
   process will have it's root directory set to `chroot_path' if not
   NULL.  Optionally stdin, stdout and stderr can be redirected to
   the filenames specified by `redir_stdin', `redir_stdout'.

   Return 0 on success, any other value on error.  */
int run_command (struct connection *vc,
		 const char *chroot_path, const char *username,
		 const char *program_name, const char *parameters,
		 const char *redir_stdin, const char *redir_stdout)
{
  int status, len;
  pid_t pid;
  char *line;

  if (program_name == NULL)
    {
      io_buffer_store (vc, "-ERR run_command: no program name");
      return -1;
    }

  len = strlen (program_name);
  if (len < sizeof ("/bin/sh") + 1)
    len = sizeof ("/bin/sh") + 1;
  if (chroot_path)
    len += strlen (chroot_path);
  len += 2; /* Cope with spaces and terminators.  */
  line = (char *) malloc (len);

  /* Check that `program_name' exists and is executable.  If we do this
     now then we can output more accurate error messages back to the
     client.  These checks should substantially reduce the risk of the
     client applications failing.  */
  line[0] = '\0';
  if (chroot_path)
    strcpy (line, chroot_path);
  strcat (line, program_name);

  if (access (line, R_OK | X_OK) == -1)
    {
      /* Don't give full pathname to the user. Only give local pathname
	 w.r.t. their virtual server.  */
      io_buffer_store (vc, "-ERR Access check on %s failed: %m",
		       program_name);

      /* Log as an error with syslogd with full pathname to aid the
	 sysadmin.  */
      syslog (LOG_ERR, "run_command: access check on %s failed: %m",
	      line);
      return -1;
    }

  /* Check that /bin/sh exists and is executable.  */
  line[0] = '\0';
  if (chroot_path)
    strcpy (line, chroot_path);
  strcat (line, "/bin/sh");

  if (access (line, R_OK | X_OK) == -1)
    {
      io_buffer_store (vc, "-ERR Access check on /bin/sh failed: %m");
      syslog (LOG_ERR, "run_command: access check on %s failed: %m", line);
      return -1;
    }

  free (line);

  pid = fork ();
  if (pid == (pid_t) 0)
    {
      /* Child side.  */
      const char *new_argv[4];
      const char *new_environ[2];
      char *shell_arg;
      int in_fd, out_fd;
#if 0
      struct rlimit lim;
#endif

      shell_arg = (char *) malloc (strlen (program_name) + strlen (parameters)
				   + 3);
      sprintf (shell_arg, "%s %s", program_name, parameters);

      new_argv[0] = "sh";
      new_argv[1] = "-c";
      new_argv[2] = shell_arg;
      new_argv[3] = NULL;

      new_environ[0] = "PATH=/usr/local/bin:/bin:/usr/bin:/usr/sbin";
      new_environ[1] = NULL;

#if 0
      /* Set CPU time and Memory usage limits. We don't want run away
	 processes fucking over our system.  */

      /* Limit CPU time to 4 seconds.  */
      lim.rlim_cur = 4;
      lim.rlim_max = 4;
      setrlimit (RLIMIT_CPU, &lim);

      /* Limit number of processes to 2.  */
      lim.rlim_cur = 4;
      lim.rlim_max = 4;
      setrlimit (RLIMIT_NPROC, &lim);

      /* Limit memory to 16Mb.  */
      lim.rlim_cur = 16*1024*1024;
      lim.rlim_max = 16*1024*1024;
      setrlimit (RLIMIT_DATA, &lim);
#endif

      if (chroot_path != NULL)
	{
	  if (chroot (chroot_path) == -1)
	    {
	      io_buffer_store (vc, "-ERR chroot failed");
	      syslog (LOG_ERR, "chroot (%s) failed: %m", chroot_path);
	      _exit (1);
	    }

	  if (chdir ("/") != 0)
	    {
	      io_buffer_store (vc, "-ERR chdir failed");
	      syslog (LOG_ERR, "chdir failed: %m");
	      _exit (1);
	    }
	}

      if (username != NULL)
	{
	  /* Become user `username'.  */
	  struct passwd *pw;

	  pw = getpwnam (username);
	  setuid (pw->pw_uid);
	}

      if (redir_stdin != NULL)
	{
	  /* Take standard input from `redir_stdin'.  */
	  close (0);
	  in_fd = open (redir_stdin, O_RDONLY);
	}

      if (redir_stdout != NULL)
	{
	  /* Redirect child process output to a file.  */
	  close (1);
	  close (2);
	  out_fd = open (redir_stdout, O_CREAT | O_WRONLY, 0600);
	  fcntl (out_fd, F_DUPFD, 2);
	}

      /* Exec the shell.  */
      execve ("/bin/sh", (char *const *) new_argv, (char *const *) new_environ);
      io_buffer_store (vc, "-ERR cannot execute %s", new_argv[0]);

      syslog (LOG_ERR, "cannot execute %s: %m", new_argv[0]);
      _exit (127);
    }
  else if (pid < (pid_t) 0)
    {
      /* The fork failed.  */
      status = -1;
    }
  else
    {
      /* Parent side.  */
      if (waitpid (pid, &status, 0) != pid)
	status = -1;

      /* Did exit occur successfully ?  */
      if (WIFEXITED (status) && WEXITSTATUS (status) == 0)
	return 0;

      /* Output an error if the program terminated without success,  */
      if (WIFSIGNALED (status))
	{
#if 0
	  io_buffer_store (vc, "-ERR Program %s got fatal signal %d",
		  program_name, WTERMSIG (status));
#endif

	  /* Log to syslog to aid the sysadmin.  */
	  syslog (LOG_ERR, "run_command: program %s%s got fatal signal %d",
		  (chroot_path == NULL) ? "" : chroot_path,
		  program_name, WTERMSIG (status));
	}
      else if (WIFEXITED (status)
	       && WEXITSTATUS (status) != 0)
	{
#if 0
	  io_buffer_store (vc, "-ERR Program %s exited with status %d",
		  program_name, WEXITSTATUS (status));
#endif

	  syslog (LOG_ERR, "run_command: program %s%s exited with status %d",
		  (chroot_path == NULL) ? "" : chroot_path,
		  program_name, WEXITSTATUS (status));
	}
    }

  return status;
}
