/* Control virtual server operation.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>

#include "vsd.h"
#include "vs.h"
#include "module.h"
#include "vsd.h"
#include "libvsd.h"

static void send_output (struct connection *vc,
			 const char *server_root, const char *file)
{
  char *temp;
  FILE *stream;
  char line[256];

  if (server_root)
    {
      temp = (char *) malloc (strlen (server_root) + strlen (file) + 3);
      strcpy (temp, server_root);
      strcat (temp, file);
    }
  else
    temp = (char *) file;

  stream = fopen (temp, "r");
  if (stream != NULL)
    {
      while (fgets (line, sizeof (line), stream))
	io_buffer_store (vc, "%s", line);

      fclose (stream);
    }
  unlink (temp);

  if (server_root)
    free (temp);
}

/* Restart all processes on a virtual server.  */
void vs_reboot (struct connection *vc, int argc, char *argv[])
{
  char line[256];
  int status;
  char server_temp[128];

  /* VS_REBOOT */

  /* Stop the virtual server (switches to runlevel 0).
     Send all output to a file. */
  sprintf (server_temp, "/tmp/vsd.reboot.%s", vc->virtual_server);
  sprintf (line, "--restart %s", vc->virtual_server);
  status = run_command (vc, NULL, NULL, VSBOOT, line, NULL,
			server_temp);

  /* Open file and send all data back to the client.  */
  send_output (vc, NULL, server_temp);
}

/* Send a signal to all processes on a virtual server.  */
void vs_signal (struct connection *vc, int argc, char *argv[])
{
  int *procs = vsd_enum_procs (vc->virtual_server);
  int x = 0;
  int signum = atoi (argv[0]);

  /* VS_SIGNAL <signal number>  */
  while (procs[x] != -1)
    {
      kill (procs[x], signum);
      x++;
    }
  free (procs);
}  

/* Change the status of an account to active, then start all virtual
   server services.  */
void vs_account_enable (struct connection *vc, int argc, char *argv[])
{
  char line[256], server_temp[128];
  struct vsd_vs *vs;
  const char *st;

  /* VS_ENABLE <virtual-server>  */
  if ((st = vsd_valid_vsname (argv[0])))
    {
      io_buffer_store (vc, "-ERR Virtual server name is invalid: %s", st);
      return;
    }

  /* Check virtual server exists.  */
  vs = vsd_getmapent (vc->map, argv[0], NULL);
  if (vs == NULL)
    {
      io_buffer_store (vc, "-ERR Server %s is not known", argv[0]);
      return;
    }

  /* Activate the account.  */
  if (vsd_map_set_status (vc->map, argv[0], "active"))
    {
      io_buffer_store (vc, "-ERR Could not find server %s: %m", argv[0]);
      return;
    }

  vsd_map_save (vc->map);

  /* Start the virtual server and assign its ethernet address. */
  sprintf (line, "--start %s", argv[0]);
  sprintf (server_temp, "/tmp/vsd.enable.%s", argv[0]);
  if (! run_command (vc, NULL, NULL, VSBOOT, line, NULL,
		     server_temp))
    {
      /* Open file and send all data back to the client.  */
      send_output (vc, NULL, server_temp);
    }
}

/* Stop all services on a virtual server, then change the status
   to disabled.  */
void vs_account_disable (struct connection *vc, int argc, char *argv[])
{
  char line[256], server_temp[128];
  struct vsd_vs *vs;
  const char *st;

  /* VS_DISABLE <virtual-server>  */
  if ((st = vsd_valid_vsname (argv[0])))
    {
      io_buffer_store (vc, "-ERR Virtual server name is invalid: %s", st);
      return;
    }

  /* Check virtual server exists.  */
  vs = vsd_getmapent (vc->map, argv[0], NULL);
  if (vs == NULL)
    {
      io_buffer_store (vc, "-ERR Server %s is not known", argv[0]);
      return;
    }

  /* Stop the virtual server and deassign it's ethernet address. Send all
     output to a file. */
  sprintf (line, "--stop %s", argv[0]);
  sprintf (server_temp, "/tmp/vsd.disable.%s", argv[0]);
  if (run_command (vc, NULL, NULL, VSBOOT, line, NULL, server_temp))
    return;

  /* Disable the account.  */
  if (vsd_map_set_status (vc->map, argv[0], "disabled"))
    io_buffer_store (vc, "-ERR Could not find server %s: %m", argv[0]);
  else
    {
      vsd_map_save (vc->map);
      /* Open file and send all data back to the client.  */
      send_output (vc, NULL, server_temp);
    }
}

/* Create a new virtual server account.  */
void vs_account_create (struct connection *vc, int argc, char *argv[])
{
  FILE *stream;
  int num_uid, start_uid, diskquota;
  const char *st, *ns1, *ns2;

  /* VS_CREATE <virtual server> <ip-addr> <fqdn> <numuid> <quota>
               [<pri ns>] [<sec ns>] */
  if ((st = vsd_valid_vsname (argv[0])))
    {
      io_buffer_store (vc, "-ERR Virtual server name is invalid: %s", st);
      return;
    }

  /* Does virtual server exist ?  */
  if (vsd_getmapent (vc->map, argv[0], NULL))
    {
      io_buffer_store (vc, "-ERR Server %s already exists", argv[0]);
      return;
    }

  /* Check IP address is of a valid format.  */
  if ((st = vsd_check_ip_address (argv[1])))
    {
      io_buffer_store (vc, "-ERR IP address error: %s", st);
      return;
    }

  if (vsd_getmapent (vc->map, NULL, argv[1]))
    {
      io_buffer_store (vc, "-ERR IP address %s is already in use", argv[1]);
      return;
    }

  /* Find a free UID allocation.  */
  num_uid = atoi (argv[3]);
  if (num_uid <= 0 || num_uid >= 65000)
    {
      io_buffer_store (vc, "-ERR Number of users (%d) is incorrect", num_uid);
      return;
    }

  start_uid = vsd_alloc_uid (vc->map, num_uid);
  if (start_uid == -1)
    {
      io_buffer_store (vc, "-ERR Could not allocate a UID chunk");
      return;
    }

  diskquota = atoi (argv[4]);
  ns1 = (argc >= 6) ? argv[5] : NULL;
  ns2 = (argc >= 7) ? argv[6] : NULL;

  /* Check primary nameserver IP address is of a valid format.  */
  if (ns1 && (st = vsd_check_ip_address (ns1)))
    {
      io_buffer_store (vc, "-ERR Primary nameserver IP address %s error: %s",
		       ns1, st);
      return;
    }

  /* Check secondary name server IP address is of a valid format.  */
  if (ns2 && (st = vsd_check_ip_address (ns2)))
    {
      io_buffer_store (vc, "-ERR Secondary nameserver IP address %s error: %s",
		       ns2, st);
      return;
    }

  if (vsd_map_add_server (vc->map, argv[0], argv[1], start_uid, num_uid,
			  diskquota, ns1, ns2))
    io_buffer_store (vc, "-ERR Cannot create account: %s",
		     (errno == EMFILE) ? "host server is full"
		     : strerror (errno));
  else
    {
      vsd_map_save (vc->map);

      stream = fopen ("/tmp/vsd.accounts", "a");
      if (stream != NULL)
	{
	  fprintf (stream, "create %s %s\n", argv[0], argv[2]);
	  fclose (stream);
	}
      else
	io_buffer_store (vc, "-ERR Account created but not physically: %m");
    }
}

void vs_account_delete (struct connection *vc, int argc, char *argv[])
{
  FILE *stream;
  struct vsd_vs *vs;
  char line[128];
  const char *st;

  /* VS_DELETE <virtual-server> */
  if ((st = vsd_valid_vsname (argv[0])))
    {
      io_buffer_store (vc, "-ERR Virtual server name is invalid: %s", st);
      return;
    }

  vs = vsd_getmapent (vc->map, argv[0], NULL);
  if (vs == NULL)
    {
      io_buffer_store (vc, "-ERR Server %s is not known", argv[0]);
      return;
    }

  /* Stop the virtual server and deassign it's ethernet address. Send all
     output to a file. */
  sprintf (line, "--stop %s", argv[0]);
  if (run_command (vc, NULL, NULL, VSBOOT, line, NULL, "/dev/null"))
    return;

  stream = fopen ("/tmp/vsd.accounts", "a");
  if (stream != NULL)
    {
      fprintf (stream, "delete %s %s\n", argv[0],
	       vc->map->partition[vs->partition].mount);
      fclose (stream);
    }
  else
    io_buffer_store (vc, "-ERR Account deleted but not physically: %m");
  vsd_map_delete_server (vc->map, argv[0]);
  vsd_map_save (vc->map);
}

/* Return configuration details about a virtual server.  */
void vs_details (struct connection *vc, int level, int argc, char *argv[])
{
  struct vsd_vs *vs;
  int x, y;

  /* VS_DETAILS [<virtual-server>]
     Returns a single line per virtual server:
       name=<vsname> ip=<main ipaddr> uid=<starting uid>
       quota=<disk limit in megabytes> path=<path to virtual server>
       prins=<primary nameserver> secns=<secondary nameserver> aip=<ip alias>
  */
  if ((level == ADMIN_VS && argc == 0) || (level == ADMIN_HOST && argc == 1))
    {
      if (argc == 1)
	{
	  const char *st;
	  /* Check validity of virtual server name.  */
	  if ((st = vsd_valid_vsname (argv[0])))
	    {
	      io_buffer_store (vc, "-ERR Virtual server name is invalid: %s", st);
	      return;
	    }

	  vs = vsd_getmapent (vc->map, argv[0], NULL);
	  if (vs == NULL)
	    {
	      io_buffer_store (vc, "-ERR Server %s is not known", argv[0]);
	      return;
	    }
	}
      else
	vs = vc->vs;

      io_buffer_store (vc,
		       "name=%s ip=%s uid=%d users=%d quota=%d path=\"%s\" prins=%s secns=%s",
		       vs->name, vsd_ip_ntoa (vs->ip),
		       vs->startuid, vs->users, vs->quota,
		       vc->map->partition[vs->partition].mount,
		       (vs->ns1[0]) ? vs->ns1 : vc->map->global.ns1,
		       (vs->ns2[0]) ? vs->ns2 : vc->map->global.ns2);
      for (y = 0; y < vs->ipalias_total; y++)
	io_buffer_store (vc, " aip=%s", vsd_ip_ntoa (&vs->ipalias[y]));
      
      io_buffer_store (vc, "\n");
    }
  else if (level == ADMIN_HOST)
    for (x = 0; x < vc->map->servers; x++)
      {
	vs = &vc->map->server[x];
	if (vs->name[0] != '\0')
	  {
	    io_buffer_store (vc,
			     "name=%s ip=%s uid=%d users=%d quota=%d path=\"%s\" prins=%s secns=%s",
			     vs->name, vsd_ip_ntoa (vs->ip),
			     vs->startuid, vs->users, vs->quota,
			     vc->map->partition[vs->partition].mount,
			     (vs->ns1[0]) ? vs->ns1 : vc->map->global.ns1,
			     (vs->ns2[0]) ? vs->ns2 : vc->map->global.ns2);
	    for (y = 0; y < vs->ipalias_total; y++)
	      io_buffer_store (vc, " aip=%s", vsd_ip_ntoa (&vs->ipalias[y]));
	    io_buffer_store (vc, "\n");
	  }
      }
  else
    io_buffer_store (vc, "Permission denied\n");
}

/* Rename a virtual server.  This does most of the boring work involved
   in altering a virtual server.  Assuming the virtual server user hasn't
   made any assumptions about the virtual server name (and they shouldn't
   have because there is no point knowing it) the following files need
   modifying:

       /etc/FQDN      (in virtual server)
       virtual server directory name (on host server)
       /etc/vsd.conf  (on host server)

   Normally though, we change the fully qualified domain name of the
   virtual server at the same time as renaming it.  In this case, the
   following files need to be changed:

       virtual server directory name  (on host server)
       /etc/FQDN         (in virtual server)
       /etc/issue.net    (in virtual server)
       /etc/httpd/conf/httpd.conf  (in virtual server)
       /etc/hosts        (in virtual server)
       /etc/sendmail.cw  (in virtual server)
       /etc/vsd.conf     (on host server)

   Also, we provide the option of changing the virtual server IP address.
   We can't cope with everything - just mess with the files we know will
   exist and leave it up to the user to fix the files we don't know anything
   about.  The following files will need modifying:

       /etc/vsd.conf (on host server)
       /etc/hosts    (in virtual server)
       /etc/httpd/conf/httpd.conf (in virtual server)  */
void vs_modify (struct connection *vc, int argc, char *argv[])
{
  const char *status, *path = NULL;
  struct vsd_file_line *file, *line;
  int result;

  /* VS_MODIFY <old-vs> <new-vs> <old-fqdn> <new-fqdn> <old-ip> <new-ip> */

  /* Check arguments.  */
  status = vsd_valid_vsname (argv[0]);
  if (status)
    {
      io_buffer_store (vc, "-ERR Old vs name is invalid: %s", status);
      return;
    }
  status = vsd_valid_vsname (argv[1]);
  if (status)
    {
      io_buffer_store (vc, "-ERR New vs name is invalid: %s", status);
      return;
    }

  if (strcmp (argv[0], argv[1]) != 0)
    {
      /* Virtual server name is changing.  */
      char *oldpath;

      /* Check new virtual server name is available and old virtual
         server exists.  If so, rename virtual server in /etc/vsd.conf
         and rename the physical directory.  */
      path = vsd_map_server_root (vc->map, argv[0]);
      if (! path)
	{
	  io_buffer_store (vc, "-ERR Server %s is not known", argv[0]);
	  return;
	}

      oldpath = strdup (path);
      result = vsd_map_rename_server (vc->map, argv[0], argv[1]);
      if (! result)
	path = vsd_map_server_root (vc->map, argv[1]);
      if (result || ! path)
	{
	  io_buffer_store (vc, "-ERR Server %s already exists", argv[1]);
	  return;
	}
      
      /* Rename virtual server directory.  */
      if (rename (oldpath, path))
	{
	  io_buffer_store (vc, "-ERR Cannot rename VS from %s to %s: %m",
			   argv[0], argv[1]);
	  free (oldpath);
	  return;
	}

      vsd_map_save (vc->map);
      free (oldpath);

      /* Change /etc/FQDN on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/FQDN"))
	io_buffer_store (vc, "-ERR Cannot modify /etc/FQDN: %m");
      else
	{
	  char fqdn[256], vsname[256];
	  /* There is only one line in /etc/FQDN. It is of the format:
	        <fqdn> <virtualserver-name>
	  */
	  sscanf (file->line, "%s %s", fqdn, vsname);
	  free (file->line);
	  file->line = (char *) malloc (strlen (fqdn) + 1
					+ strlen (argv[1]) + 1);
	  sprintf (file->line, "%s %s", fqdn, argv[1]);
	  vsd_save_file (file, path, "/etc/FQDN");
	  vsd_free_file (file);
	}
    }

  if (! path)
    path = vsd_map_server_root (vc->map, argv[1]);

  if (strcmp (argv[2], argv[3]) != 0)
    {
      /* Virtual server FQDN is changing.  The strategy here
         is to try and modify as many files as possible.  Report
         errors for files we have been unsuccessful in changing and
         the user can fix these by hand :-)  */

      /* Change /etc/FQDN on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/FQDN"))
	io_buffer_store (vc, "-ERR Cannot modify /etc/FQDN: %m");
      else
	{
	  char fqdn[256], vsname[256];
	  /* There is only one line in /etc/FQDN. It is of the format:
	        <fqdn> <virtualserver-name>
	  */
	  sscanf (file->line, "%s %s", fqdn, vsname);
	  free (file->line);
	  file->line = (char *) malloc (strlen (argv[3]) + 1
					+ strlen (vsname) + 1);
	  sprintf (file->line, "%s %s", argv[3], vsname);
	  vsd_save_file (file, path, "/etc/FQDN");
	  vsd_free_file (file);
	}

      /* Change /etc/issue.net on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/issue.net"))
	io_buffer_store (vc, "-ERR Cannot modify /etc/issue.net: %m");
      else
	{
	  line = vsd_file_str_search (file, argv[2]);
	  vsd_file_str_replace (line, argv[2], argv[3]);
	  vsd_save_file (file, path, "/etc/issue.net");
	  vsd_free_file (file);
	}

      /* Change /etc/hosts on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/hosts"))
	io_buffer_store (vc, "-ERR Cannot modify /etc/hosts: %m");
      else
	{
	  line = vsd_file_str_search (file, argv[2]);
	  vsd_file_str_replace (line, argv[2], argv[3]);
	  vsd_save_file (file, path, "/etc/hosts");
	  vsd_free_file (file);
	}

      /* Change /etc/sendmail.cw on the virtual server.  Check whether
         the new domain already exists.  If it does, then don't
         bother with the rename.  */
      if (vsd_load_file (&file, path, "/etc/sendmail.cw"))
	io_buffer_store (vc, "-ERR Cannot modify /etc/sendmail.cw: %m");
      else
	{
	  line = vsd_file_str_search (file, argv[3]);
	  if (! line)
	    {
	      line = vsd_file_str_search (file, argv[2]);
	      vsd_file_str_replace (line, argv[2], argv[3]);
	      vsd_save_file (file, path, "/etc/sendmail.cw");
	      vsd_free_file (file);
	    }
	}

      /* Change /etc/httpd/conf/httpd.conf on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/httpd/conf/httpd.conf"))
	io_buffer_store (vc,
			 "-ERR Cannot modify /etc/httpd/conf/httpd.conf: %m");
      else
	{
	  /* Replace all occurances.  */
	  vsd_file_str_search_replace (file, argv[2], argv[3]);
	  vsd_save_file (file, path, "/etc/httpd/conf/httpd.conf");
	  vsd_free_file (file);
	} 
   }

  if (strcmp (argv[4], argv[5]) != 0)
    {
      /* Virtual server IP address has changed.  */
      const char *st;

      /* Check IP address is of a valid format.  */
      if ((st = vsd_check_ip_address (argv[4])))
	{
	  io_buffer_store (vc, "-ERR Old IP address %s error: %s", argv[4],st);
	  return;
	}

      /* Check IP address is of a valid format.  */
      if ((st = vsd_check_ip_address (argv[5])))
	{
	  io_buffer_store (vc, "-ERR Old IP address %s error: %s", argv[5],st);
	  return;
	}

      result = vsd_map_change_server_ip (vc->map, argv[1], argv[4], argv[5]);
      if (! result)
	vsd_map_save (vc->map);
      else if (result == 1)
	{
	  io_buffer_store (vc, "-ERR Cannot change IP: old IP not found");
	  return;
	}
      else if (result == 2)
	{
	  io_buffer_store (vc, "-ERR Cannot change IP: new IP in use");
	  return;
	}

      /* Change /etc/hosts on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/hosts"))
	io_buffer_store (vc, "-ERR Cannot modify /etc/hosts: %m");
      else
	{
	  line = vsd_file_str_search (file, argv[4]);
	  vsd_file_str_replace (line, argv[4], argv[5]);
	  vsd_save_file (file, path, "/etc/hosts");
	  vsd_free_file (file);
	}

      /* Change /etc/httpd/conf/httpd.conf on the virtual server.  */
      if (vsd_load_file (&file, path, "/etc/httpd/conf/httpd.conf"))
	io_buffer_store (vc,
			 "-ERR Cannot modify /etc/httpd/conf/httpd.conf: %m");
      else
	{
	  /* Replace all occurances.  */
	  vsd_file_str_search_replace (file, argv[4], argv[5]);
	  vsd_save_file (file, path, "/etc/httpd/conf/httpd.conf");
	  vsd_free_file (file);
	}
    }
}

/* Add an AliasIP to a virtual server.  */
void vs_add_ipalias (struct connection *vc, int argc, char *argv[])
{
  int status;
  const char *st;

  /* VS_ADDIPALIAS <ip-addr>  */

  /* Check IP address is of a valid format.  */
  if ((st = vsd_check_ip_address (argv[0])))
    {
      io_buffer_store (vc, "-ERR IP address %s error: %s", argv[0], st);
      return;
    }

  status = vsd_map_add_ipalias (vc->map, vc->virtual_server, argv[0]);
  if (status == 0)
    {
      if (vsd_map_save (vc->map))
	io_buffer_store (vc, "-ERR Cannot update vsd.conf: %m");
    }
  else if (status == 1)
    io_buffer_store (vc, "-ERR Virtual server %s does not exist",
		     vc->virtual_server);
  else if (status == 2)
    io_buffer_store (vc, "-ERR IP address %s is already defined", argv[0]);
  else if (status == 3)
    io_buffer_store (vc, "-ERR IP address %s is already defined by %s",
		     argv[0], vc->virtual_server);
  else if (status == 4)
    io_buffer_store (vc, "-ERR Cannot add %s: memory allocation error",
		     argv[0]);
  else
    io_buffer_store (vc, "-ERR vs_add_ipalias: unknown error (%d) occurred",
		     status);
}

/* Remove an AliasIP from a virtual server.  */
void vs_del_ipalias (struct connection *vc, int argc, char *argv[])
{
  int status;
  const char *st;

  /* VS_DELIPALIAS <ip-addr>  */

  /* Check IP address is of a valid format.  */
  if ((st = vsd_check_ip_address (argv[0])))
    {
      io_buffer_store (vc, "-ERR IP address %s error: %s", argv[0], st);
      return;
    }

  status = vsd_map_delete_ipalias (vc->map, vc->virtual_server, argv[0]);
  if (status == 0)
    {
      if (vsd_map_save (vc->map))
	io_buffer_store (vc, "-ERR Cannot update vsd.conf: %m");
    }
  else if (status == 1)
    io_buffer_store (vc, "-ERR Virtual server %s does not exist",
		     vc->virtual_server);
  else if (status == 2)
    io_buffer_store (vc, "-ERR IP address %s is not defined", argv[0]);
  else if (status == 3)
    io_buffer_store (vc, "-ERR Cannot delete IP address %s: not owner",
		     argv[0]);
  else if (status == 4)
    io_buffer_store (vc, "-ERR Cannot delete %s: memory allocation error",
		     argv[0]);
  else
    io_buffer_store (vc, "-ERR vs_del_ipalias: unknown error (%d) occurred",
		     status);
}
