/* Module command interface.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include "module.h"
#include "mod_vs.h"
#include "vs.h"

static struct command_tag clist[] =
{
  {
    CMD_VS_CREATE, ADMIN_HOST,
    "Create a new virtual server account",
    "VS_CREATE", 5, "<vs> <ip-addr> <fqdn> <numuid> <quota> [<ns1>] [<ns2>]",
    "vs_create", 6, "<host svr> <vs> <ipaddr> <fqdn> <numuid> <quota>"
  },
  {
    CMD_VS_DELETE, ADMIN_HOST,
    "Delete a virtual server account",
    "VS_DELETE", 1, "<vs>",
    "vs_delete", 2, "<host svr> <vs>"
  },
  {
    CMD_VS_ENABLE, ADMIN_HOST,
    "Enable a virtual server account",
    "VS_ENABLE", 1, "<vs>",
    "vs_enable", 2, "<host svr> <vs>"
  },
  {
    CMD_VS_DISABLE, ADMIN_HOST,
    "Disable a virtual server account",
    "VS_DISABLE", 1, "<virtual-server>",
    "vs_disable", 2, "<host svr> <vs>"
  },
  {
    CMD_VS_DETAILS, ADMIN_VS | ADMIN_HOST,
    "Return config details for a virtual server",
    "VS_DETAILS", 0, "[<vs>]",
    "vs_details", 1, "<host svr> [<vs>]"
  },
  {
    CMD_VS_REBOOT, ADMIN_VS, 
    "Restart services on a virtual server",
    "VS_REBOOT", 0, "",
    "vs_reboot", 2, "<host svr> [<vs>]"
  },
  {
    CMD_VS_SIGNAL, ADMIN_VS,
    "Send a signal to all processes",
    "VS_SIGNAL", 1, "<signal number>",
    "vs_signal", 3, "<host svr> <vs> <signum>"
  },
  {
    CMD_VS_MODIFY, ADMIN_HOST,
    "Modify characteristics of a virtual server",
    "VS_MODIFY", 6, "<old-vs> <new-vs> <old-fqdn> <new-fqdn> <old-ip> <new-ip>",
    "vs_modify", 7, "<host svr> <oldvs> <newvs> <old-fqdn> <new-fqdn> <old-ip> <new-ip>"
  },
  {
    CMD_VS_ADDIPALIAS, ADMIN_VS,
    "Add an IP alias to a virtual server",
    "VS_ADDIPALIAS", 1, "<ip-addr>",
    "vs_addipalias", 3, "<host svr> <vs> <ipaddr>"
  },
  {
    CMD_VS_DELIPALIAS, ADMIN_VS,
    "Delete an IP alias from a virtual server",
    "VS_DELIPALIAS", 1, "<ip-addr>",
    "vs_delipalias", 3, "<host svr> <vs> <ipaddr>"
  },
  {
    0, 0, NULL, NULL, 0, NULL, NULL, 0, NULL
  }
};

/* Return -1 if `cmd' does not exist.
   Return 1 if there is a syntax error.
   Return 0 if everything is cool.  */
int mod_vs_check_cmd (int level, const char *cmd, int argc, char *argv[],
		      struct command_tag **list)
{
  if (cmd == NULL)
    {
      /* Return a pointer to all commands.  */
      *list = clist;
      return 0;
    }

  /* Find command in clist.  */
  *list = module_lookup_command (clist, level, cmd);
  if (*list == NULL)
    return -1;  /* Command does not exist.  */

  /* Check command syntax.  Pretty basic at the moment.  */
#ifdef VSD_LIB
  if (argc < (*list)->vsdadm_args)
    return 1; /* Syntax error.  */
#else
  if (argc < (*list)->vsd_args)
    return 1; /* Syntax error.  */
#endif

  /* Syntax is ok.  */
  return 0;
}

#ifdef VSD_LIB
char *mod_vs_cmd (void *io, int level, const char *cmd,
		  int argc, char **argv)
{
  struct command_tag *list;
  char *status = NULL;

  list = module_lookup_command (clist, level, cmd);
  if (! list)
    return "notexist";

  vsd_set_level (io, list->level);
  switch (list->token)
    {
    case CMD_VS_CREATE:
      vsd_send_command (io, NULL, "VS_CREATE %s %s %s %s %s",
			argv[1] /* vs */, argv[2] /* ipaddr */,
			argv[3] /* fqdn */, argv[4] /* numuid */,
			argv[5] /* quota */);
      break;
    case CMD_VS_DELETE:
      vsd_send_command (io, NULL, "VS_DELETE %s", argv[1] /* vs */);
      break;
    case CMD_VS_ENABLE:
      vsd_send_command (io, NULL, "VS_ENABLE %s", argv[1] /* vs */);
      break;
    case CMD_VS_DISABLE:
      vsd_send_command (io, NULL, "VS_DISABLE %s", argv[1] /* vs */);
      break;
    case CMD_VS_DETAILS:
      vsd_set_level (io, (argv[1]) ? ADMIN_VS : ADMIN_HOST);
      vsd_send_command (io, NULL, "VS_DETAILS");
      break;
    case CMD_VS_REBOOT:
      vsd_send_command (io, NULL, "VS_REBOOT");
      break;
    case CMD_VS_SIGNAL:
      vsd_send_command (io, NULL, "VS_SIGNAL %d", argv[2] /* signum */);
      break;
    case CMD_VS_MODIFY:
      vsd_send_command (io, NULL, "VS_MODIFY %s %s %s %s %s %s",
			argv[1] /* old-vs */, argv[2] /* new-vs */,
			argv[3] /* old-fqdn */, argv[4] /* new-fqdn */,
			argv[5] /* old-ip */, argv[6] /* new-ip */);
      break;
    case CMD_VS_ADDIPALIAS:
      vsd_send_command (io, NULL, "VS_ADDIPALIAS %s",
			argv[2] /* ipaddr */);
      break;
    case CMD_VS_DELIPALIAS:
      vsd_send_command (io, NULL, "VS_DELIPALIAS %s",
			argv[2] /* ipaddr */);
      break;
    default:
      return "mod_vs: command not recognised";
      break;
    }

  status = vsd_process_reply (io);
  return status;
}

#else /* ! VSD_LIB */
/* Return 0 if command processed, 1 if not recognised.  */
int mod_vs_cmd (struct connection *vc, int level, const char *command,
		int argc, char *argv[], const char *data)
{
  int token;
  struct command_tag *tag;

  tag = module_lookup_command (clist, level, command);
  if (tag == NULL)
    return 1;

  token = tag->token;
  switch (token)
    {
    case CMD_VS_CREATE:
      vs_account_create (vc, argc, argv);
      break;
    case CMD_VS_DELETE:
      vs_account_delete (vc, argc, argv);
      break;
    case CMD_VS_ENABLE:
      vs_account_enable (vc, argc, argv);
      break;
    case CMD_VS_DISABLE:
      vs_account_disable (vc, argc, argv);
      break;
    case CMD_VS_DETAILS:
      vs_details (vc, level, argc, argv);
      break;
    case CMD_VS_REBOOT:
      vs_reboot (vc, argc, argv);
      break;
    case CMD_VS_SIGNAL:
      vs_signal (vc, argc, argv);
      break;
    case CMD_VS_MODIFY:
      vs_modify (vc, argc, argv);
      break;
    case CMD_VS_ADDIPALIAS:
      vs_add_ipalias (vc, argc, argv);
      break;
    case CMD_VS_DELIPALIAS:
      vs_del_ipalias (vc, argc, argv);
      break;
    default:
      return 1;
      break;
    }

  return 0;
}
#endif

module vs_module =
{
  mod_vs_check_cmd,
  mod_vs_cmd
};
