/* Sendmail configuration file manipulation.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Written by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>

#include "vsd.h"
#include "libvsd.h"
#include "sendmail.h"

/* Read /etc/sendmail.cf to get the paths for virtusertable, aliases
   and sendmail.cw.   If sendmail.cf contains the relevant information
   then allocate some memory and store the paths in the appropriate
   variable.  Otherwise, store NULL in the pointer.  */
static int read_sendmail_cf (const char *server_root,
			     char **virtusertable, 
			     char **aliases,
			     char **sendmail_cw)
{
  struct vsd_file_line *head, *line;
  char buf[256];

  if (virtusertable)
    *virtusertable = NULL;
  if (aliases)
    *aliases = NULL;
  if (sendmail_cw)
    *sendmail_cw = NULL;

  if (vsd_load_file (&head, server_root, "/etc/sendmail.cf"))
    return;

  line = head;
  while (line)
    {
      if (aliases && strncasecmp (line->line, "O AliasFile", 11) == 0)
	{
	  /* O AliasFile=/etc/aliases  */
	  char *p = strchr (line->line, '=');
	  if (p != NULL)
	    {
	      p++;
	      while (isspace (*p))
		p++;
	      *aliases = (char *) malloc (strlen (p) + 1);
	      strcpy (*aliases, p);
	    }
	}
      if (virtusertable && strncasecmp (line->line, "Kvirtuser", 9) == 0)
	{
	  /* Kvirtuser hash -o /etc/virtusertable.  */
	  char *p = strstr (line->line, "-o");
	  char *q;
	  if (p != NULL)
	    {
	      /* Skip past the -o and any spaces.  */
	      p += 2;
	      while (isspace (*p))
		p++;

	      q = p;
	      while (! isspace (*q))
		q++;

	      *virtusertable = (char *) malloc ((q - p) + 2);
	      strncpy (*virtusertable, p, q - p + 1);
	      (*virtusertable)[q - p + 1] = '\0';
	    }
	}
      if (sendmail_cw && strncasecmp (line->line, "Fw", 2) == 0)
	{
	  /* Fw/etc/sendmail.cw  */
	  *sendmail_cw = (char *) malloc (strlen (line->line + 2) + 1);
	  strcpy (*sendmail_cw, line->line + 2);
	}
      line = line->next;
    }

  return 0;
}

/* /etc/virtusertable manipulation.

   The syntax of /etc/virtusertable is:
     <e-mail alias><space|tab><pop3 account>

   Lines beginning with a # are comments.  */

struct virt_entry
{
  char *alias;
  char *pop_account;
  struct virt_entry *next;
};

/* Add an entry to the virtusertable linked list.
   Return NULL on success
   Return an error string on failure.  */
static const char *virt_add_entry (struct virt_entry **list,
				   const char *alias, const char *pop)
{
  struct virt_entry *entry;

  if (alias == NULL || *alias == '\0')
    return "LHS is empty";
  if (pop == NULL || *pop == '\0')
    return "RHS is empty";

  {
    const char *p;
    int at;

    /* Check for invalid characters.  */
    at = 0;
    for (p = alias; *p; p++)
      {
	if (isspace (*p) || iscntrl (*p))
	  return "LHS contains invalid characters";
	if (*p == '@')
	  at ++;
	if (*p == '.' && p[1] == '.')
	  return "LHS contains `..' sequence";
      }

    if (at >= 2)
      return "LHS contains more than one `@' character";

    if (! isalnum (p[-1]))
      return "LHS must end with an alphanumeric character";

    /* Do the same for the pop account.  */
    at = 0;
    for (p = pop; *p; p++)
      {
	if (isspace (*p) || iscntrl (*p))
	  return "RHS contains invalid characters";
	if (*p == '@')
	  at ++;

	if (*p == '.' && p[1] == '.')
	  return "RHS contains `..' sequence";
      }

    if (at >= 2)
      return "RHS contains more than one `@' character";

    if (! isalnum (p[-1]))
      return "RHS must end with an alphanumeric character";
  }

  /* Check whether `alias' already exists.  */
  if (*list != NULL)
    {
      struct virt_entry *llist = *list;
      while (llist)
	{
	  if (strcasecmp (llist->alias, alias) == 0)
	    return "LHS already exists";
	  llist = llist->next;
	}
    }

  entry = (struct virt_entry *) malloc (sizeof (struct virt_entry));
  if (entry == NULL)
    return "Memory allocation error";

  entry->alias = strdup (alias);
  entry->pop_account = strdup (pop);
  if (entry->alias == NULL || entry->pop_account == NULL)
    return "Memory allocation error";
  entry->next = NULL;

  {
    char *p;
    /* Lowercase alias and pop_account.  */
    for (p = entry->alias; *p;)
      *p++ = tolower (*p);
    for (p = entry->pop_account; *p;)
      *p++ = tolower (*p);
  }
  if (*list == NULL)
    *list = entry;
  else
    {
      struct virt_entry *llist = *list;

      /* Add entry to end of list.  */
      while (llist->next)
	llist = llist->next;

      llist->next = entry;
    }

  return NULL;
}

/* Delete an entry from the virtusertable linked list.
   Return NULL on success
   Return an error string on failure.  */
static const char *virt_delete_entry (struct virt_entry **list,
				      const char *alias, const char *pop)
{
  struct virt_entry *prev, *entry;
  const char *p;

  if (alias == NULL)
    return "LHS is empty";
  if (pop == NULL)
    return "RHS is empty";

  /* Check for invalid characters.  */
  for (p = alias; *p; p++)
    if (isspace (*p) || iscntrl (*p))
      return "LHS contains invalid characters";

  if (! isalnum (p[-1]))
    return "LHS must end with an alphanumeric character";

  /* Do the same for the pop account.  */
  for (p = pop; *p; p++)
    if (isspace (*p) || iscntrl (*p))
      return "RHS contains invalid characters";

    if (! isalnum (p[-1]))
      return "RHS must end with an alphanumeric character";

  /* Check whether `alias' already exists.  */
  if (*list != NULL)
    {
      struct virt_entry *llist = *list;
      int found = 0;

      while (llist)
	{
	  if (strcasecmp (llist->alias, alias) == 0)
	    {
	      found = 1;
	      break;
	    }
	  llist = llist->next;
	}
      if (! found)
	/* No alias found.  */
	return "LHS does not exist";
    }

  prev = NULL;
  entry = *list;
  while (entry)
    {
      /* Find a matching entry.  */
      if (strcasecmp (entry->alias, alias) == 0
	  && strcasecmp (entry->pop_account, pop) == 0)
	{
	  if (prev == NULL)
	    {
	      /* Entry is first in the list.  */
	      prev = entry->next;
	      free (entry->alias);
	      free (entry->pop_account);
	      free (entry);
	      *list = prev;
	      return NULL;
	    }

	  /* Unlink entry from middle of the list.  */
	  prev->next = entry->next;
	  free (entry->alias);
	  free (entry->pop_account);
	  free (entry);
	  return NULL;
	}
      prev = entry;
      entry = entry->next;
    }

  /* No matching entry.  */
  errno = ENOENT;
  return "LHS does not exist";
}

/* Free the memory allocated to the virtusertable structure.  */
static void free_virtusertable (struct virt_entry *list)
{
  struct virt_entry *temp;

  while (list)
    {
      temp = list->next;
      free (list);
      list = temp;
    }
}
	  
/* Load /etc/virtusertable into memory and store in a linked list.
   Return a pointer to the head of the linked list on success,
   or NULL on error and set errno.

   The values of errno are:
      ESRCH - Could not find the virtual server root directory
              in /etc/passwd.
      ENOENT - /etc/virtusertable does not exist
      ENOSPC - there are no entries in /etc/virtusertable.  */
static struct virt_entry *load_virtusertable (struct connection *vc,
					      const char *server)
{
  FILE *stream;
  char line[512], *p, alias[256], pop_account[256], *fname;
  struct virt_entry *list;
  int x;
  const char *status;

  read_sendmail_cf (vc->server_root, &fname, NULL, NULL);
  if (fname == NULL)
    return NULL;
  
  sprintf (line, "%s%s", vc->server_root, fname);
  stream = fopen (line, "r");
  if (stream == NULL)
    return NULL;

  free (fname);
  list = NULL;
  while (fgets (line, sizeof (line), stream))
    {
      /* Skip lines with comments.  */
      p = line;
      while (isspace (*p))
	p++;
      if (*p == '#' || *p == '\0')
	continue;

      /* Obtain the e-mail alias.  */
      x = 0;
      while (!isspace (*p) && x < sizeof (alias))
	alias[x++] = *p++;
      alias[x] = '\0';

      /* Skip white space. */
      while (isspace (*p))
	p++;

      /* Obtain the pop3 account.  */
      x = 0;
      while (*p && *p != '\n' && *p != '\r' && ! isspace (*p)
	     && x < sizeof (pop_account))
	pop_account[x++] = *p++;
      pop_account[x] = '\0';

      status = virt_add_entry (&list, alias, pop_account);
      if (status)
	{
	  /* Abort immediately if there is a fault in /etc/virtusertable.
	     We could otherwise suffer from data loss.  */
	  fclose (stream);
	  free_virtusertable (list);
	  io_buffer_store (vc,
			   "-ERR Fatal error: Cannot load virtusertable: %s\n",
			   status);
	  io_buffer_store (vc,
			   "-ERR Erroneous line: LHS `%s', RHS `%s'", alias,
			   pop_account);
	  errno = EFAULT;
	  return NULL;
	}
    }

  fclose (stream);

  if (list == NULL)
    errno = ENOSPC;

  return list;
}

/* Save the virtusertable, stored in the linked `list' to /etc/virtusertable
   on the virtual server `server'.
   Return 0 on success
   Return 1 on read error from sendmail_cf.
   Return 2 if we can't open new virtusertable
   Return 3 if have a fatal error when writing new virtusertable  */
static int save_virtusertable (struct connection *vc, struct virt_entry *list,
			       const char *server)
{
  FILE *stream;
  char line[256], line1[256], *fname;
  int fatal_error, fd;

  read_sendmail_cf (vc->server_root, &fname, NULL, NULL);
  if (fname == NULL)
    return 1;

#if 0
  /* We need to work around the problem of disk quotas.  Do this by
     saving virtusertable as a new file, and if it completes successfully
     then rename it back to the original file.  */
  sprintf (line, "%s%s.1", vc->server_root, fname);
  stream = fopen (line, "w");
  close (fd);

  /* Set file ownership to the admin user.  */
  chown (line, vc->vs->startuid, vc->vs->startuid);

  fatal_error = 0;
  while (list)
    {
      int len;
      len = fprintf (stream, "%s %s\n", list->alias, list->pop_account);
      if (len < strlen (list->alias) + strlen (list->pop_account) + 2)
	{
	  /* Probably a disk quota error.  */
	  fatal_error = 1;
	  break;
	}
      list = list->next;
    }
  fclose (stream);

  if (! fatal_error)
    {
      sprintf (line1, "%s%s", vc->server_root, fname);
      /* Delete original file.  */
      unlink (line1);
      rename (line, line1);
    }
  else
    /* Delete temporary file.  */
    unlink (line);

  return (fatal_error) ? 3 : 0;
#else
  sprintf (line, "%s%s", vc->server_root, fname);
  free (fname);
  stream = fopen (line, "w");
  if (stream == NULL)
    return 2;
  chown (line, vc->vs->startuid, vc->vs->startuid);

  while (list)
    {
      fprintf (stream, "%s %s\n", list->alias, list->pop_account);
      list = list->next;
    }
  fclose (stream);
  return 0;
#endif
}

static int rebuild_virtusertable_db (struct connection *vc, const char *server)
{
  char *fname, *cmd, *fname1, *fname2;
  int status, t1;
  char template[128];

  read_sendmail_cf (vc->server_root, &fname, NULL, NULL);
  if (fname)
    {
      cmd = (char *) malloc (strlen (fname) + 9);
      sprintf (cmd, "hash %s.db", fname);
    }

  /* The consistency of virtusertable.db is quite important if the user
     desires a mail facility.  Keep a backup of the file until we know
     that the new file has re-generated successfully.  Usually it doesn't
     because of a disk quota problem.  */
  fname1 = (char *) malloc (strlen (fname) + strlen (vc->server_root) + 7);
  fname2 = (char *) malloc (strlen (fname) + strlen (vc->server_root) + 7);
  sprintf (fname1, "%s/%s.db", vc->server_root, fname);
  sprintf (fname2, "%s/%s.db.1", vc->server_root, fname);
  rename (fname1, fname2);

  status = open (fname1, O_CREAT | O_TRUNC | O_WRONLY, 0644);
  close (status);
  chown (fname1, vc->vs->startuid, vc->vs->startuid);

  srand (time (NULL));
  t1 = rand () % 1383252;
  sprintf (template, "/tmp/vsd%06u", t1);
  status = run_command (vc, vc->server_root, "admin",
			"/usr/sbin/makemap", cmd, fname, template);
  sprintf (template, "%s/tmp/vsd%06u", vc->server_root, t1);
  if (status == 0)
    {
      /* Makemap returned successfully.  */
      unlink (fname2);
    }
  else
    {
      /* Makemap failed.  */
      char line[1024];
      FILE *stream;
      
      stream = fopen (template, "r");
      if (stream != NULL)
	{
	  while (fgets (line, sizeof (line), stream))
	    io_buffer_store (vc, "%s", line);
	  fclose (stream);
	}

      unlink (fname1);
      rename (fname2, fname1);
    }
  unlink (template);

  free (cmd);
  free (fname);
  free (fname1);
  free (fname2);
  return status;
}

/* Comparison function for qsort.  */
static int virt_compare (const void *a, const void *b)
{
  const struct virt_entry *e1 = (const struct virt_entry *) a;
  const struct virt_entry *e2 = (const struct virt_entry *) b;
  char *d1, *d2;
  int cmp;

  d1 = strchr (e1->alias, '@');
  d2 = strchr (e2->alias, '@');

  /* Might be trying to process an invalid entry.  */
  if (d1 == NULL)
    d1 = e1->alias;
  if (d2 == NULL)
    d2 = e2->alias;

  cmp = strcasecmp (d1 + 1, d2 + 1);
  if (cmp == 0)
    {
      /* Comparing the same domain.  */

      /* Check for the catch all line. Return as a greater than to ensure
	 it always ends up at the bottom of the domain list.  */
      if (e1->alias[0] == '@' && e2->alias[0] == '@')
	return 0;
      if (e1->alias[0] == '@')
	return 1;
      if (e2->alias[0] == '@')
	return 1;

      /* Otherwise return that they are the same.  */
      return strcasecmp (e1->alias, e2->alias);
    }

  return cmp;
}

/* Sort the virtusertable by the domain name in the alias column
   and ensure the catch-all clause comes last.  */
static void sort_virtusertable (struct virt_entry **list)
{
  struct virt_entry *array, *entry, *list_new;
  int elements, x;

  /* Empty list.  */
  if (*list == NULL)
    return;

  /* Count the number of elements.  */
  for (elements = 0, entry = *list; entry; elements ++, entry = entry->next)
    ;
  if (! elements)
    return;
  elements --;

  /* Build the linked list into an array to be sorted by qsort.  */
  array = (struct virt_entry *) malloc ((elements + 1)
					* sizeof (struct virt_entry));
  for (x = 0, entry = *list; x <= elements; x++, entry = entry->next)
    array[x] = *entry;

  /* Sort the list.  */
  qsort (array, elements, sizeof (struct virt_entry), virt_compare);

  /* Free the old list and create a new (sorted) list.  */
  list_new = NULL;
  for (x = 0; x <= elements; x++)
    if (virt_add_entry (&list_new, array[x].alias, array[x].pop_account))
      {
	/* If an error occurs, it ain't so bad.  It just means that our
	   virtusertable won't be sorted.  */
	free_virtusertable (list_new);
	free (array);
	return;
      }

  free_virtusertable (*list);
  *list = list_new;
  free (array);
}

/* List the e-mail addresses associated with `domain'.  */
void sendmail_virt_list_domain (struct connection *vc, int argc, char *argv[])
{
  struct virt_entry *list, *entry;
  char *p;

  list = load_virtusertable (vc, vc->virtual_server);
  if (list == NULL)
    {
      if (errno == ENOSPC)
	/* Nothing in table. Not a problem.  */
	return;
      else if (errno == EFAULT)
	/* Error loading table.  */
	return;
      else if (errno == ESRCH)
	io_buffer_store (vc, "-ERR Virtual server %s does not exist",
			 vc->virtual_server);
      else
	io_buffer_store (vc, "-ERR Fatal error accessing virtusertable: %m");
      return;
    }

  entry = list;
  while (entry)
    {
      p = strchr (entry->alias, '@');
      /* Check for invalid entries.  */
      if (p != NULL && strcasecmp (p + 1, argv[0]) == 0)
	io_buffer_store (vc, "alias=\"%s\" address=\"%s\"\n",
			 entry->alias, entry->pop_account);
      entry = entry->next;
    }

  free_virtusertable (list);
}

void sendmail_virt_modify_table (struct connection *vc, 
				 int argc, 
				 char *argv[],
				 const char *data)
{
  struct virt_entry *list;
  char *ibuf;
  int fatal_error;

  list = load_virtusertable (vc, vc->virtual_server);
  if (list == NULL && errno != ENOSPC && errno != ENOENT)
    {
      if (errno == EFAULT)
	/* Error loading table.  */
	return;
      if (errno == ESRCH)
	io_buffer_store (vc, "-ERR Virtual server %s does not exist",
			 vc->virtual_server);
      else
	io_buffer_store (vc, "-ERR Fatal error accessing virtusertable: %m");
      return;
    }

  fatal_error = 0;
  /* Read entries from the client.  */
  while ((ibuf = io_getline ('\n', &data)))
    {
      int vecc;
      const char *status;
      char **vecv = vsd_argv_parse (' ', ibuf, &vecc);

      /* add <alias> <address>  */
      /* delete <alias> <address>  */
      if (strcmp (vecv[0], "add") == 0)
	{
	  if (strchr (vecv[1], ',') || strchr (vecv[2], ','))
	    {
	      io_buffer_store (vc, "-ERR Comma seperated lists of addresses is not supported in virtusertable");
	      status = NULL;
	    }
	  else
	    status = virt_add_entry (&list, vecv[1], vecv[2]);

	  if (status)
	    {
	      io_buffer_store (vc, "-ERR Cannot add %s: %s", vecv[1], status);
	      if (strcmp (status, "Memory allocation error") == 0)
		fatal_error = 1;
	    }
	}
      else if (strcmp (vecv[0], "delete") == 0)
	{
	  status = virt_delete_entry (&list, vecv[1], vecv[2]);
	  if (status)
	    {
	      io_buffer_store (vc, "-ERR Cannot delete %s: %s", vecv[1],
			       status);
	      if (strcmp (status, "Memory allocation error") == 0)
		fatal_error = 1;
	    }
	}
      free (ibuf);
      free (vecv);
    }

  if (! fatal_error)
    {
      sort_virtusertable (&list);
      fatal_error = save_virtusertable (vc, list, vc->virtual_server);
      if (fatal_error == 0)
	{
	  if (rebuild_virtusertable_db (vc, vc->virtual_server))
	    io_buffer_store (vc, "-ERR Re-building of virtusertable.db failed. Perhaps admin user has exceeded it's disk quota.");
	}
      else if (fatal_error == 1)
	io_buffer_store (vc, "-ERR Read error on sendmail.cf");
      
      else if (fatal_error == 2)
	io_buffer_store (vc, "-ERR Cannot open virtusertable.1 for writing: %m");
      else if (fatal_error == 3)
	io_buffer_store (vc, "-ERR virtusertable not written.  Is admin user over quota ?");
    }
  free_virtusertable (list);
}

void sendmail_cw_list_domain (struct connection *vc, int argc, char *argv[])
{
  struct vsd_file_line *head, *entry;
  char *fname;

  read_sendmail_cf (vc->server_root, NULL, NULL, &fname);
  if (fname == NULL)
    {
      io_buffer_store (vc, "-ERR Cannot find `Fw' in sendmail.cf");
      return;
    }

  if (vsd_load_file (&head, vc->server_root, fname))
    io_buffer_store (vc, "-ERR Could not load sendmail.cw: %m");
  else
    {
      for (entry = head; entry; entry = entry->next)
	{
	  const char *p = vsd_file_skip_spaces (entry->line);
      
	  /* Skip lines with comments.  */
	  if (*p == '#')
	    continue;
	  
	  /* Skip empty lines.  */
	  if (*p == '\0')
	    continue;
  
	  /* Anything else is assumed to be a domain name that the virtual
	     server receives mail for.  */
	  io_buffer_store (vc, "%s\n", p);
	}

      vsd_free_file (head);
    }
  free (fname);
}

/* Add a domain name to /etc/sendmail.cw.  */
void sendmail_cw_add_domain (struct connection *vc, int argc, char *argv[])
{
  struct vsd_file_line *head;
  char *fname;

  read_sendmail_cf (vc->server_root, NULL, NULL, &fname);
  if (fname == NULL)
    {
      io_buffer_store (vc, "-ERR Cannot find `Fw' in sendmail.cf");
      return;
    }

  if (vsd_load_file (&head, vc->server_root, fname))
    {
      io_buffer_store (vc, "-ERR Could not load sendmail.cw: %m");
      free (fname);
      return;
    }

  if (vsd_file_find_line (head, argv[0]))
    io_buffer_store (vc, "-ERR Domain %s already exists", argv[0]);
  else
    {
      /* Prevent the user adding domains that end in a
	 non-alphanumeric.  */
      char *p, *q;
      int error = 0;
      
      q = strdup (argv[0]);
      for (p = q; *p; p++)
	{
	  if (isspace (*p) || iscntrl (*p))
	    {
	      io_buffer_store (vc,"-ERR Domain %s contains non printable characters or spaces",
			       argv[0]);
	      error = 1;
	      break;
	    }
	  /* Make the domain lower-case.  */
	  *p = tolower (*p);
	}
      
      if (! isalnum (p[-1]))
	{
	  io_buffer_store (vc, "-ERR Domain %s must end with an alpha-numeric character",
			   argv[0]);
	  error = 1;
	}
      
      if (! error)
	{
	  /* Insert domain name into /etc/sendmail.cw.  */
	  if (vsd_file_insert_line (vsd_file_tail (head), q))
	    io_buffer_store (vc, "-ERR Could not add domain: %m");
	  else if (vsd_save_file (head, vc->server_root, fname))
	    io_buffer_store (vc, "-ERR Could not save sendmail.cw: %m");
	}
      free (q);
    }
  vsd_free_file (head);
  free (fname);
}

/* Delete a domain name from /etc/sendmail.cw.  */
void sendmail_cw_delete_domain (struct connection *vc, int argc,
				char *argv[])
{
  struct vsd_file_line *head, *entry;
  char *fname;

  read_sendmail_cf (vc->server_root, NULL, NULL, &fname);
  if (fname == NULL)
    {
      io_buffer_store (vc, "-ERR Cannot find `Fw' in sendmail.cf");
      return;
    }

  if (vsd_load_file (&head, vc->server_root, fname))
    io_buffer_store (vc, "-ERR Could not load sendmail.cw: %m");
  else
    {
      /* Find domain in /etc/sendmail.cw.  */
      entry = vsd_file_find_line (head, argv[0]);
      if (entry == NULL)
	io_buffer_store (vc, "-ERR Domain %s not found", argv[0]);
      else
	{
	  /* Delete domain from /etc/sendmail.cw.  */
	  head = vsd_file_delete_line (head, entry);
	  if (head == NULL)
	    {
	      struct vsd_file_line line;
	      line.prev = line.next = NULL;
	      line.line = strdup ("");
	      if (vsd_save_file (&line, vc->server_root, fname))
		io_buffer_store (vc, "-ERR Could not save sendmail.cw: %m");
	      free (line.line);
	    }
	  else
	    if (vsd_save_file (head, vc->server_root, fname))
	      io_buffer_store (vc, "-ERR Could not save sendmail.cw: %m");
	}
      vsd_free_file (head);
    }

  free (fname);
}

/* /etc/aliases manipulation.

   The syntax of /etc/aliases is:
     <lhs><space|tab><rhs>

   Lines beginning with a # are comments.  */

struct aliases_entry
{
  char *lhs;
  char *rhs;
  struct aliases_entry *next;
};

/* Save aliases, stored in the linked `list' to /etc/aliases
   on the virtual server `server'.
   Return 0 on success
   Return 1 on read error from sendmail_cf.
   Return 2 if we can't open new aliases
   Return 3 if have a fatal error when writing new virtusertable  */
static int save_aliases (struct connection *vc, 
			 struct aliases_entry *list,
			 const char *server)
{
  FILE *stream;
  char line[256], line1[256], *fname;
  int fatal_error, fd;

  read_sendmail_cf (vc->server_root, NULL, &fname, NULL);
  if (fname == NULL)
    return 1;

#if 0
  /* We need to work around the problem of disk quotas.  Do this by
     saving virtusertable as a new file, and if it completes successfully
     then rename it back to the original file.  */
  sprintf (line, "%s%s.1", vc->server_root, fname);
  stream = fopen (line, "w");
  close (fd);

  /* Set file ownership to the admin user.  */
  chown (line, vc->vs->startuid, vc->vs->startuid);

  fatal_error = 0;
  while (list)
    {
      int len;
      len = fprintf (stream, "%s %s\n", list->alias, list->pop_account);
      if (len < strlen (list->alias) + strlen (list->pop_account) + 2)
	{
	  /* Probably a disk quota error.  */
	  fatal_error = 1;
	  break;
	}
      list = list->next;
    }
  fclose (stream);

  if (! fatal_error)
    {
      sprintf (line1, "%s%s", vc->server_root, fname);
      /* Delete original file.  */
      unlink (line1);
      rename (line, line1);
    }
  else
    /* Delete temporary file.  */
    unlink (line);

  return (fatal_error) ? 3 : 0;
#else
  sprintf (line, "%s%s", vc->server_root, fname);
  free (fname);
  stream = fopen (line, "w");
  if (stream == NULL)
    return 2;
  chown (line, vc->vs->startuid, vc->vs->startuid);

  while (list)
    {
      fprintf (stream, "%s: %s\n", list->lhs, list->rhs);
      list = list->next;
    }
  fclose (stream);
  return 0;
#endif
}

static int rebuild_aliases (struct connection *vc, const char *server)
{
  char *fname, *cmd, *fname1, *fname2;
  int status, t1;
  char template[128];

  read_sendmail_cf (vc->server_root, NULL, &fname, NULL);

  /* The consistency of aliases.db is quite important if the user
     desires a mail facility.  Keep a backup of the file until we know
     that the new file has re-generated successfully.  Usually it doesn't
     because of a disk quota problem.  */
  fname1 = (char *) malloc (strlen (fname) + strlen (vc->server_root) + 7);
  fname2 = (char *) malloc (strlen (fname) + strlen (vc->server_root) + 7);
  sprintf (fname1, "%s/%s.db", vc->server_root, fname);
  sprintf (fname2, "%s/%s.db.1", vc->server_root, fname);
  rename (fname1, fname2);

  status = open (fname1, O_CREAT | O_TRUNC | O_WRONLY, 0644);
  close (status);
  chown (fname1, vc->vs->startuid, vc->vs->startuid);

  srand (time (NULL));
  t1 = rand () % 1383252;
  sprintf (template, "/tmp/vsd%06u", t1);
  /*
  status = run_command (vc, vc->server_root, "admin",
			"/usr/bin/newaliases", " ", NULL, template);
  */
  status = run_command (vc, "/home/vsd/vs/vsone", "admin",
			"/usr/bin/newaliases", " ", NULL, template);
  sprintf (template, "%s/tmp/vsd%06u", vc->server_root, t1);
  if (status == 0)
    {
      /* newaliases returned successfully.  */
      unlink (fname2);
    }
  else
    {
      /* newaliases failed.  */
      char line[1024];
      FILE *stream;
      
      io_buffer_store (vc, "path: %s", vc->server_root); 

      stream = fopen (template, "r");
      if (stream != NULL)
	{
	  while (fgets (line, sizeof (line), stream))
	    io_buffer_store (vc, "%s", line);
	  fclose (stream);
	}

      unlink (fname1);
      rename (fname2, fname1);
    }
  unlink (template);

  free (cmd);
  free (fname);
  free (fname1);
  free (fname2);
  return status;
}

/* Add an entry to the aliases linked list.
   Return NULL on success
   Return an error string on failure.  */
static const char *aliases_add_entry (struct aliases_entry **list,
				      char *lhs, 
				      char *rhs)
{
  struct aliases_entry *entry;

  if (lhs == NULL || *lhs == '\0')
    return "LHS is empty";
  if (rhs == NULL || *rhs == '\0')
    return "RHS is empty";

  {
    const char *p, *q;
    int at;

    /* Check for invalid characters.  */
    at = 0;
    for (p = lhs; *p; p++)
      {
	if (isspace (*p) || iscntrl (*p))
	  return "LHS contains invalid characters";
	if (*p == '@')
	  at ++;
	if (*p == '.' && p[1] == '.')
	  return "LHS contains `..' sequence";
      }

    if (at >= 2)
      return "LHS contains more than one `@' character";

    if (q = strchr(lhs,':'))
      *q = '\0';

    /* Do the same for the rhs.  */
    at = 0;
    for (p = rhs; *p; p++)
      {
	if (isspace (*p) || iscntrl (*p))
	  return "RHS contains invalid characters";
	if (*p == '@')
	  at ++;

	if (*p == '.' && p[1] == '.')
	  return "RHS contains `..' sequence";
      }

    if (! isalnum (p[-1]))
      return "RHS must end with an alphanumeric character";
  }

  /* Check whether `lhs' already exists.  */
  if (*list != NULL)
    {
      struct aliases_entry *llist = *list;
      while (llist)
	{
	  if (strcasecmp (llist->lhs, lhs) == 0)
	    return "LHS already exists";
	  llist = llist->next;
	}
    }

  entry = (struct aliases_entry *) malloc (sizeof (struct aliases_entry));
  if (entry == NULL)
    return "Memory allocation error";

  entry->lhs = strdup (lhs);
  entry->rhs = strdup (rhs);
  if (entry->lhs == NULL || entry->rhs == NULL)
    return "Memory allocation error";
  entry->next = NULL;

  {
    char *p;
    /* Lowercase lhs and rhs.  */
    for (p = entry->lhs; *p;)
      *p++ = tolower (*p);
    for (p = entry->rhs; *p;)
      *p++ = tolower (*p);
  }
  if (*list == NULL)
    *list = entry;
  else
    {
      struct aliases_entry *llist = *list;

      /* Add entry to end of list.  */
      while (llist->next)
	llist = llist->next;

      llist->next = entry;
    }

  return NULL;
}

/* Delete an entry from the virtusertable linked list.
   Return NULL on success
   Return an error string on failure.  */
static const char *aliases_delete_entry (struct aliases_entry **list,
					 char *lhs, 
					 char *rhs)
{
  struct aliases_entry *prev, *entry;
  const char *p, *q;

  if (lhs == NULL)
    return "LHS is empty";
  if (rhs == NULL)
    return "RHS is empty";

  /* Check for invalid characters.  */
  for (p = lhs; *p; p++)
    if (isspace (*p) || iscntrl (*p))
      return "LHS contains invalid characters";

    if (q = strchr(lhs,':'))
      *q = '\0';

  /* Do the same for the rhs.  */
  for (p = rhs; *p; p++)
    if (isspace (*p) || iscntrl (*p))
      return "RHS contains invalid characters";

    if (! isalnum (p[-1]))
      return "RHS must end with an alphanumeric character";

  /* Check whether `alias' already exists.  */
  if (*list != NULL)
    {
      struct aliases_entry *llist = *list;
      int found = 0;

      while (llist)
	{
	  if (strcasecmp (llist->lhs, lhs) == 0)
	    {
	      found = 1;
	      break;
	    }
	  llist = llist->next;
	}
      if (! found)
	/* No lhs found.  */
	return "LHS does not exist";
    }

  prev = NULL;
  entry = *list;
  while (entry)
    {
      /* Find a matching entry.  */
      if (strcasecmp (entry->lhs, lhs) == 0
	  && strcasecmp (entry->rhs, rhs) == 0)
	{
	  if (prev == NULL)
	    {
	      /* Entry is first in the list.  */
	      prev = entry->next;
	      free (entry->lhs);
	      free (entry->rhs);
	      free (entry);
	      *list = prev;
	      return NULL;
	    }

	  /* Unlink entry from middle of the list.  */
	  prev->next = entry->next;
	  free (entry->lhs);
	  free (entry->rhs);
	  free (entry);
	  return NULL;
	}
      prev = entry;
      entry = entry->next;
    }

  /* No matching entry.  */
  errno = ENOENT;
  return "LHS does not exist";
}

/* Free the memory allocated to the aliases structure.  */
static void free_aliases (struct aliases_entry *list)
{
  struct aliases_entry *temp;

  while (list)
    {
      temp = list->next;
      free (list);
      list = temp;
    }
}
	  
/* Load /etc/aliases into memory and store in a linked list.
   Return a pointer to the head of the linked list on success,
   or NULL on error and set errno.

   The values of errno are:
      ESRCH - Could not find the virtual server root directory
              in /etc/passwd.
      ENOENT - /etc/aliases does not exist
      ENOSPC - there are no entries in /etc/aliases.  */
static struct aliases_entry *load_aliases (struct connection *vc,
					   const char *server)
{
  FILE *stream;
  char line[512];
  char *p; 
  char lhs[256];
  char rhs[1024]; 
  char buffer[1024];
  char *fname;
  struct aliases_entry *list;
  int x;
  const char *status;

  read_sendmail_cf (vc->server_root, NULL, &fname,  NULL);
  if (fname == NULL)
    return NULL;
  
  sprintf (line, "%s%s", vc->server_root, fname);
  stream = fopen (line, "r");
  if (stream == NULL)
    return NULL;

  free (fname);
  list = NULL;
  while (fgets (line, sizeof (line), stream))
    {
      /* Skip lines with comments.  */
      p = line;
      while (isspace (*p))
	p++;
      if (*p == '#' || *p == '\0')
	continue;

      /* Obtain the lhs.  */
      x = 0;
      while (!isspace (*p) && x < sizeof (lhs))
	lhs[x++] = *p++;
      lhs[x - 1] = '\0';

      /* Skip whitespace. */
      while (isspace (*p))
	p++;

      /* Obtain the rhs.  */
      x = 0;
      while (*p && *p != '\n' && *p != '\r' && x < sizeof (rhs))
	rhs[x++] = *p++;
      rhs[x] = '\0';
      
      status = aliases_add_entry (&list, lhs, rhs);
      if (status)
	{
	  /* Abort immediately if there is a fault in /etc/aliases.
	     We could otherwise suffer from data loss.  */
	  fclose (stream);
	  free_aliases (list);
	  io_buffer_store (vc,
			   "-ERR Fatal error: Cannot load aliases: %s\n",
			   status);
	  io_buffer_store (vc,
			   "-ERR Erroneous line: LHS `%s', RHS `%s'", lhs,
			   rhs);
	  errno = EFAULT;
	  return NULL;
	}
    }

  fclose (stream);

  if (list == NULL)
    errno = ENOSPC;

  return list;
}

void sendmail_alist (struct connection *vc, int argc, char *argv[])
{
  struct aliases_entry *list, *entry;
  char *p;

  list = load_aliases (vc, vc->virtual_server);
  if (list == NULL)
    {
      if (errno == ENOSPC)
	/* Nothing in table?  */
	return;
      else if (errno == EFAULT)
	/* Error loading table.  */
	return;
      else if (errno == ESRCH)
	io_buffer_store (vc, "-ERR Virtual server %s does not exist",
			 vc->virtual_server);
      else
	io_buffer_store (vc, "-ERR Fatal error accessing aliases: %m");
      return;
    }

  entry = list;
  while (entry)
    {
      p = entry->lhs;
      /* Check for invalid entries.  */
      // if (p != NULL && strcasecmp (p + 1, argv[0]) == 0)
	io_buffer_store (vc, "lhs=\"%s\" rhs=\"%s\"\n",
			 entry->lhs, entry->rhs);
      entry = entry->next;
    }

  free_aliases (list);
}

void sendmail_amodify (struct connection *vc, 
		       int argc, 
		       char *argv[],
		       const char *data)
{
  struct aliases_entry *list;
  char *ibuf;
  int fatal_error;

  list = load_aliases (vc, vc->virtual_server);
  if (list == NULL && errno != ENOSPC && errno != ENOENT)
    {
      if (errno == EFAULT)
	/* Error loading file.  */
	return;
      if (errno == ESRCH)
	io_buffer_store (vc, "-ERR Virtual server %s does not exist",
			 vc->virtual_server);
      else
	io_buffer_store (vc, "-ERR Fatal error accessing aliases: %m");
      return;
    }

  fatal_error = 0;
  /* Read entries from the client.  */
  while ((ibuf = io_getline ('\n', &data)))
    {
      int vecc;
      const char *status;
      char **vecv = vsd_argv_parse (' ', ibuf, &vecc);

      /* add <key> <value>  */
      /* delete <key> <value>  */
      if (strcmp (vecv[0], "add") == 0)
	{
	  if (strchr (vecv[1], ','))
	    {
	      io_buffer_store (vc, "-ERR Comma seperated lists of keys not supported");
	      status = NULL;
	    }
	  else
	    status = aliases_add_entry (&list, vecv[1], vecv[2]);

	  if (status)
	    {
	      io_buffer_store (vc, "-ERR Cannot add %s: %s", vecv[1], status);
	      if (strcmp (status, "Memory allocation error") == 0)
		fatal_error = 1;
	    }
	}
      else if (strcmp (vecv[0], "delete") == 0)
	{
	  status = aliases_delete_entry (&list, vecv[1], vecv[2]);
	  if (status)
	    {
	      io_buffer_store (vc, "-ERR Cannot delete %s: %s", vecv[1],
			       status);
	      if (strcmp (status, "Memory allocation error") == 0)
		fatal_error = 1;
	    }
	}
      free (ibuf);
      free (vecv);
    }

  if (! fatal_error)
    {
      //      sort_aliases (&list);
      fatal_error = save_aliases (vc, list, vc->virtual_server);
      if (fatal_error == 0)
	{
	  if (rebuild_aliases (vc, vc->virtual_server))
	  io_buffer_store (vc, "-ERR Re-building of aliases.db failed. Perhaps admin user has exceeded it's disk quota.");
	}
      else if (fatal_error == 1)
	io_buffer_store (vc, "-ERR Read error on sendmail.cf");
      
      else if (fatal_error == 2)
	io_buffer_store (vc, "-ERR Cannot open aliases.1 for writing: %m");
      else if (fatal_error == 3)
	io_buffer_store (vc, "-ERR aliases not written.  Is admin user over quota ?");
    }
  free_aliases (list);
}


