/* Module command interface.
   Copyright (c) 1999, 2000 Idaya Ltd.

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "module.h"
#include "sendmail.h"
#include "mod_sendmail.h"

static struct command_tag clist[] =
{
  {
    CMD_SENDMAIL_VLIST, ADMIN_VS,
    "List e-mail aliases for a domain",
    "SENDMAIL_VLIST", 1, "<domain-name>",
    "sendmail_vlist", 3, "<host svr> <vs> <domain>"
  },
  {
    CMD_SENDMAIL_VMODIFY, ADMIN_VS,
    "Modify e-mail aliases",
    "SENDMAIL_VMODIFY", 0, "",
    "sendmail_vmodify", 3, "<host svr> <vs> <filename>"
  },
  {
    CMD_SENDMAIL_CWLIST, ADMIN_VS,
    "List domains configured in sendmail.cw",
    "SENDMAIL_CWLIST", 0, "",
    "sendmail_cwlist", 2, "<host svr> <vs>"
  },
  {
    CMD_SENDMAIL_CWADD, ADMIN_VS,
    "Add a domain to sendmail.cw",
    "SENDMAIL_CWADD", 1, "<domain-name>",
    "sendmail_cwadd", 4, "<host svr> <vs> <user> <domain name>"
  },
  {
    CMD_SENDMAIL_CWDELETE, ADMIN_VS,
    "Delete a domain from sendmail.cw",
    "SENDMAIL_CWDELETE", 1, "<domain-name>",
    "sendmail_cwdelete", 3, "<host svr> <vs> <domain name>"
  },
   {
    CMD_SENDMAIL_ALIST, ADMIN_VS,
    "List entries in /etc/aliases",
    "SENDMAIL_ALIST", 0, "",
    "sendmail_alist", 2, "<host svr> <vs>",
  },
  {
    CMD_SENDMAIL_AMODIFY, ADMIN_VS,
    "Modify entries in /etc/aliases",
    "SENDMAIL_AMODIFY", 0, "",
    "sendmail_amodify", 3, "<host svr> <vs> <filename>",
  },
  {
    0, 0, NULL, NULL, 0, NULL, NULL, 0, NULL
  }
};

/* Return -1 if `cmd' does not exist.
   Return 1 if there is a syntax error.
   Return 0 if everything is cool.  */
int mod_sendmail_check_cmd (int level, const char *cmd, int argc, char *argv[],
			    struct command_tag **list)
{
  if (cmd == NULL)
    {
      /* Return a pointer to all commands.  */
      *list = clist;
      return 0;
    }

  /* Find command in clist.  */
  *list = module_lookup_command (clist, level, cmd);
  if (*list == NULL)
    return -1;  /* Command does not exist.  */

  /* Check command syntax.  Pretty basic at the moment.  */
#ifdef VSD_LIB
  if (argc < (*list)->vsdadm_args)
    return 1; /* Syntax error.  */
#else
  if (argc < (*list)->vsd_args)
    return 1; /* Syntax error.  */
#endif

  /* Syntax is ok.  */
  return 0;
}

#ifdef VSD_LIB
char *mod_sendmail_cmd (void *io, int level, const char *cmd,
			int argc, char **argv)
{
  struct command_tag *list;
  char *status = NULL;

  list = module_lookup_command (clist, level, cmd);
  if (! list)
    return "notexist";

  vsd_set_level (io, list->level);
  switch (list->token)
    {
    case CMD_SENDMAIL_VLIST:
      vsd_send_command (io, NULL, 
			"SENDMAIL_VLIST %s", 
			argv[3] /* domain */); 
      status = vsd_process_reply (io);
      break;
    case CMD_SENDMAIL_VMODIFY:
      {
	char *buffer = vsd_read_file (io, argv[3] /* filename */);
	if (buffer)
	  {
	    vsd_send_command (io, buffer, 
			      "SENDMAIL_VMODIFY");
	    free (buffer);
	    status = vsd_process_reply (io);
	  }
	else
	  status = strerror (errno);
      }
      break;
    case CMD_SENDMAIL_CWLIST:
      vsd_send_command (io, NULL, 
			"SENDMAIL_CWLIST"); 
      status = vsd_process_reply (io);
      break;
    case CMD_SENDMAIL_CWADD:
      vsd_send_command (io, NULL, 
			"SENDMAIL_CWADD %s", 
			argv[3] /* domain */); 
      status = vsd_process_reply (io);
      break;
    case CMD_SENDMAIL_CWDELETE:
      vsd_send_command (io, NULL, 
			"SENDMAIL_CWDELETE %s",
			argv[3] /* domain */); 
      status = vsd_process_reply (io);
      break;
    case CMD_SENDMAIL_ALIST:
      vsd_send_command (io, NULL, 
			"SENDMAIL_ALIST"); 
      status = vsd_process_reply (io);
      break;
    case CMD_SENDMAIL_AMODIFY:
      {
	char *buffer = vsd_read_file (io, argv[3] /* filename */);
	if (buffer)
	  {
	    vsd_send_command (io, buffer, 
			      "SENDMAIL_AMODIFY");
	    free (buffer);
	    status = vsd_process_reply (io);
	  }
	else
	  status = strerror (errno);
      }
      break;
    default:
      status = "mod_sendmail: command not recognised";
      break;
    }
  return status;
}

#else /* ! VSD_LIB */
/* Return 0 if command processed, 1 if not recognised.  */
int mod_sendmail_cmd (struct connection *vc,
		      int level, const char *command,
		      int argc, char *argv[], const char *data)
{
  int token;
  struct command_tag *tag;

  tag = module_lookup_command (clist, level, command);
  if (tag == NULL)
    return 1;

  token = tag->token;
  switch (token)
    {
    case CMD_SENDMAIL_VLIST:
      sendmail_virt_list_domain (vc, argc, argv);
      break;
    case CMD_SENDMAIL_VMODIFY:
      sendmail_virt_modify_table (vc, argc, argv, data);
      break;
    case CMD_SENDMAIL_CWLIST:
      sendmail_cw_list_domain (vc, argc, argv);
      break;
    case CMD_SENDMAIL_CWADD:
      sendmail_cw_add_domain (vc, argc, argv);
      break;
    case CMD_SENDMAIL_CWDELETE:
      sendmail_cw_delete_domain (vc, argc, argv);
      break;
    case CMD_SENDMAIL_ALIST:
      sendmail_alist (vc, argc, argv);
      break;
    case CMD_SENDMAIL_AMODIFY:
      sendmail_amodify (vc, argc, argv, data);
      break;
    default:
      return 1;
      break;
    }

  return 0;
}
#endif /* VSD_LIB */

module sendmail_module =
{
  mod_sendmail_check_cmd,
  mod_sendmail_cmd
};
