/* Module command interface.
   Copyright (c) 1999, 2000 Idaya Ltd.

   This file is part of the Virtual Server Administrator (FreeVSD).

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "module.h"
#include "quota.h"
#include "mod_quota.h"
#include "libvsd.h"

static struct command_tag clist[] =
{
  {
    CMD_QUOTA_VSSET, ADMIN_VS,
    "Set disk quota limits for a virtual server",
    "QUOTA_VSSET", 1, "<quota>{K|M}",
    "quota_vsset", 3, "<host svr> <vs> [<+|->]<kbytes>[<M>]"
  },
  {
    CMD_QUOTA_VSGET, ADMIN_VS,
    "Get disk quota limits for a virtual server",
    "QUOTA_VSGET", 0, "",
    "quota_vsget", 2, "<host svr> <vs>"
  },
  {
    CMD_QUOTA_USERGET, ADMIN_VS,
    "Get quota limits for a user",
    "QUOTA_USERGET", 1, "<user>",
    "quota_userget", 3, "<host svr> <vs> <user>"
  },
  {
    CMD_QUOTA_USERSET, ADMIN_VS,
    "Set quota limits for a user",
    "QUOTA_USERSET", 2, "<user> [+|-]size{K|M}",
    "quota_userset", 4, "<host svr> <vs> <user> [<+|->]<bytes>[<K|M>]"
  },
  {
    CMD_QUOTA_STATS, ADMIN_VS,
    "Get quota details for all users on a server",
    "QUOTA_STATS", 0, "",
    "quota_stats", 2, "<host svr> <vs>"
  },
  {
    0, 0, NULL, NULL, 0, NULL, NULL, 0, NULL
  }
};

/* Return -1 if `cmd' does not exist.
   Return 1 if there is a syntax error.
   Return 0 if everything is cool.  */
int mod_quota_check_cmd (int level, const char *cmd, int argc, char *argv[],
			 struct command_tag **list)
{
  if (cmd == NULL)
    {
      /* Return a pointer to all commands.  */
      *list = clist;
      return 0;
    }

  /* Find command in clist.  */
  *list = module_lookup_command (clist, level, cmd);
  if (*list == NULL)
    return -1;  /* Command does not exist.  */

  /* Check command syntax.  Pretty basic at the moment.  */
#ifdef VSD_LIB
  if (argc < (*list)->vsdadm_args)
    return 1; /* Syntax error.  */
#else
  if (argc < (*list)->vsd_args)
    return 1; /* Syntax error.  */
#endif

  /* Syntax is ok.  */
  return 0;
}


#ifdef VSD_LIB
char *mod_quota_cmd (void *io, int level, const char *cmd,
		     int argc, char **argv)
{
  struct command_tag *list;
  char *status = NULL, *line;
  int uid, hardb, softb, curb;
  dstring_t str;

  list = module_lookup_command (clist, level, cmd);
  if (! list)
    return "notexist";

  vsd_set_level (io, list->level);
  switch (list->token)
    {
    case CMD_QUOTA_VSSET:
      vsd_send_command (io, NULL, "QUOTA_VSSET %s", argv[2]);
      status = vsd_process_reply (io);
      break;
    case CMD_QUOTA_VSGET:
      vsd_send_command (io, NULL, "QUOTA_VSGET");
      status = vsd_process_reply (io);
      break;
    case CMD_QUOTA_STATS:
      str = dstring_new (512);
      vsd_send_command (io, NULL, "QUOTA_STATS");
      status = vsd_process_reply (io);
      while ((line = io_getline ('\n', (const char **) &status)) != NULL)
	{
	  if (strstr (line, "user=") != NULL)
	    dstring_fmt_append (str, "%s\n", line);
	  else
	    {
	      sscanf (line, "hardb=%d softb=%d curb=%d\n",
		      &hardb, &softb, &curb);
	      dstring_fmt_append (str, "subtotal: softb=%d, curb=%d\n",
				  softb, curb);
	    }
	  free (line);
	}
      status = dstring_yield (str);
      dstring_free (str);
      break;
    case CMD_QUOTA_USERSET:
      vsd_send_command (io, NULL, "QUOTA_USERSET %s %s", argv[2], argv[3]);
      status = vsd_process_reply (io);
      break;
    case CMD_QUOTA_USERGET:
      vsd_send_command (io, NULL, "QUOTA_USERGET %s", argv[2]);
      status = vsd_process_reply (io);
      break;
    default:
      status = "mod_quota: command not recognised";
      break;
    }

  return status;
}
#else /* ! VSD_LIB */
/* Return 0 if command processed, 1 if not recognised.  */
int mod_quota_cmd (struct connection *vc, int level,
		   const char *command, int argc, char *argv[],
		   const char *data)
{
  int token;
  struct command_tag *tag;

  tag = module_lookup_command (clist, level, command);
  if (tag == NULL)
    return 1;

  token = tag->token;
  switch (token)
    {
    case CMD_QUOTA_VSSET:
      quota_vsset (vc, argc, argv);
      break;
    case CMD_QUOTA_VSGET:
      quota_vsget (vc, argc, argv);
      break;
    case CMD_QUOTA_USERGET:
      quota_userget (vc, argc, argv);
      break;
    case CMD_QUOTA_USERSET:
      quota_userset (vc, argc, argv);
      break;
    case CMD_QUOTA_STATS:
      quota_stats (vc, argc, argv);
      break;
    default:
      return 1;
      break;
    }

  return 0;
}
#endif /* VSD_LIB */

module quota_module =
{
  mod_quota_check_cmd,
  mod_quota_cmd
};
