/* Module command interface.
   Copyright (c) 1999, 2000 Idaya Ltd.

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include "vsd.h"
#include "module.h"
#include "priv.h"
#include "mod_priv.h"

static struct command_tag clist[] =
{
  {
    CMD_PRIV_MODIFY, ADMIN_VS,
    "Set privileges for a user on a virtual server",
    "PRIV_MODIFY", 2, "<user> <+|-><privilege>",
    "priv_modify", 3, "<host svr> <vs> <user> <+|-><privilege>"
  },
  {
    CMD_PRIV_LIST, ADMIN_VS,
    "List privileges that are assigned to a user",
    "PRIV_LIST", 0, "[<user>]",
    "priv_list", 3, "<host svr> <vs> <user>"
  },
  {
    CMD_PRIV_AVAIL, ADMIN_VS,
    "List all available privileges on a virtual server",
    "PRIV_AVAIL", 0, "",
    "priv_avail", 2, "<host svr> <vs>"
  },
  {
    0, 0, NULL, NULL, 0, NULL, NULL, 0, NULL
  }
};

/* Return -1 if `cmd' does not exist.
   Return 1 if there is a syntax error.
   Return 0 if everything is cool.  */
int mod_priv_check_cmd (int level, const char *cmd, int argc, char *argv[],
			struct command_tag **list)
{
  if (cmd == NULL)
    {
      /* Return a pointer to all commands.  */
      *list = clist;
      return 0;
    }

  /* Find command in clist.  */
  *list = module_lookup_command (clist, level, cmd);
  if (*list == NULL)
    return -1;  /* Command does not exist.  */

  /* Check command syntax.  Pretty basic at the moment.  */
#ifdef VSD_LIB
  if (argc < (*list)->vsdadm_args)
    return 1; /* Syntax error.  */
#else
  if (argc < (*list)->vsd_args)
    return 1; /* Syntax error.  */
#endif

  /* Syntax is ok.  */
  return 0;
}

#ifdef VSD_LIB
char *mod_priv_cmd (void *io, int level, const char *cmd,
		    int argc, char **argv)
{
  char *status = NULL;
  struct command_tag *list = module_lookup_command (clist, level, cmd);
  if (! list)
    return "notexist";

  vsd_set_level (io, list->level);
  switch (list->token)
    {
    case CMD_PRIV_MODIFY:
      vsd_send_command (io, NULL, "PRIV_MODIFY %s %s", argv[1] /* user */,
			argv[2] /* priv */);
      break;
    case CMD_PRIV_LIST:
      if (argv[1])
	vsd_send_line (io, "PRIV_LIST %s", argv[1]);
      else
	vsd_send_line (io, "PRIV_LIST");
      break;
    case CMD_PRIV_AVAIL:
      vsd_send_line (io, "PRIV_AVAIL");
      break;
    default:
      return "mod_priv: command not recognised";
      break;
    }

  status = vsd_process_reply (io);
  return status;
}

#else /* ! VSD_LIB */

/* Return 0 if command processed, 1 if not recognised.  */
int mod_priv_cmd (struct connection *vc,
		  int level, const char *command,
		  int argc, char *argv[], const char *data)
{
  int token;
  struct command_tag *tag = module_lookup_command (clist, level, command);

  if (tag == NULL)
    return 1;

  token = tag->token;
  switch (token)
    {
    case CMD_PRIV_MODIFY:
      priv_modify (vc, argc, argv);
      break;
    case CMD_PRIV_LIST:
      priv_list (vc, argc, argv);
      break;
    case CMD_PRIV_AVAIL:
      priv_avail (vc, argc, argv);
      break;
    default:
      return 1;
      break;
    }

  return 0;
}

#endif

module priv_module =
{
  mod_priv_check_cmd,
  mod_priv_cmd
};
