/* Module command interface.
   Copyright (c) 1999, 2000 Idaya Ltd.

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include "vsd.h"
#include "module.h"
#include "apache.h"
#include "mod_apache.h"

static struct command_tag clist[] = 
{
  {
    CMD_APACHE_VHENUM, ADMIN_VS,
    "List all configured VirtualHost declarations",
    "APACHE_VHENUM", 0, "",
    "apache_vhenum", 2, "<host svr> <vs>"
  },
  {
    CMD_APACHE_VHDELETE, ADMIN_VS,
    "Delete a VirtualHost declaration",
    "APACHE_VHDELETE", 2, "<host svr> <sname>",
    "apache_vhdelete", 4, "<host svr> <vs> <vhost> <sname>"
  },
  {
    CMD_APACHE_VHMODIFY, ADMIN_VS,
    "Modify a VirtualHost declaration",
    "APACHE_VHMODIFY", 2, "<host svr> <sname>",
    "apache_vhmodify", 5, "<host svr> <vs> <vhost> <sname> <vhost file>"
  },
  {
    CMD_APACHE_VHADD, ADMIN_VS,
    "Add a new VirtualHost declaration",
    "APACHE_VHADD", 0, "",
    "apache_vhadd", 3, "<host svr> <vs> <vhost file>"
  },
  {
    CMD_APACHE_VHGET, ADMIN_VS,
    "Get the contents of a VirtualHost declaration",
    "APACHE_VHGET", 2, "<vhost> <servername>",
    "apache_vhget", 4, "<host svr> <vs> <vhost> <servername>"
  },
  {
    0, 0, NULL, NULL, 0, NULL, NULL, 0, NULL
  }
};

/* Return -1 if `cmd' does not exist.
   Return 1 if there is a syntax error.
   Return 0 if everything is cool.  */
int mod_apache_check_cmd (int level, const char *cmd, int argc, char *argv[],
			  struct command_tag **list)
{
  if (cmd == NULL)
    {
      /* Return a pointer to all commands.  */
      *list = clist;
      return 0;
    }

  /* Find command in clist.  */
  *list = module_lookup_command (clist, level, cmd);
  if (*list == NULL)
    return -1;  /* Command does not exist.  */

  /* Check command syntax.  Pretty basic at the moment.  */
#ifdef VSD_LIB
  if (argc < (*list)->vsdadm_args)
    return 1; /* Syntax error.  */
#else
  if (argc < (*list)->vsd_args)
    return 1; /* Syntax error.  */
#endif

  /* Syntax is ok.  */
  return 0;
}

#ifdef VSD_LIB
char *mod_apache_cmd (void *io, int level, const char *cmd,
		      int argc, char **argv)
{
  char *status = NULL;
  struct command_tag *list = module_lookup_command (clist, level, cmd);
  if (! list)
    return "notexist";

  vsd_set_level (io, list->level);
  switch (list->token)
    {
    case CMD_APACHE_VHENUM:
      vsd_send_command (io, NULL, "APACHE_VHENUM");
      break;
    case CMD_APACHE_VHDELETE:
      vsd_send_command (io, NULL, "APACHE_VHDELETE \"%s\" \"%s\"",
			argv[1] /* virtual host */,
			argv[2] /* server name */);
      break;
    case CMD_APACHE_VHGET:
      vsd_send_command (io, NULL, "APACHE_VHGET \"%s\" \"%s\"",
			argv[1] /* virtual host */,
			argv[2] /* server name */);
      break;
    case CMD_APACHE_VHADD:
      vsd_send_command_file (io, argv[1], "APACHE_VHADD %s");
      break;
    case CMD_APACHE_VHMODIFY:
      vsd_send_command_file (io, argv[3], "APACHE_VHMODIFY \"%s\" \"%s\"",
			     argv[1] /* virtual host */,
			     argv[2] /* server name */);
      break;
    default:
      return "mod_apache: command not recognised";
      break;
    }

  status = vsd_process_reply (io);
  return status;
}

#else /* ! VSD_LIB */
/* Return 0 if command processed, 1 if not recognised.  */
int mod_apache_cmd (struct connection *vc,
		    int level, const char *command,
		    int argc, char *argv[], const char *data)
{
  int token;
  struct command_tag *tag = module_lookup_command (clist, level, command);
  if (tag == NULL)
    return 1;

  token = tag->token;
  switch (token)
    {
    case CMD_APACHE_VHENUM:
      httpd_enumerate_virtual_hosts (vc, argc, argv);
      break;
    case CMD_APACHE_VHADD:
      httpd_add_virtual_host (vc, argc, argv, data);
      break;
    case CMD_APACHE_VHDELETE:
      httpd_delete_virtual_host (vc, argc, argv);
      break;
    case CMD_APACHE_VHMODIFY:
      httpd_modify_virtual_host (vc, argc, argv, data);
      break;
    case CMD_APACHE_VHGET:
      httpd_get_virtual_host (vc, argc, argv);
      break;
    default:
      return 1;
      break;
    }

  return 0;
}
#endif /* VSD_LIB */

module apache_module =
{
  mod_apache_check_cmd,
  mod_apache_cmd
};
