/* Apache configuration file manipulation.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>

#include "vsd.h"
#include "libvsd.h"

/* Should be moved to config.h and configure.in.  */
#define HTTPD_CONF "/etc/httpd/conf/httpd.conf"

static char **scan_virtual_hosts (struct vsd_file_line *head, int *total);
static int delete_virtual_host (struct vsd_file_line *head,
				const char *virtual_host,
				const char *server_name);
static int add_virtual_host (struct connection *vc,
			     struct vsd_file_line *head, const char *data);
static struct vsd_file_line *find_virtual_host (struct vsd_file_line *head,
						const char *server,
						const char *servername);
static int check_vh_config (struct connection *vc, struct vsd_file_line *line,
			    struct vsd_file_line *last);

/* Look in `line' for a token that starts with the characters in `directive'
   and ends with `terminator'.  If found, return a malloc`d string consisting
   of the characters between the `directive' and the `terminator'.  Return
   NULL if not found.  */
static char *scan_directive (const char *line, const char *directive,
			     char terminator)
{
  int count;
  const char *p;
  char *buffer;

  line = vsd_file_skip_spaces (line);

  if (strncmp (line, directive, strlen (directive)) != 0)
    return NULL;

  /* Skip past directive.  */
  line += strlen (directive);
  line = vsd_file_skip_spaces (line);

  /* Determine length of directive argument.  */
  for (count = 0, p = line; *p && *p != terminator; count++, p++)
    ;

  if (p == line)
    /* Nothing after the directive.  */
    return NULL;

  /* Strip trailing spaces.  */
  while (*p == '\0' || *p == terminator || isspace (*p))
    p--, count--;

  buffer = (char *) malloc (count + 2);
  strncpy (buffer, line, count + 1);
  buffer[count + 1] = '\0';

  return buffer;
} 

/* Fill `host_list' with a vector of virtual host entries.  */
static char **scan_virtual_hosts (struct vsd_file_line *head, int *total)
{
  struct vsd_file_line *entry;
  int x;
  char *vhost, *sname, **host_list;

  /* Count the number of <VirtualHost ...> entries that exist in the
     file.  */
  *total = 0;
  entry = head;
  while (entry)
    {
      if (strncmp (vsd_file_skip_spaces (entry->line),
		   "<VirtualHost", 12) == 0)
	*total = *total + 1;
      entry = entry->next;
    }

  if (*total == 0)
    return NULL;

  host_list = (char **) malloc (((*total) + 1) * sizeof (char *));
  x = 0;
  entry = head;
  while (entry)
    {
      /* Look for line containing <VirtualHost ...>.  */
      if ((vhost = scan_directive (entry->line, "<VirtualHost", '>')))
	{
	  int found = 0;

	  while (entry)
	    {
	      /* Within the VirtualHost decl. look for a ServerName decl.
	         Both the VirtualHost header and the ServerName decl are
	         used together to uniquify a VirtualHost decl.  */
	      if ((sname = scan_directive (entry->line, "ServerName", '\n')))
		{
		  host_list[x] = (char *) malloc (strlen (vhost)
						  + strlen (sname) + 12);
		  sprintf (host_list[x], "vh=\"%s\" sn=\"%s\"", vhost, sname);
		  x++;
		  found = 1;
		  free (sname);
		  break;
		}
	      if (! strncmp (vsd_file_skip_spaces (entry->line),
			     "</VirtualHost>", 14))
		break;

	      entry = entry->next;
	    }
	  if (! found)
	    {
	      /* No ServerName found.  Use `unknown' instead.  */
	      host_list[x] = (char *) malloc (12 + strlen (vhost) + 1);
	      sprintf (host_list[x], "%s;unknown", vhost);
	      x++;
	    }

	  free (vhost);
	}
      entry = entry->next;
    }

  return host_list;
}

/* Delete a VirtualHost declaration defined by `virtual_host' and `servername'
   from thefile.  Return 0 on success or -1 on error.  */
static int delete_virtual_host (struct vsd_file_line *head,
				const char *virtual_host,
				const char *servername) 
{
  struct vsd_file_line *host_start, *host_end, *link;

  /* Point `host_start' to the start of the VirtualHost declaration.  */
  host_start = find_virtual_host (head, virtual_host, servername);
  if (host_start == NULL)
    return -1;

  /* Point `host_end' to the end of the VirtualHost declaration.  */
  host_end = host_start;
  while (host_end)
    {
      if (strncmp (vsd_file_skip_spaces (host_end->line),
		  "</VirtualHost>", 14) == 0)
	break;

      host_end = host_end->next;
    }

  if (host_end == NULL)
    {
      /* Closing </VirtualHost> couldn't be found.  */
      errno = ESRCH;
      return -1;
    }

  /* Unlink the virtual host from the configuration file.  */
  link = host_start->prev;
  link->next = host_end->next;

  /* Remove next pointer and free the allocated memory.  */
  host_end->next = NULL;
  vsd_free_file (host_start);

  return 0;
}

/* Add a virtual host declaration at the next line after `head'.
   Perform some rudimentary syntax checking on the entry - people don't
   seem to be able to type correctly.

   Return 0 on success, -1 on error.  */
static int add_virtual_host (struct connection *vc,
			     struct vsd_file_line *head,
			     const char *data) 
{
  struct vsd_file_line *first = head, *line;
  char *p, *ibuf;
  int c, c1;

  /* Read incoming data, a line at a time.  */
  while ((ibuf = io_getline ('\n', &data)))
    {
      p = ibuf;
      /* Skip whitespace and ignore blank lines.  */
      while (*p && isspace (*p))
	p++;

      if (*p)
	{
	  /* Clean up by removing trailing spaces.  */
	  char *q = p + strlen (p) - 1;
	  while (q > p && isspace (*q))
	    q--;
	  q[1] = '\0';

	  if (vsd_file_insert_line (head, p))
	    {
	      io_buffer_store (vc, "-ERR Out of memory");
	      return -1;
	    }
	  head = head->next;
	}
      free (ibuf);
    }

  /* Perform some simple checks on the data just received.  Most critically
     we need to ensure <VirtualHost> and </VirtualHost> are the first and
     last lines and that they are spelt correctly because the scanning
     code elsewhere in this source will otherwise not find them.  */
  first = first->next;

  /* Report an error if there are multiple VirtualHost decls.  */
  c = c1 = 0;
  for (line = first; line != head; line = line->next)
    {
      if (strstr (line->line, "<VirtualHost") != NULL)
	c++;
      if (strstr (line->line, "/VirtualHost") != NULL)
	c1++;
    }
  if (c > 1 || c1 > 1)
    {
      io_buffer_store (vc, "-ERR Cannot store multiple VirtualHost declarations");
      return -1;
    }

  /* The first line should solely contain `<VirtualHost xxx>'.  */
  if (strncmp (first->line, "<VirtualHost ", 13) != 0)
    {
      io_buffer_store (vc, "-ERR The first line must be `<VirtualHost ...>'");
      return -1;
    }
  if (first->line[strlen (first->line) - 1] != '>')
    {
      io_buffer_store (vc, "-ERR <VirtualHost line has missing `>' or contains junk");
      return -1;
    }

  if (strcmp (head->line, "</VirtualHost>") != 0)
    {
      /* No </VirtualHost>. A common problem is that the browser has
	 appended it to the end of the previous line.  */
      if ((p = strstr (head->line, "</VirtualHost>")) != NULL)
	{
	  *p = '\0';
	  vsd_file_insert_line (head, "</VirtualHost>");
	}
      else
	{
	  io_buffer_store (vc, "-ERR The last line must be `</VirtualHost>'");
	  return -1;
	}
    }

  return check_vh_config (vc, first, head);
}

/* Extra features. Make sure pathname for DocumentRoot, ErrorLog
   TransferLog and CustomLog exists.

   Return 0 on success, -1 on error.  */
static int check_vh_config (struct connection *vc, struct vsd_file_line *line,
			    struct vsd_file_line *last)
{
  const char *tokens[] = { "DocumentRoot", "ErrorLog", "TransferLog",
			   "CustomLog", NULL };
  int dir;

  for (; line != last; line = line->next)
    {
      struct stat sb;
      char *p;
      const char *status;
      int ret, arglen, len, x;

      for (x = 0; tokens[x] && strncmp (line->line, tokens[x],
					strlen (tokens[x])) != 0; x++)
	;

      if (! tokens[x])
	continue;

      len = strlen (tokens[x]);

      /* Check there is at least one space between token and argument.  */
      if (line->line[len] != ' ' && line->line[len] != '\t')
	{
	  io_buffer_store (vc, "-ERR Need a space between directive and argument");
	  return -1;
	}


      /* Now check the file path exists.  */
      p = line->line + len;
      /* Skip spaces between directive and path.  */
      while (*p && isspace (*p))
	p++;
      dir = (strcmp (tokens[x], "DocumentRoot") == 0) ? 1 : 0;
      status = vsd_check_and_make_path (vc->server_root, p, dir);
      if (status)
	{
	  io_buffer_store (vc, "-ERR %s", status);
	  return -1;
	}
    }
  return 0;
}

/* Return a pointer to the line that starts to define the VirtualHost
   defined by `virtual_host' and `servername'.  If no such VirtualHost
   can be found then return NULL.  */
static struct vsd_file_line *find_virtual_host (struct vsd_file_line *head,
						const char *virtual_host,
						const char *servername)
{
  char *vhost, *sname;

  for (; head; head = head->next)
    if ((vhost = scan_directive (head->line,"<VirtualHost", '>')))
      if (strcmp (vhost, virtual_host) == 0)
	{
	  /* Found a matching VirtualHost. Now look for a matching
	     ServerName.  */
	  struct vsd_file_line *x = head->next;
	  int found = 0;

	  free (vhost);
	  for (x = head->next; x && ! found; x = x->next)
	    {
	      if ((sname = scan_directive (x->line, "ServerName", '\n')))
		{
		  char *p;

		  /* Strip trailing junk from the server name.  */
		  for (p = sname; *p && ! isspace (*p) && *p != '#'; p++)
		    ;
		  *p = '\0';

		  if (strcmp (servername, sname) == 0)
		    {
		      /* We have found a matching VirtualHost decl.  */
		      free (sname);
		      return head;
		    }

		  free (sname);
		}

	      if (! strncmp (vsd_file_skip_spaces (x->line),
			     "</VirtualHost>", 14))
		found = 1;
	    }

	  /* x is either NULL, or host_start points to a VirtualHost
	     with no ServerName.  */
	  if (found && ! strcmp (servername, "unknown"))
	    return head;
	}

  /* Couldn't find VirtualHost.  */
  errno = ENOENT;
  return NULL;
}

/* Return a list of VirtualHost declarations within httpd.conf.  */
void httpd_enumerate_virtual_hosts (struct connection *vc, int argc,
				    char *argv[])
{
  struct vsd_file_line *httpd_conf;

  if (vsd_load_file (&httpd_conf, vc->server_root, HTTPD_CONF))
    io_buffer_store (vc, "-ERR Cannot load httpd.conf: %m");
  else
    {
      int num_hosts;
      char **host_list = scan_virtual_hosts (httpd_conf, &num_hosts);

      if (num_hosts > 0)
	{
	  int x;
	  for (x = 0; x < num_hosts; x++)
	    {
	      io_buffer_store (vc, "%s\n", host_list[x]);
	      free (host_list[x]);
	    }

	  free (host_list);
	}
      vsd_free_file (httpd_conf);
    }
}

/* We can't cope with every syntax error that the user puts
   into httpd.conf.  Hopefully this will help though.  We save
   the updated httpd.conf as a different file, then run `httpd -t'
   on that file to make sure the syntax is OK.  If it is, then
   re-save over the original httpd.conf.  If it isn't then
   delete the temporary and complain.

   Return 0 on success, anything else on error.  */
static int check_syntax_with_httpd (struct connection *vc,
				    struct vsd_file_line *httpd_conf)
{
  unsigned int t1, t2;
  char template[128], template1[128], *httpd_path;
  int fd, status;

  srand (time (NULL));
  t1 = rand () % 1479349;
  t2 = rand () % 2340985;

  sprintf (template, "-t -D SSL -f /tmp/vsd%06u", t1);
  sprintf (template1, "/tmp/vsd%06u", t2);

  vsd_save_file (httpd_conf, vc->server_root, template + 13);

  /* Look for the httpd binary.  */
  if (access ("/usr/sbin/httpd", R_OK | X_OK) == 0)
    httpd_path = "/usr/sbin/httpd";
  else if (access ("/usr/libexec/httpd", R_OK | X_OK) == 0)
    httpd_path = "/usr/libexec/httpd";
  else if (access ("/bin/httpd", R_OK | X_OK) == 0)
    httpd_path = "/bin/httpd";
  else
    {
      io_buffer_store (vc, "-ERR Cannot find web server binary");
      return 1;
    }

  status = run_command (vc, vc->server_root, "web", httpd_path,
			template, NULL, template1);

  sprintf (template, "%s/tmp/vsd%06u", vc->server_root, t2);
  if (status)
    {
      /* Looks like we found an error.  Send httpd output back
	 to the user.  */
      char line[1024];
      FILE *stream;
      
      stream = fopen (template, "r");
      if (stream != NULL)
	{
	  int p = 0;
	  while (fgets (line, sizeof (line), stream))
	    io_buffer_store (vc, "%s", line), p = 1;
	  fclose (stream);

	  if (p == 0)
	    io_buffer_store (vc, "-ERR Webserver check failed. Perhaps admin user has exceeded it's quota");
	}
    }
  unlink (template);
  sprintf (template1, "%s/tmp/vsd%06u", vc->server_root, t1);
  unlink (template1);

  return status;
}

/* Protocol interface for deleting VirtualHost declarations.  */
void httpd_delete_virtual_host (struct connection *vc, int argc, char *argv[])
{
  struct vsd_file_line *httpd_conf;

  if (vsd_load_file (&httpd_conf, vc->server_root, HTTPD_CONF))
    io_buffer_store (vc, "-ERR Cannot load httpd.conf: %m");
  else
    {
      if (! delete_virtual_host (httpd_conf, argv[0], argv[1]))
	{
	  int status = check_syntax_with_httpd (vc, httpd_conf);
	  if (! status)
	    /* No errors found.  Save over original httpd.conf.  */
	    vsd_save_file (httpd_conf, vc->server_root, HTTPD_CONF);
	}
      vsd_free_file (httpd_conf);
    }
}

/* Protocol interface for adding VirtualHost declarations.  */
void httpd_add_virtual_host (struct connection *vc, int argc, char *argv[],
			     const char *data)
{
  struct vsd_file_line *httpd_conf;

  if (vsd_load_file (&httpd_conf, vc->server_root, HTTPD_CONF))
    io_buffer_store (vc, "-ERR Cannot load httpd.conf: %m");
  else
    {
      if (! add_virtual_host (vc, vsd_file_tail (httpd_conf), data))
	{
	  int status = check_syntax_with_httpd (vc, httpd_conf);
	  if (! status)
	    /* No errors found.  Save over original httpd.conf.  */
	    vsd_save_file (httpd_conf, vc->server_root, HTTPD_CONF);
	}
      vsd_free_file (httpd_conf);
    }
}

/* Protocol interface for modifying VirtualHost declarations.
   Basically we find and delete the old VirtualHost decl. and add the
   new VirtualHost decl in it's place.  */
void httpd_modify_virtual_host (struct connection *vc, int argc, char *argv[],
				const char *data)
{
  struct vsd_file_line *httpd_conf, *p;

  if (vsd_load_file (&httpd_conf, vc->server_root, HTTPD_CONF))
    io_buffer_store (vc, "-ERR Cannot load httpd.conf: %m");
  else
    {
      p = find_virtual_host (httpd_conf, argv[0], argv[1]);
      if (p == NULL)
	io_buffer_store (vc, "-ERR VirtualHost '%s' not found", argv[0]);
      else
	{
	  p = p->prev;
	  if (! delete_virtual_host (httpd_conf, argv[0], argv[1]) 
	      && ! add_virtual_host (vc, p, data))
	    {
	      int status = check_syntax_with_httpd (vc, httpd_conf);
	      if (! status)
		/* No errors found.  Save over original httpd.conf.  */
		vsd_save_file (httpd_conf, vc->server_root, HTTPD_CONF);
	    }
	}
      vsd_free_file (httpd_conf);
    }
}

/* Find and return a VirtualHost declaration.  */
void httpd_get_virtual_host (struct connection *vc, int argc, char *argv[])
{
  struct vsd_file_line *httpd_conf, *start, *end;

  if (vsd_load_file (&httpd_conf, vc->server_root, HTTPD_CONF))
    {
      io_buffer_store (vc, "-ERR Cannot load httpd.conf: %m");
      return;
    }

  start = find_virtual_host (httpd_conf, argv[0], argv[1]);
  if (start == NULL)
    io_buffer_store (vc, "-ERR VirtualHost `%s' (%s) not found",
		     argv[0], argv[1]);
  else
    {
      /* Find the end of the VirtualHost declaration.  */
      for (end = start; end; end = end->next)
	if (strncmp (vsd_file_skip_spaces (end->line), "</VirtualHost>",
		     14) == 0)
	  break;

      if (end == NULL)
	io_buffer_store (vc, "-ERR Could not find the end of the VirtualHost declaration");
      else /* Output VirtualHost definition.  */
	for (; start != end->next; start = start->next)
	  io_buffer_store (vc, "%s\n", start->line);
    }
  vsd_free_file (httpd_conf);
}

