/* Add user command.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <pwd.h>
#include "libvsd.h"

int main (int argc, char *argv[])
{
  int c, status, uid = 0, gid = 0;
  char *comment = NULL;
  char *homedir = NULL;
  char *expire_date = NULL;
  char *inactive_time = NULL;
  char *skeleton_dir = NULL;
  char *shell = NULL;
  char *username = NULL;
  char *initial_group = NULL;
  char *additional_groups = NULL;
  int create_group = 1, create_dir = 1;
  struct passwd *pw;

  pw = getpwuid (getuid ());
  if (! pw
      || (strcmp (pw->pw_name, "admin") && strcmp (pw->pw_name, "root")))
    {
      fprintf (stderr, "%s: permission denied\n", argv[0]);
      return 1;
    }

  while ((c = getopt (argc, argv, "-c:mMonrd:e:f:G:g:k:s:u:")) != -1)
    switch (c)
      {
      case 'c':
	comment = optarg;
	break;
      case 'd':
	homedir = optarg;
	break;
      case 'e':
	expire_date = optarg;
	break;
      case 'f':
	inactive_time = optarg;
	break;
      case 'g':
	initial_group = optarg;
	break;
      case 'G':
	additional_groups = optarg;
	break;
      case 'k':
        skeleton_dir = optarg;
	break;
      case 'm': /* Create user dir if it doesn't exist.  */
      	create_dir = 1;
      	break;
      case 'M': /* Don't create user directory.  */
	create_dir = 0;
	break;
      case 's':
	shell = optarg;
	break;
      case 'u':
	uid = atoi (optarg);
	break;
      case 'n':
	create_group = 0;
	break;
      case '?':
	fprintf (stderr, "Unknown option `-%c'.\n", optopt);
	return 1;
      default:
        username = optarg;
	break;
      }

  if (username == NULL)
    {
      fprintf (stderr, "syntax: useradd [-u uid [-o]] [-g group] [-G group,...]\n");
      fprintf (stderr, "          [-d home [-m]] [-s shell] [-c comment] [-m [-k template]]\n");
      fprintf (stderr, "          [-f inactive] [-e expire] [-p passwd] [-n] [-r] name\n");
      return 1;
    }

  if (! homedir)
    {
      homedir = (char *) malloc (sizeof ("/home/") + strlen (username) + 1);
      strcpy (homedir, "/home/");
      strcat (homedir, username);
    }
  if (! initial_group)
    initial_group = username;
  if (! shell)
    shell = "/bin/sh";

  /* By default, create the initial group with the same name as the
     user, unless the `-n' option was passed.  This behaviour mirrors
     RedHat.  */
  if (create_group)
    status = vsd_groupadd ("/", username, &gid);

  status = vsd_useradd ("/", username, homedir, initial_group,
			additional_groups, shell, &uid, comment);
  if (status == 13)
    printf ("%s: user `%s' contains invalid characters\n", argv[0], username);
  else if (status == 14)
    printf ("%s: there are no free uids/gids\n", argv[0]); 
  else if (status == 10)
    printf ("%s: couldn't access /etc/passwd: %s\n", argv[0],
	    strerror (errno));
  else if (status == 9)
    printf ("%s: user `%s' is already in use\n", argv[0], username);
  else if (status == 6)
    printf ("%s: group `%s' does not exist\n", argv[0], initial_group);
  else if (status == 4)
    printf ("%s: uid %d is already in use\n", argv[0], uid);
  else if (status)
    printf ("%s: there is an error somewhere (status = %d)\n", argv[0],
	    status);

  /* Create user home directory.  */
  if (! status && create_dir)
    {
      /* We have to be very careful when creating the user directory.
	 If it already exists, then don't create it.  If it is owned
	 by another user, then don't chown it (particular for root owned
	 directories).  Don't allow home directory to be created in a
	 root owned directory.  */
      const char *status = vsd_check_and_make_path ("/", homedir, 1);

      if (status)
	{
	  printf ("%s: cannot create home directory: %s\n", argv[0], status);
	  return 2;
	}
      else
	{
	  struct stat sb;
	  uid_t min_uid, max_uid;

	  vsd_get_uidgid ("/", &min_uid, &max_uid, NULL, NULL);
	  if (stat (homedir, &sb))
	    printf ("%s: stat of %s failed: %s\n", argv[0], homedir,
		    strerror (errno));
	  if (sb.st_uid < min_uid || sb.st_uid > max_uid)
	    printf ("%s: cannot chown home directory: permission denied\n",
		    argv[0]);
	  else
	    {
	      /* vsd_check_and_make_path will create the home directory but it
		 will probably have the wrong ownership.  Fix that here.  */
	      chown (homedir, uid, gid);
	    }
	}
    }

  return status;
}
