/* Wrapper program to restrict access to a binary to users
   with the privilege system.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <alloca.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <pwd.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/resource.h>
#include <sys/stat.h>

#ifndef PROCNAME
#error PROCNAME not defined
#endif

#ifdef PRIVILEGE
#include "../libvsd/libvsd.h"
#endif

static const char *basename (const char *name)
{
  const char *p = strrchr (name, '/');
  return p ? p + 1 : name;
}

/* Global variables. These are defined on-the-fly by the program
   that customises the restrict wrapper for the specified binary. */
int main (int argc, char **argv, char **environ) 
{ 
  struct passwd *pw; 
  char template[128]; 
  struct rlimit lim; 
  struct stat sb;
 
  /* Limit CPU time to 60 seconds.  */	
  lim.rlim_cur = 60;
  lim.rlim_max = 60;
  setrlimit (RLIMIT_CPU, &lim); 

#if 1
  lim.rlim_cur = 48*1024*1024;
  lim.rlim_max = 48*1024*1024;
  setrlimit (RLIMIT_AS, &lim);
#else 
  /* Limit memory to 48Mb.  */	
  lim.rlim_cur = 48*1024*1024;	
  lim.rlim_max = 48*1024*1024;	
  setrlimit (RLIMIT_DATA, &lim);

  /* Limit stack to 2Mb.  */	
  lim.rlim_cur = 2*1024*1024;	
  lim.rlim_max = 2*1024*1024;	
  setrlimit (RLIMIT_STACK, &lim); 

  /* Limit file size to 128Mb.  */	
  lim.rlim_cur = 256*1024*1024;	
  lim.rlim_max = 256*1024*1024;	
  setrlimit (RLIMIT_FSIZE, &lim); 
#endif

  /* Only allow 64 processes per user.  */
  lim.rlim_cur = 64;	
  lim.rlim_max = 64;	
  setrlimit (RLIMIT_NPROC, &lim); 
 
  /* Set core dump files to 0 bytes.  */ 
  lim.rlim_cur = 0; 
  lim.rlim_max = 0; 
  setrlimit (RLIMIT_CORE, &lim); 
 
  pw = getpwuid (getuid ()); 
  if (pw == NULL) 
    { 
      fprintf (stderr, "Permission denied - user unknown\n"); 
      return 1; 
    } 
 
#ifdef PRIVILEGE
  if (vsd_priv_access (pw->pw_name, PRIVILEGE))
    { 
      fprintf (stderr,
               "Permission denied (user doesn't have privilege %s)\n",
	       PRIVILEGE);
      return 1; 
    } 
#endif

#if 0
  template = (char *) alloca (strlen (PROCNAME) + 12);
  strcpy (template, "/sbin/vsd/");
  strcat (template, basename (argv[0]));
#endif
  strcpy (template, "/sbin/vsd/" PROCNAME);
  if (stat (template, &sb))
    {
      fprintf (stderr, "Cannot execute %s: %m\n", argv[0]);
      return 1;
    }

  execve (template, argv, environ); 
  printf ("can't execute %s: %s\n", template, strerror (errno)); 
  return 1;
}
