/* Utilities for string processing.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <string.h>
#include <stdlib.h>
#include "libvsd.h"

/* Return the length of string `s' terminated by `term'.  */
size_t string_lenc (const char *s, char term)
{
  const char *e = s - 1;
  while (*++e != term)
    ;

  return e - s;
}

/* Copy string `s' to `d', using `term' as the string terminator
   character.  */
char *string_cpyc (char *d, const char *s, char term)
{
  char *d1 = d;

  while ((*d1++ = *s++) != term)
    ;

  /* Terminate the string with a NULL.  */
  d1[-1] = '\0';
  return d;
}


/* Operations on a character seperated list.  */

/* `list' is a pointer to a string consisting of tokens separated by the
   character `sep'.  Search `list' for the token `token' and return a
   pointer to the token if the search is successful, otherwise return NULL.  */
const char *string_member (const char *list, const char *token, const char sep)
{
  int token_len;
  const char *e;

  /* Fast case, `list' contains one entry.  */
  if (strcmp (list, token) == 0)
    return list;

  /* For speed, we can quickly search though the list only comparing
     entries where the name is the same length as `token'.  */
  token_len = strlen (token);

  while (1)
    {
      /* Set a marker for the start of the token.  */

      /* Find the terminating comma for the token.  */
      e = list;
      while (*e && *e != sep)
	e++;

      if (e - list == token_len)
	{
	  /* Found a token with a name of the same length, so compare the
	     name to see if we have a match.  */
	  if (strncmp (token, list, token_len) == 0)
	    return list;
	}

      /* Check if we have reached the end of the list.  */
      if (*e == '\0')
	return NULL; 

      /* Move to the next entry in the list, skipping past the comma
	 on the way.  */
      list = e + 1;
    }

  /* Stop compiler warning.  */
  return NULL;
}

/* Delete `token' from a `list' of strings separated by character `sep'.
   Return 0 on success, 1 if the token is not on the list.  */
int string_member_delete (char *list, const char *token, const char sep)
{
  char *p = (char *) string_member (list, token, sep);
  char *q;

  if (p == NULL)
    return 1; /* Token is not on the list.  */

  q = strchr (p, sep);
  if (q == NULL)
    {
      if (p > list)
	p[-1] = '\0'; /* Token is the last entry on the list.  */
      else
	*p = '\0'; /* Token is the only entry on the list.  */
    }
  else
    /* Token is somewhere in the middle of the list.  */
    memmove (p, q + 1, strlen (q + 1) + 1);

  return 0;
}

/* Add `token' as a new member of `list', separated by `sep'.

   The `sep' seperated list of tokens might not be at the start of the
   string.  For this situation `search_from' will point to the start
   of the `sep' seperated list.

   Return 0 on success, 1 on failure.  */
int string_member_add (char **list, char *search_from,
		       const char *token, const char sep)
{
  char *p;
  int len;
  char buf[4];

  if (string_member (search_from, token, sep) != NULL)
    return 0; /* token is already a member.  */

  /* Allocate space for the token.  */
  len = strlen (*list) + strlen (token) + 2;
  *list = (char *) realloc (*list, len);
  if (! *list)
    return 1;

  /* Look to the end of the string to see if we need to add the `sep'
     character before we append the token.  */
  p = *list;
  if (*p)
    {
      for (; *p; p++)
	;
      p--;
    }
  /* If the token is the first item in the list then we don't want to
     start with a comma.  */
  if (*p != sep && *p != ':' && *p)
    { 
      buf[0] = sep;
      buf[1] = '\0';
      strcat (*list, buf);
    }
  strcat (*list, token);

  return 0;
}
