/* Provide a portable interface for obtaining process information.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef TARGET_LINUX
#include <dirent.h>
#include <sys/stat.h>
#endif
#ifdef TARGET_FREEBSD
#include <kvm.h>
#include <sys/sysctl.h>
#include <sys/param.h>
#include <sys/filedesc.h>
#include <sys/user.h>
#endif

#include "libvsd.h"

/* Return a list of all processes that belong to a virtual server.
   Assume we are doing this from the root of the box.
   Memory Terminate the list with a -1.  */
int *vsd_enum_procs (const char *vs)
{
  int procs_size = 16;
  int procs_index = 0;
  int *procs = (int *) malloc (procs_size * sizeof (int));

#ifdef TARGET_LINUX
  /* Linux stores all process information in /proc/<pid>.  Our method
     is to examine /proc/<pid>/root and if we get permission denied
     then we are not the owner of that process.  */
  DIR *dir = opendir ("/proc");
  struct dirent *ent;

  while ((ent = readdir (dir)))
    if (*ent->d_name > '0' && *ent->d_name <= '9')
      {
	int len;
	char path[64], buf[256];
	struct stat sb;
	char *p;
	
	/* Found a process entry.  */
        sprintf (path, "/proc/%s", ent->d_name);
	if (stat (path, &sb) == -1)
	  continue; /* Process no longer exists.  */

	/* Read the symbolic link for /proc/<pid>/root. The virtual server
	   name should be somewhere within it.  */
	strcat (path, "/root");

	len = readlink (path, buf, sizeof (buf));
	if (len == -1)
	  continue;

	buf[len] = '\0';

	/* Attempt to match `/<vs>' to symbolic link.  */
	path[0] = '/'; strcpy (path + 1, vs);
	if ((p = strstr (buf, path)) && strcmp (p, path) == 0)
	  {
	    /* Add process to the list.  */
	    procs[procs_index++] = atoi (ent->d_name);
	    if (procs_index >= procs_size)
	      {
		procs_size += 8;
		procs = (int *) realloc (procs, procs_size * sizeof (int));
	      }
	  }
      }

  closedir (dir);
#endif

#ifdef TARGET_FREEBSD
  kvm_t *kd;

  kd = kvm_open (NULL, NULL, NULL, O_RDONLY, "kvm_open");
  if (kd != NULL)
    {
      struct kinfo_proc *proc;
      int cnt, x;

      /* Get the list of symbols we want to access in the kernel.  */
      proc = kvm_getprocs (kd, KERN_PROC_ALL, 0, &cnt);
      for (x = 0; x <= cnt; x++)
	{
#if 0
	  proc[x].kp_proc.p_fd.fd_rdir /* root dir */
	    proc[x].kp_proc.p_fd.fd_jdir /* root dir */
#endif
	    procs[procs_index++] = proc[x].kp_proc.p_pid;
	  if (procs_index >= procs_size)
	    {
	      procs_size += 8;
	      procs = (int *) realloc (procs, procs_size * sizeof (int));
	    }
	}
      kvm_close (kd);
    }
  else
    /* Probably not got access to /dev/mem.  */
    return NULL;

#endif

  procs[procs_index] = -1;
  return procs;
}

/* Return a list of all processes that belong to a virtual server.
   Assume we are chroot'ed to the VS when we do this.
   Memory Terminate the list with a -1.  */
int *vsd_enum_procs_vs (const char *vs)
{
  int procs_size = 16;
  int procs_index = 0;
  int *procs = (int *) malloc (procs_size * sizeof (int));

#ifdef TARGET_LINUX
  /* Linux stores all process information in /proc/<pid>.  Our method
     is to examine /proc/<pid>/root and if we get permission denied
     then we are not the owner of that process.  */
  DIR *dir = opendir ("/proc");
  struct dirent *ent;

  while ((ent = readdir (dir)))
    if (*ent->d_name > '0' && *ent->d_name <= '9')
      {
	char path[64];
	struct stat sb;

	/* Found a process entry.  */
        sprintf (path, "/proc/%s", ent->d_name);
	if (stat (path, &sb) == -1)
	  continue; /* Process no longer exists.  */

	/* Attempt to open /proc/<pid>/root. If this fails then this process
	   does not belong to virtual server `vs'.  */
	strcat (path, "/root");

#ifdef HAVE_ACCESS
	if (access (path, R_OK | X_OK))
	  continue;
#else
#ifdef HAVE_OPEN
	fd = open (path, O_RDONLY);
	if (fd == -1)
	  continue;
	close (fd);
#endif
#endif
	/* Add process to the list.  */
	procs[procs_index++] = atoi (ent->d_name);
	if (procs_index >= procs_size)
	  {
	    procs_size += 8;
	    procs = (int *) realloc (procs, procs_size * sizeof (int));
	  }
      }

  closedir (dir);

#endif

#ifdef TARGET_FREEBSD
  kvm_t *kd;

  kd = kvm_open (NULL, NULL, NULL, O_RDONLY, "kvm_open");
  if (kd != NULL)
    {
      struct kinfo_proc *proc;
      int cnt, x;

      /* Get the list of symbols we want to access in the kernel.  */
      proc = kvm_getprocs (kd, KERN_PROC_ALL, 0, &cnt);
      for (x = 0; x <= cnt; x++)
	{
	  procs[procs_index++] = proc[x].kp_proc.p_pid;
	  if (procs_index >= procs_size)
	    {
	      procs_size += 8;
	      procs = (int *) realloc (procs, procs_size * sizeof (int));
	    }
	}
      kvm_close (kd);
    }
  else
    /* Probably not got access to /dev/kmem.  */
    return NULL;
#endif

  procs[procs_index] = -1;
  return procs;
}

/* Return the parent process id of process `p'.  */
pid_t vsd_getppid (pid_t p)
{
  pid_t ppid = -1;

#ifdef TARGET_LINUX
  char fname[128];
  FILE *handle;

  /* Linux stores parent process id info in /proc/<pid>/stat.  */
  sprintf (fname, "/proc/%d/stat", p);
  handle = fopen (fname, "r");
  if (handle != NULL)
    {
      fscanf (handle, "%*d %*s %*s %u ", &ppid);
      fclose (handle);
    }
#endif

#ifdef TARGET_FREEBSD
  kvm_t *kd;

  kd = kvm_open (NULL, NULL, NULL, O_RDONLY, "kvm_open");
  if (kd != NULL)
    {
      struct kinfo_proc *proc;
      int cnt, x;

      /* Get the list of symbols we want to access in the kernel.  */
      proc = kvm_getprocs (kd, KERN_PROC_ALL, 0, &cnt);
      for (x = 0; x <= cnt; x++)
	if (proc[x].kp_proc.p_pid == p)
	  {
	    /* ppid = proc[x].kp_proc.p_pid;*/
	    ppid = proc[x].kp_proc.p_pptr->p_pid;
	    break;
	  }
      kvm_close (kd);
    }
#endif

  return ppid;
}

#ifdef TEST
int main (int argc, char *argv[])
{
  int *procs;
  int x;

  if (argc < 2)
    {
      printf ("syntax: %s <virtual server>\n", argv[0]);
      return 1;
    }

  procs = vsd_enum_procs (argv[1]);
  if (procs == NULL)
    return 1;

    
  x = 0;
  printf ("Process on virtual server '%s': ", argv[1]);
  while (procs[x] != -1)
    {
      printf ("%d, ", procs[x]);
      x++;
    }
  printf ("\n");
  free (procs);
  return 0;
}
#endif

