/* User privilege/validation checking.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>

#include "libvsd.h"

/* File format for the privilege file:
     <privilege>:[<username>][,<username>][,<username>][,...]
     <privilege>:[<username>][,<username>][,<username>][,...]
*/

/* Pathname to the privilege file.  */
#define VSD_PRIV "/etc/vsd/priv"

int vsd_priv_load (struct vsd_file_line **list, const char *server_root)
{
  return vsd_load_file (list, server_root, VSD_PRIV);
}

int vsd_priv_save (struct vsd_file_line *list, const char *server_root)
{
  return vsd_save_file (list, server_root, VSD_PRIV);
}

void vsd_priv_free (struct vsd_file_line *list)
{
  vsd_free_file (list);
}

/* Return a pointer to the line if `username' is in list, otherwise
   return NULL.  */
static const char *find_member (const char *list, const char *username)
{
  if (username == NULL || list == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  if (strchr (list, ':'))
    {
      /* We have been supplied a list which contains the privilege name.
	 Skip past it.  */
      while (*list && *list != ':')
	list++;

      if (*list == '\0')
	{
	  /* Something weird happened.  */
	  errno = EINVAL;
	  return NULL;
	}
      /* Skip over the colon.  */
      list ++;
    }

  /* Usernames are comma separated. Check each user one at a time.  */
  return string_member (list, username, ',');
}

/* Return 0 if `username' is in `list'.
   Return 1 if `username' is not in `list'.
   Return -1 on error.  */
static int member (const char *list, const char *username)
{
  if (find_member (list, username))
    return 0;

  return 1;
}

static struct vsd_file_line *find_privilege (struct vsd_file_line *head,
					      const char *privilege)
{
  struct vsd_file_line *line;

  if (head == NULL || privilege == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  for (line = head; line; line = line->next)
    {
      char *p;

      /* Skip blank lines.  */
      if (line->line[0] == '\0')
	continue;

      p = line->line;
      /* Skip initial whitespace.  */
      while (isspace (*p))
	p++;

      /* Skip comment lines.  */
      if (*p == '#')
	continue;

      /* Is this the privilege we are looking for ? */
      if (strncmp (privilege, p, strlen (privilege)) == 0)
	return line;
    }

  /* Privilege was not found.  */
  errno = ENOENT;
  return NULL;
}

/* Return 0 if `username' has been granted `privilege' on the virtual
   server.
   Return 1 if `username' has not been granted `privilege'.
   Return -1 on error and errno is set appropriately:
       EINVAL - Supplied arguments are invalid.
       ENOENT - Privilege does not exist.  */
int vsd_check_privilege (struct vsd_file_line *priv_file,
			  const char *username, const char *privilege)
{
  struct vsd_file_line *line;

  if (priv_file == NULL || username == NULL || privilege == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  /* Find the privilege.  */
  line = find_privilege (priv_file, privilege);
  if (line == NULL)
    return -1; /* Privilege does not exist.  */

  return member (line->line, username);
}

int vsd_priv_add_member (struct vsd_file_line *priv_file,
			 const char *privilege, const char *username)
{
  struct vsd_file_line *line;
  char *p;

  if (priv_file == NULL || username == NULL || privilege == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  /* Find the privilege.  */
  line = find_privilege (priv_file, privilege);
  if (line == NULL)
    return -1; /* Privilege does not exist.  */

  p = strrchr (line->line, ':') + 1;
  return string_member_add (&line->line, p, username, ',');
}

int vsd_priv_delete_member (struct vsd_file_line *priv_file,
			    const char *privilege, const char *username)
{
  struct vsd_file_line *line;
  char *p;

  if (priv_file == NULL || username == NULL || privilege == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  /* Find the privilege.  */
  line = find_privilege (priv_file, privilege);
  if (line == NULL)
    return -1; /* Privilege does not exist.  */

  p = strchr (line->line, ':');
  if (*p != '\0')
    string_member_delete (p + 1, username, ',');
  return 0;
}

/* List privileges that can be set for a user on a virtual server.  */
char **vsd_priv_list_privs (struct vsd_file_line *head, int *argcount)
{
  int argc;
  char **argv;
  char temp[256];

  argv = NULL;
  argc = 0;
  while (head)
    {
      if (head->line)
	{
	  char *p = head->line;

	  while (isspace (*p))
	    p++;
	  if (*p != '#')
	    {
	      char *q = temp;
	      while (*p && (q - temp < sizeof (temp) - 1) && *p != ':')
		*q++ = *p++;
	      *q = '\0';

	      argv = vsd_argv_add (argv, &argc, temp);
	    }
	}
      head = head->next;
    }

  *argcount = argc - 1;
  return argv;
}

/* Return a list of privileges that `username' is a member of.  */
char **vsd_priv_get_members (struct vsd_file_line *priv_file,
			     const char *username, int *membcount)
{
  struct vsd_file_line *line;
  char **privv, **membv;
  int x, privc;

  if (priv_file == NULL || username == NULL || membcount == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  membv = NULL;
  *membcount = 0;
  /* Get a vector of privileges available on the system.  */
  privv = vsd_priv_list_privs (priv_file, &privc);
  for (x = 0; x <= privc; x++)
    {
      /* Create a vector of privileges that `username' is a member of.  */
      line = find_privilege (priv_file, privv[x]);
      if (find_member (line->line, username))
	membv = vsd_argv_add (membv, membcount, privv[x]);
    }

  vsd_argv_free (privv, privc);
  return membv;
}

static int on_list (char **list, const char *member)
{
  while (*list)
    {
      if (strcmp (*list, member) == 0)
	return 1;

      list++;
    }

  return 0;
}

/* Return a list of privileges (implemented as groups) that `username' is
   a member of.  */
char **vsd_priv_get_group_members (const char *server_root,
				   const char *username, int *membcount)
{
  struct group *gp;
  FILE *grp_stream;
  char *grps[] = { "admin-users", NULL };
  char strip[32];
  int x;
  char **membv;

  grp_stream = vsd_opengrp (server_root);
  if (grp_stream == NULL)
    return NULL;

  /* Convert comma-separated list into a vector.  */
  membv = NULL;
  *membcount = 0;

  x = 0;
  while (grps[x] != NULL)
    {
      gp = vsd_getgrnam (grp_stream, grps[x]);
      if (gp != NULL)
	if (on_list (gp->gr_mem, username))
	  {
	    char *p;

	    strcpy (strip, grps[x]);
	    p = strip;
	    while (*p && *p != '-')
	      p++;
	    *p = '\0';

	    membv = vsd_argv_add (membv, membcount, strip);
	  }
      x++;
    }

  vsd_closegrp (grp_stream);

  return membv;
}

/* Return 0 if `username' has privilege `priv'.  */
int vsd_priv_access (const char *username, const char *priv)
{
  struct vsd_file_line *file;
  int status;

  if (vsd_priv_load (&file, "/"))
    {
      printf ("vsd_priv_access: failed with error: %s\n", strerror (errno));
      return -1;
    }

  status = vsd_check_privilege (file, username, priv);
  vsd_priv_free (file);

  return status;
}

/* Delete a user from the privilege database.  */
int vsd_priv_delete_user (const char *server_root, const char *user)
{
  struct vsd_file_line *priv_file;
  char **membv;
  int membc, x;

  vsd_priv_load (&priv_file, server_root);

  /* Get a list of privileges that `old_user' is a member of.  */
  membv = vsd_priv_get_members (priv_file, user, &membc);

  /* Delete `old_user' from the privileges and add `new_user' to
     the privilegs.  */
  for (x = 0; x <= membc; x++)
    vsd_priv_delete_member (priv_file, membv[x], user);

  vsd_priv_save (priv_file, server_root);
  vsd_priv_free (priv_file);
  vsd_argv_free (membv, membc);

  return 0;
}

/* Rename a user in the privilege database.  */
int vsd_priv_rename_user (const char *server_root,
			  const char *old_user, const char *new_user)
{
  struct vsd_file_line *priv_file;
  char **membv;
  int membc, x;

  vsd_priv_load (&priv_file, server_root);

  /* Get a list of privileges that `old_user' is a member of.  */
  membv = vsd_priv_get_members (priv_file, old_user, &membc);

  /* Delete `old_user' from the privileges and add `new_user' to
     the privilegs.  */
  for (x = 0; x < membc; x++)
    {
      vsd_priv_delete_member (priv_file, membv[x], old_user);
      vsd_priv_add_member (priv_file, membv[x], new_user);
    }

  vsd_priv_save (priv_file, server_root);
  vsd_priv_free (priv_file);
  vsd_argv_free (membv, membc);

  return 0;
}


