/* Change entries in the user group file.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <grp.h>
#include "libvsd.h"

int vsd_group_load (struct vsd_file_line **list, const char *server_root)
{
  return vsd_load_file (list, server_root, "/etc/group");
}

int vsd_group_save (struct vsd_file_line *list, const char *server_root)
{
  return vsd_save_file (list, server_root, "/etc/group");
}

void vsd_group_free (struct vsd_file_line *list)
{
  vsd_free_file (list);
}

/* Return 1 if a match for `gid' is found in `line'. Return 0 if not.  */
static int match (struct vsd_file_line *line, int gid)
{
  int id;

  for (; line; line = line->next)
    {
      char *p = line->line;
      int i = 0;
      /* Search for the GID.  */
      while (*p && i != 2)
	if (*p++ == ':')
	  i++;

      /* Convert to a number.  */
      id = strtol (p, NULL, 10);
      if (id == gid)
	return 1;
    }

  return 0;
}
/* If `gid' is not 0 then check it is unique.  If it is unique then return
   `gid', otherwise return 0.
   If `gid' is 0 then find a unique group id and return that value. If
   there are no spare entries then return 1.  */
static gid_t unique_gid (struct vsd_file_line *file, gid_t gid,
			 gid_t min, gid_t max)
{
  int unique = (gid) ? 1 : 0;

  /* Search through the group file.  Start with the lowest group id and
     keep searching until we either exceed `max' or find something unique.  */
  if (unique)
    return (match (file, gid)) ? 0 : gid;

  for (gid = min; gid <= max; gid ++)
    if (! match (file, gid))
      return gid;
  return 1;
}

/* Create a new group `group_name' and add it to the user group file.
   If `gid' is not within the virtual server range then allocate a group id. 
   Return 0 on success.
   Return 1 if the parameters are invalid.
   Return 4 if the gid is not unique.  
   Return 9 if the group name is not unique.
   Return 10 on file error.
   Return 14 if there are no more gids available.  */
int vsd_groupadd (const char *server_root, const char *group_name, gid_t *gid)
{
  struct vsd_file_line *file, *line;
  gid_t min, max;
  int ret;

  if (vsd_group_load (&file, server_root))
    return 10;

  /* Is `gid' valid within the virtual server gid range ?  */
  vsd_get_uidgid (server_root, NULL, NULL, &min, &max);
  if (*gid < min || *gid > max)
    {
      /* No, or the user wants us to allocate a gid. */
      *gid = unique_gid (file, 0, min, max);
      if (*gid == 1)
	return 14;
    }
  else
    /* Is `gid' unique ?  */
    if (unique_gid (file, *gid, min, max) != *gid)
      {
	/* No.  */
	vsd_group_free (file);
	return 4; /* No.  */
      }

  /* Check `group_name' is unique.  */
  line = vsd_file_find_entry (file, group_name);
  if (line)
    ret = 9;
  else if (errno == EINVAL)
    ret = 1;
  else
    {
      /* Create the new group entry.  */
      dstring_t ds = dstring_new (64);
      dstring_fmt_append (ds, "%s::%d:", group_name, *gid);
      /* Add the entry to the group file.  */
      vsd_file_insert_line (vsd_file_tail (file), dstring_ptr (ds));
      vsd_group_save (file, server_root);
  
      dstring_free (ds);
      ret = 0;
    }

  vsd_group_free (file);
  return ret;
}

/* Delete a `group_name' from the user group file.
   Return 0 on success.
   Return 1 if the parameters are invalid.
   Return 6 if group does not exist.
   Return 10 on file error.  */
int vsd_groupdel (const char *server_root, const char *group_name)
{
  struct vsd_file_line *file, *line;
  int ret;

  if (vsd_group_load (&file, server_root))
    return 10;

  /* Match the line and the group.  */
  line = vsd_file_find_entry (file, group_name);
  if (line == NULL)
    ret = (errno == ENOENT) ? 6 : 1;
  else
    {
      /* Delete the group.  */
      file = vsd_file_delete_line (file, line);
      /* Save.  */
      vsd_group_save (file, server_root);
      ret = 0;
    }

  vsd_group_free (file);
  return ret;
}

/* Modify a group `group_name'.
   If `gid' is not zero then change the group id.
   If `new_group' is not NULL then change the group name.
   Return 0 on success.
   Return 2 if the parameters are invalid.
   Return 4 if the gid is already in use.
   Return 6 if the specified `group_name' does not exist.
   Return 9 if the specified `new_group' is already in use.
   Return 10 on file error.  */
int vsd_groupmod (const char *server_root,
		  const char *group_name, gid_t gid, const char *new_group)
{
  struct vsd_file_line *file, *line, *line1;
  dstring_t ds;
  char *p;

  if (vsd_group_load (&file, server_root))
    return 10;

  if (gid)
    {
      gid_t min, max;

      /* Is `gid' valid within the virtual server gid range ?  */
      vsd_get_uidgid (server_root, NULL, NULL, &min, &max);
      if (gid < min || gid > max)
	/* No.  */
	return 2;

      /* Is `gid' unique ?  */
      if (unique_gid (file, gid, min, max) != gid)
	{
	  /* No.  */
	  vsd_group_free (file);
	  return 4; /* No.  */
	}
    }

  /* Check `group_name' exists.  */
  line = vsd_file_find_entry (file, group_name);
  if (line == NULL)
    {
      vsd_group_free (file);
      return (errno == ENOENT) ? 6 : 2;
    }

  if (new_group)
    {
      /* Check `new_group' is unique.  */
      line1 = vsd_file_find_entry (file, new_group);
      if (line1 != NULL)
	{
	  vsd_group_free (file);
	  return 9;
	}
      else if (line1 == NULL && errno == EINVAL)
	{
	  vsd_group_free (file);
	  return 2;
	}
    }

  /* Create the new group entry.  */
  ds = dstring_new (64);
  dstring_append (ds, (new_group) ? new_group : group_name);
  dstring_append (ds, ":");

  /* Skip past the group name.  Append the group password.  */
  p = line->line;
  while (*p != ':')
    p++;
  p++; /* Skip past colon.  */
  if (*p != ':') /* Is password field empty ?  */
    {
      /* Append password.  */
      char buf[4];
      buf[1] = '\0';
      while (*p != ':')
	{
	  buf[0] = *p++;
	  dstring_append (ds, buf);
	}
    }

  dstring_append (ds, ":");
  p++; /* Skip past colon.  */
  if (gid)
    {
      dstring_fmt_append (ds, "%d", gid);
      /* Skip gid.  */
      while (*p != ':')
	p++;
    }
  else
    {
      /* Append old gid.  */
      char buf[4];
  
      buf[1] = '\0';
      while (*p != ':')
	{
	  buf[0] = *p++;
	  dstring_append (ds, buf);
	}
    }
  /* Append rest of group line, which should be a colon and a comma seperated
     list of users.  */
  dstring_append (ds, p);

  /* Add the entry to the group file.  */
  vsd_file_insert_line (line, dstring_ptr (ds));
  /* Delete the old line.  */
  vsd_file_delete_line (file, line);

  /* Save and exit with success.  */
  vsd_group_save (file, server_root);
  dstring_free (ds);
  vsd_group_free (file);
  return 0;
}

struct vsd_file_line *vsd_groupgetnam (struct vsd_file_line *file,
				       const char *group_name)
{
  /* Match the line and the group.  */
  return vsd_file_find_entry (file, group_name);
}

/* Decode a group line into a group structure.  On entry, `line' is
   a pointer to the line that is to be decoded.  */
struct group *vsd_group_decode (struct vsd_file_line *line,
			       struct group *gr)
{
  char *p = line->line, *q;
  char buf[12];
  int num_users;

  /* Get group name.  */
  gr->gr_name = (char *) malloc (string_lenc (p, ':') + 1);
  string_cpyc (gr->gr_name, p, ':');
  p += string_lenc (p, ':') + 1;

  /* Get group password.  */
  gr->gr_passwd = (char *) malloc (string_lenc (p, ':') + 1);
  string_cpyc (gr->gr_passwd, p, ':');
  p += string_lenc (p, ':') + 1;

  /* Get group id.  */
  string_cpyc (buf, p, ':');
  gr->gr_gid = (gid_t) atoi (buf);
  p += string_lenc (p, ':') + 1;

  /* Get users that are a member of this group.  */

  /* Count number of users.  */
  q = p;
  num_users = (*q) ? 1 : 0;
  while (*q)
    {
      if (*q++ == ',')
	num_users++;
    }

  gr->gr_mem = (char **) malloc (sizeof (char *) * num_users);
  gr->gr_mem[0] = (char *) malloc ((q - p + 1) * sizeof (char));
  strcpy (gr->gr_mem[0], p);
  p = gr->gr_mem[0];
  num_users = 1;
  while (*p)
    if (*p++ == ',')
      {
	p[-1] = '\0';
	gr->gr_mem[num_users++] = p;
      }

  return gr;
}

/* Free memory alloocated in a group structure previously created by
   vsd_group_decode.  */
void vsd_group_struct_free (struct group *gr)
{
  if (gr)
    {
      free (gr->gr_name);
      free (gr->gr_passwd);
      free (gr->gr_mem[0]);
      free (gr->gr_mem);
    }
}

/* Make `user_name' a member of the group pointed to by `line'.  */
void vsd_group_add_user (struct vsd_file_line *line, const char *user_name)
{
  string_member_add (&line->line, strrchr (line->line, ':') + 1,
		     user_name, ',');
}

/* Revoke membership group pointed to by `line' for `user_name'.  */
void vsd_group_del_user (struct vsd_file_line *line, const char *user_name)
{
  char *p = strrchr (line->line, ':') + 1;

  if (*p != '\0')
    string_member_delete (p, user_name, ',');
}

/* Delete user from all groups.  */
void vsd_group_remove_user (struct vsd_file_line *head, const char *user_name)
{
  while (head)
    {
      vsd_group_del_user (head, user_name);
      head = head->next;
    }
}

/* Rename a user in all groups.  */
void vsd_group_rename_user (struct vsd_file_line *head,
			    const char *old_user, const char *new_user)
{
  while (head)
    {
      char *p = strrchr (head->line, ':') + 1;

      if (*p != '\0' && ! string_member_delete (p, old_user, ','))
	string_member_add (&head->line, p, new_user, ',');
      head = head->next;
    }
}
