#!/usr/bin/perl
# Script for creating virtual servers from a skeleton template.
# RedHat GNU Linux version.
# Copyright (c) 1999, 2000 Idaya Ltd.
#
# This file is part of the Virtual Server Administrator (FreeVSD)
#
# FreeVSD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# FreeVSD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with FreeVSD; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#

# To run this script, you must have previously created a skel with
# vsd-genskel.pl.
#
# This script could be called as a cron job, periodically process
# /tmp/vsd.accounts creating and deleting accounts whenever necessary.
#
use Sys::Hostname;
use IO::File;
use File::Copy;
use lib qw(@vsdperllib@);
use VSD;

# /tmp/vsd.accounts is a temporary file that contains the queue of
# virtual servers that need creating.
unless (-e "/tmp/vsd.accounts") { exit 0; }

# If this lock file exists then account creation is already in
# progress, don't start another job.
if (-e "/tmp/vsd.accounts.lock") { exit 0; }
open (OUT, ">/tmp/vsd.accounts.lock"); close (OUT);
rename ("/tmp/vsd.accounts", "/tmp/vsd.accounts.bak");

# Location of sendmail
$sendmail = "/bin/sendmail" if (-x "/bin/sendmail");
$sendmail = "/usr/libexec/sendmail" if (-x "/usr/libexec/sendmail");
$sendmail = "/usr/sbin/sendmail" if (-x "/usr/sbin/sendmail");
$hostname = hostname ();
srand (time ^ $$);

do "@freevsdconf@";
require ($site_vs_install) if (defined $site_vs_install);

# Array of characters suitable for an account password.
@chars = ("A".."Z", "a".."z", 0..9);

open (JOB, "/tmp/vsd.accounts.bak");
while (<JOB>) {
  chomp;
  my ($action, $vs, $param) = split (/ /, $_);
  if ($action eq "create") {
    $fqdn = $param;
    ($vsconf, $aliasip) = vsd_map_read ("@vsdconf@", $vs);

    if (! -d $vsconf->{"Partition"}) {
      printf ("Couldn't process %s: mount point `%s' does not exist\n", $name, $vsconf->{"Partition"});
    } else {
      # Create a random password for the account
      $password = join ("", @chars[map {rand @chars} (1..8)]);

      $mntpoint = $vsconf->{"Partition"};
      $skelroot = $mntpoint."/skel"; # Used in all sub-procs
      $vsdir = $mntpoint."/vs/".$vs; # Used in all sub-procs

      make_vs ($vsconf);
      site_make_vs ($vsconf) if (defined $site_vs_install);
      config_vs ($vsconf, $fqdn);
      site_config_vs ($vsconf, $fqdn) if (defined $site_vs_install);

      if ($send_mail eq "YES") {
	email ($vsconf, $fqdn, $password, $mail_template);
      }
    }
  }
  if ($action eq "delete") {
    if (! -d "$param") {
      printf ("Couldn't delete %s: mount point '%s' does not exist\n", $vs, $param);
    } else {
      `rm -r -f $param/vs/$vs`;
    }
  }
}
close (JOB);
unlink ("/tmp/vsd.accounts.bak");
unlink ("/tmp/vsd.accounts.lock");
exit 0;

# email <vsconf> <fqdn> <admin-password> <template>
sub email {
  my ($fqdn, $ipaddr, $vs, $pw, $email, $tmpl);

  $vsconf = $_[0];
  $fqdn = $_[1];
  $pw = $_[2];
  $email = $mail_addr;
  $tmpl = $_[3];
  
  open (TMPL, "< $tmpl")
    or warn "cannot open $tmpl: $!\n";
  open (SMAIL, "|$sendmail -oi -t")
    or die "cannot open a pipe to $sendmail: $!";
  printf (SMAIL "From: vsd@%s (Virtual Server Administrator)\n", $hostname);
  printf (SMAIL "To: %s\n", $email);
  while (<TMPL>) {
    s/\#VS\#/$vsconf->{"Name"}/;
    s/\#HOSTNAME\#/$fqdn/;
    s/\#IPADDRESS\#/$vsconf->{"IP"}/;
    s/\#PASSWORD\#/$pw/;
    printf SMAIL "%s", $_;
  }
  close (SMAIL);
  close (TMPL);
}

# make_vs <vsconf>
sub make_vs {
  my (@nam1, @nam2, $x, $y, $vs);

  $vsconf = $_[0];

  mkdir ("$mntpoint/vs", 0755) unless (-d "$mntpoint/vs");
  mkdir ("$vsdir", 0755) unless (-d "$vsdir");

  # Files are copied as hard links except /etc
  system ("cp -al $skelroot/* $vsdir");
  system ("rm -r -f $vsdir/etc $vsdir/home/httpd");
  system ("cp -a $skelroot/etc $vsdir");
  system ("cp -a $skelroot/home/httpd $vsdir/home");

  # Create some additional directories
  @dirs = ("etc/vsd", "etc/cron.d", "dev", "root", "var", "var/run",
	   "home/ftp", "home/ftp/incoming");
  foreach $file (@dirs) {
      mkdir ("$vsdir/$file", 0755);
  }
  mkdir ("$vsdir/etc/vsd", 0755);

  # Softlink /var/tmp to /tmp
  symlink ("../tmp", "$vsdir/var/tmp");
  
  # Set the timezone. Er. Shouldn't be blighty specific.
  symlink ("../usr/share/zoneinfo/Europe/London", "$vsdir/etc/localtime");

  # Create the tty devices
  @nam1 = ("p".."z", "a".."e");
  @nam2 = (0..9, "a".."f");
  for ($x = 0; $x <= 15; $x++) {
      for ($y = 0; $y <= 15; $y++) {
	  $minor = ($x * 16) + $y;
	  system ("mknod", "-m", "0620", "$vsdir/dev/pty$nam1[$x]$nam2[$y]",
		  "c", "2", "$minor");
	  system ("mknod", "-m", "0620", "$vsdir/dev/tty$nam1[$x]$nam2[$y]",
		  "c", "3", "$minor"); 
      }
  }
  
  # Make other useful devices
  system ("mknod", "-m", "0666", "$vsdir/dev/zero", "c", "1", "5");
  unlink ("$vsdir/dev/null");
  system ("mknod", "-m", "0666", "$vsdir/dev/null", "c", "1", "3");
  system ("mknod", "-m", "0666", "$vsdir/dev/random", "c", "1", "8");
  system ("mknod", "-m", "0666", "$vsdir/dev/urandom", "c", "1", "9");
  system ("mknod", "-m", "0666", "$vsdir/dev/tty", "c", "5", "0");
  mkdir ("$vsdir/dev/fd", 0755);
  symlink ("../var/run/log", "$vsdir/dev/log"); # For syslogd
}

# config_vs <vsconf> <fqdn>
sub config_vs {
  my ($vs, $fqdn, $startuid, $enduid, $c, $fh, $childpid);

  $vsconf = $_[0];
  $fqdn = $_[1];

  $startuid = $vsconf->{"StartUID"};
  $enduid = $vsconf->{"StartUID"} + $vsconf->{"Users"} - 1;

  # This file is used by the uname system call
  open (OUT, ">$vsdir/etc/FQDN");
  printf OUT "%s %s\n", $fqdn, $vsconf->{"Name"};
  close (OUT);

  # Create /etc/hosts
  open (OUT, ">$vsdir/etc/hosts");
  printf OUT "%s %s localhost\n", $vsconf->{"IP"}, $fqdn;
  close (OUT);

  # Create /etc/resolv.conf
  open (OUT, ">$vsdir/etc/resolv.conf");
  printf OUT "domain %s\n", $fqdn;
  printf OUT "nameserver %s\n", $vsconf->{"PrimaryNS"};
  printf OUT "nameserver %s\n", $vsconf->{"SecondaryNS"};
  close (OUT);

  # Create /etc/issue.net
  open (OUT, ">$vsdir/etc/issue.net");
  printf OUT "Server %s\n", $fqdn;
  close (OUT);

  # Create /etc/sendmail.cw
  open (OUT, ">$vsdir/etc/sendmail.cw");
  printf OUT "# sendmail.cw - include all aliases for your machine here\n";
  printf OUT "%s\n", $fqdn;
  close (OUT);

  rename ("$vsdir/etc/login.defs", "$vsdir/tmp/login.defs");
  open (IN, "$vsdir/tmp/login.defs");
  open (OUT, ">$vsdir/etc/login.defs");
  while (<IN>) {
    $_ =~ s/\#UID_MIN\#/$startuid/;
    $_ =~ s/\#UID_MAX\#/$enduid/;
    $_ =~ s/\#GID_MIN\#/$startuid/;
    $_ =~ s/\#GID_MAX\#/$enduid/;
    printf OUT "%s", $_;
  }
  close (OUT);
  close (IN);
  unlink ("$vsdir/tmp/login.defs");

  if (! defined($childpid = fork ())) {
      die "cannot fork: $!";
  } elsif ($childpid == 0) {
    # child
    my ($gid_admin, $gid_mail, $gid_web, $gid_ftp);
    my ($uid_admin, $uid_web, $uid_ftp);
    chroot ($vsdir);
    chdir ("/");
    $uid_admin = $gid_admin = $startuid;
    $uid_mail = $gid_mail = $startuid + 1;
    $uid_web = $gid_web = $startuid + 2;
    $uid_ftp = $gid_ftp = $startuid + 3;
    
    system ("/usr/sbin/groupadd", "admin", "-g", "$gid_admin");
    system ("/usr/sbin/groupadd", "mail", "-g", "$gid_mail");
    system ("/usr/sbin/groupadd", "web", "-g", "$gid_web");
    system ("/usr/sbin/groupadd", "ftp", "-g", "$gid_ftp");

    system ("/usr/sbin/useradd", "-c", "administrator", "-d", "/root",
	    "-u", "$uid_admin", "-g", "admin", "-G", "web", "-M",
	    "-n", "-s", "/bin/bash", "admin");
    chown $uid_admin, $gid_admin, "/home";

    system ("/usr/sbin/useradd", "-c", "mail", "-d", "/var/spool/mail",
	    "-u", "$uid_mail", "-g", "mail", "-M", "-n",
	    "-s", "/bin/false", "mail");

    system ("/usr/sbin/useradd", "-c", "webserver", "-d", "/home/web",
	    "-u", "$uid_web", "-g", "web", "-G", "admin", "-M", "-n",
	    "-s", "/bin/bash", "web");
    mkdir ("/home/web", 0755);
    chown $uid_web, $uid_web, "/home/web";
    mkdir ("/home/web/log", 0755);
    chown $uid_web, $uid_web, "/home/web/log";

    system ("/usr/sbin/useradd", "-c", "ftpserver", "-d", "/home/ftp",
	    "-u", "$uid_ftp", "-g", "ftp", "-M", "-n",
	    "-s", "/bin/bash", "ftp");

    @admin_files = ("/home/httpd", "/home/ftp", "/tmp", "/root");
    chown $uid_admin, $gid_admin, @admin_files;
    system ("chown", "-R", "root.root", "/etc");

    # Create subdirectories of /var
    @subdir = ("catman", "lib", "lib/texmf", "lib/texmf/fonts",
	       "lib/texmf/texfonts", "local", "lock", "lock/subsys",
	       "log", "preserve", "run", "spool", "spool/at",
	       "spool/cron", "spool/mail", "spool/mqueue");
    foreach (@subdir) {
	mkdir ("/var/$_", 0755);
    }

    # Create subdirectories pf /usr/local
    mkdir ("/usr/local", 0755);
    @subdir = ("bin", "doc", "etc", "games", "include", "info", "lib",
	       "man", "sbin", "src", "man/man1", "man/man2", "man/man3",
	       "man/man4", "man/man5", "man/man6", "man/man7", "man/man8",
	       "man/mann");
    foreach (@subdir) {
	mkdir ("/usr/local/$_", 0755);
	chown $uid_admin, $gid_admin, "/usr/local/$_";
    }

    chmod (01777, "/tmp");
    chmod (0755, "/etc");
    chmod (0755, "/etc/mail");
    chmod (0755, "/etc/vsd");
    chmod (0775, "/var/lock");
    chmod (0755, "/var/spool/at");
    chmod (0775, "/var/catman");
    chmod (0775, "/var/spool/mail");
    chmod (0755, "/var/log");

    # Certain directories cannot be written to by admin
    #`chown -R root.man /var/catman`;
    #`chown -R root.uucp /var/lock`;
    #`chown -R daemon.admin /var/spool/at`;

    # Certain directories need to be written to by admin
    `chown admin.admin /var/run`;
    `chown admin.admin /var/lib`;
    `chown admin.admin /var/log`;

    # Create the dynamic library cache.
    system ("/sbin/ldconfig");
    
    # Fix file permissions of certain /etc files.
    foreach $file ("login.defs", "vsd/priv", "sendmail.cw", "sendmail.ct",
		   "virtusertable", "crontab", "sendmail.cf",
		   "httpd/conf/httpd.conf", "pam.d/*", "aliases") {
	chmod (0644, "/etc/$file");
    }

    # Fixup file permissions to stop sendmail complaining.
    chmod (0666, "/etc/mail/ip_allow");

    # Generate /etc/aliases.db.
    `/usr/bin/newaliases 2>/dev/null`;
    
    # Certain files in /etc should be owned by the admin user
    @admin_files = ("/etc/aliases", "/etc/at.deny", "/etc/bashrc",
		    "/etc/hosts",
		    "/etc/issue.net", "/etc/logrotate.conf",
		    "/etc/lynx.cfg", "/etc/ld.so.conf",
		    "/etc/ld.so.cache", "/etc/resolv.conf",
		    "/etc/proftpd.conf", "/etc/sendmail.cw",
		    "/etc/virtusertable", "/etc/virtusertable.db",
		    "/etc/rc.conf", "/etc/rc.local");
    push (@admin_files, glob ("/etc/httpd"));
    push (@admin_files, glob ("/etc/mail"));
    push (@admin_files, glob ("/etc/logrotate.d"));
    chown $uid_admin, $gid_admin, @admin_files;

    # Set the password here for the admin user.
    system ("/usr/bin/passwd", "-u", "admin", "-p", $password);

    exit 0;
  } else {
    waitpid ($childpid, 0);
  }

  if ($vsconf->{"Quota"} != 0) {
    # Set quotas for the users. Purely arbitary.
    $vs = $vsconf->{"Name"};
    system ("@vsdadm@", "quota_userset", "localhost", "$vs", "admin", "10M");
    system ("@vsdadm@", "quota_userset", "localhost", "$vs", "web", "4M");
    system ("@vsdadm@", "quota_userset", "localhost", "$vs", "mail", "4K");
    system ("@vsdadm@", "quota_userset", "localhost", "$vs", "ftp", "4K");
  }
}
