/*
 * TCFS protocol definitions
 */
#ifndef _LINUX_TCFS_H
#define _LINUX_TCFS_H

#include <linux/sunrpc/msg_prot.h>
#include <linux/types.h>
#include <linux/posix_types.h>

/* Messages */
#define TCFS_DBG_NONE		0
#define TCFS_DBG_FUNC_IN	1
#define TCFS_DBG_FUNC_OUT	2
#define TCFS_DBG_GEN		4
#define TCFS_DBG_ERR		8
#define TCFS_DBG_ALL		255

/* Errors */
#define TCFS_ERROK	0
#define TCFS_ERRGEN	1
#define TCFS_ERRNOMEM	ENOMEM
#define TCFS_ERRACCES	EACCES
#define TCFS_ERRINVAL	EINVAL;

#define KEYSIZE		16
#define TCFS_SECURE_FL	0x00000400
#define TCFS_SHARED_FL	0x00000800
#define TCFS_CLEAR_FILE	0x00000000
#define TCFS_IS_SECURE(inode)	\
	((inode)->u.tcfs_i.tcfsheader.flags & TCFS_SECURE_FL)
#define TCFS_IS_SHARED(inode)	\
	((inode)->u.tcfs_i.tcfsheader.flags & TCFS_SHARED_FL)
#define TCFS_NO_SECURE	(void *)-1
#define TCFS_HASH_SIZE		16
#define TCFS_HBLOCK_SIZE	1024
#define TCFS_BLOCK_SIZE		(TCFS_HBLOCK_SIZE-TCFS_HASH_SIZE)
#define TCFS_HEADER_SIZE	(1024-TCFS_HASH_SIZE)
#define TCFS_ENGINE_FIELD_SIZE	8
#define TCFS_FIELD_SIZE		0x0
#define TCFS_FIELD_ENGINE	0x1
#define TCFS_FIELD_FLAGS	0x2
#define NUM_CIPHER		4
#define CIPHER_NONE		0
#define CIPHER_DES		1
#define CIPHER_IDEA		2
#define CIPHER_RC5		3

extern void tcfsencode (char *, int);
extern int  tcfsdecode (char *, int);

struct tcfs_header {
	char start[4];
	unsigned int flags;
	char engine[TCFS_ENGINE_FIELD_SIZE];
	off_t size;
	unsigned char file_key[KEYSIZE];
	char pad[TCFS_HEADER_SIZE-4-TCFS_ENGINE_FIELD_SIZE-sizeof(unsigned int)-sizeof(off_t)-KEYSIZE];
};

struct new_tcfs_header {
	char start[4];
	u_char version;
	u_char flags;
	u_short header_size;
	char engine[TCFS_ENGINE_FIELD_SIZE];
	u_short key_len;
};
	
struct tcfs_inode_header {
	unsigned int flags;
	char engine[TCFS_ENGINE_FIELD_SIZE];
	unsigned char file_key[KEYSIZE];
	off_t size;
};

struct login {
	__kernel_uid_t uid;
	unsigned char deskey[KEYSIZE];
	char ciphername[TCFS_ENGINE_FIELD_SIZE];
};

struct tgroup {
	int gid;
	__kernel_uid_t uid;
	int n;
	int soglia;
	unsigned char deskey[KEYSIZE+KEYSIZE/8];
};

#define TCFS_PORT	2049
#define TCFS_MAXDATA	8192
#define TCFS_MAXPATHLEN	1024
#define TCFS_MAXNAMLEN	255
#define TCFS_MAXGROUPS	16
#define TCFS_FHSIZE	32
#define TCFS_COOKIESIZE	4
#define TCFS_FIFO_DEV	(-1)
#define TCFSMODE_FMT	0170000
#define TCFSMODE_DIR	0040000
#define TCFSMODE_CHR	0020000
#define TCFSMODE_BLK	0060000
#define TCFSMODE_REG	0100000
#define TCFSMODE_LNK	0120000
#define TCFSMODE_SOCK	0140000
#define TCFSMODE_FIFO	0010000

	
enum tcfs_stat {
	TCFS_OK = 0,
	TCFSERR_PERM = 1,
	TCFSERR_NOENT = 2,
	TCFSERR_IO = 5,
	TCFSERR_NXIO = 6,
	TCFSERR_EAGAIN = 11,
	TCFSERR_ACCES = 13,
	TCFSERR_EXIST = 17,
	TCFSERR_XDEV = 18,
	TCFSERR_NODEV = 19,
	TCFSERR_NOTDIR = 20,
	TCFSERR_ISDIR = 21,
	TCFSERR_INVAL = 22,	/* that Sun forgot */
	TCFSERR_FBIG = 27,
	TCFSERR_NOSPC = 28,
	TCFSERR_ROFS = 30,
	TCFSERR_OPNOTSUPP = 45,
	TCFSERR_NAMETOOLONG = 63,
	TCFSERR_NOTEMPTY = 66,
	TCFSERR_DQUOT = 69,
	TCFSERR_STALE = 70,
	TCFSERR_WFLUSH = 99
};

enum tcfs_ftype {
	TCFNON = 0,
	TCFNFREG = 1,
	TCFDIR = 2,
	TCFBLK = 3,
	TCFCHR = 4,
	TCFLNK = 5,
	TCFSOCK = 6,
	TCFBAD = 7,
	TCFFIFO = 8
};

struct tcfs_fh {
	char			data[TCFS_FHSIZE];
};

#define TCFS_PROGRAM		100003
#define TCFS_VERSION		2
#define TCFSPROC_NULL		0
#define TCFSPROC_GETATTR		1
#define TCFSPROC_SETATTR		2
#define TCFSPROC_ROOT		3
#define TCFSPROC_LOOKUP		4
#define TCFSPROC_READLINK	5
#define TCFSPROC_READ		6
#define TCFSPROC_WRITECACHE	7
#define TCFSPROC_WRITE		8
#define TCFSPROC_CREATE		9
#define TCFSPROC_REMOVE		10
#define TCFSPROC_RENAME		11
#define TCFSPROC_LINK		12
#define TCFSPROC_SYMLINK		13
#define TCFSPROC_MKDIR		14
#define TCFSPROC_RMDIR		15
#define TCFSPROC_READDIR		16
#define TCFSPROC_STATFS		17

/* Mount support for TCFSroot */
#ifdef __KERNEL__
#define TCFS_MNT_PROGRAM		100005
#define TCFS_MNT_VERSION		1
#define TCFS_MNT_PORT		627
#define TCFS_MNTPROC_MNT		1
#define TCFS_MNTPROC_UMNT	3
#endif

#if defined(__KERNEL__) || defined(TCFS_NEED_KERNEL_TYPES)

extern struct rpc_program	tcfs_program;
extern struct rpc_stat		tcfs_rpcstat;

struct tcfs_time {
	__u32			seconds;
	__u32			useconds;
};

struct tcfs_fattr {
	enum tcfs_ftype		type;
	__u32			mode;
	__u32			nlink;
	__u32			uid;
	__u32			gid;
	__u32			size;
	__u32			blocksize;
	__u32			rdev;
	__u32			blocks;
	__u32			fsid;
	__u32			fileid;
	struct tcfs_time		atime;
	struct tcfs_time		mtime;
	struct tcfs_time		ctime;
};

struct tcfs_sattr {
	__u32			mode;
	__u32			uid;
	__u32			gid;
	__u32			size;
	struct tcfs_time		atime;
	struct tcfs_time		mtime;
};

struct tcfs_fsinfo {
	__u32			tsize;
	__u32			bsize;
	__u32			blocks;
	__u32			bfree;
	__u32			bavail;
};

struct tcfs_writeargs {
	struct tcfs_fh *		fh;
	__u32			offset;
	__u32			count;
	const void *		buffer;
};

#ifdef TCFS_NEED_XDR_TYPES

struct tcfs_sattrargs {
	struct tcfs_fh *		fh;
	struct tcfs_sattr *	sattr;
};

struct tcfs_diropargs {
	struct tcfs_fh *		fh;
	const char *		name;
};

struct tcfs_readargs {
	struct tcfs_fh *		fh;
	__u32			offset;
	__u32			count;
	void *			buffer;
};

struct tcfs_createargs {
	struct tcfs_fh *		fh;
	const char *		name;
	struct tcfs_sattr *	sattr;
};

struct tcfs_renameargs {
	struct tcfs_fh *		fromfh;
	const char *		fromname;
	struct tcfs_fh *		tofh;
	const char *		toname;
};

struct tcfs_linkargs {
	struct tcfs_fh *		fromfh;
	struct tcfs_fh *		tofh;
	const char *		toname;
};

struct tcfs_symlinkargs {
	struct tcfs_fh *		fromfh;
	const char *		fromname;
	const char *		topath;
	struct tcfs_sattr *	sattr;
};

struct tcfs_readdirargs {
	struct tcfs_fh *		fh;
	__u32			cookie;
	void *			buffer;
	unsigned int		bufsiz;
};

struct tcfs_diropok {
	struct tcfs_fh *		fh;
	struct tcfs_fattr *	fattr;
};

struct tcfs_readres {
	struct tcfs_fattr *	fattr;
	unsigned int		count;
};

struct tcfs_readlinkres {
	char **			string;
	unsigned int *		lenp;
	unsigned int		maxlen;
	void *			buffer;
};

struct tcfs_readdirres {
	void *			buffer;
	unsigned int		bufsiz;
};

#endif /* TCFS_NEED_XDR_TYPES */
#endif /* __KERNEL__ */

#endif
