#define __NO_VERSION__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/kmod.h>
#include <asm/uaccess.h>

#include <linux/tcfs.h>

extern unsigned int tcfsusers;
extern u_int tcfs_numread;
extern u_int tcfs_numwrite;
extern u_int tcfs_numreadget;
extern u_int tcfs_numofgetheader;
extern u_int tcfs_numreadflags;
extern u_int tcfs_numwritepage, tcfs_numreadpage;
extern u_int tcfs_numputwrite, tcfs_numputflags;
extern u_int tcfs_numofputheader;

static int modcount;
static char tcfs_input[1024];

static ssize_t tcfs_module_output (struct file *file, char *buf, size_t len,
		loff_t *offset);
static ssize_t tcfs_module_input (struct file *file, const char *buf,
		size_t len, loff_t *offset);

static int tcfs_module_perm (struct inode *inode, int op);
static int tcfs_module_open (struct inode *inode, struct file *file);
static int tcfs_module_close (struct inode *inode, struct file *file);

static struct file_operations tcfs_file_ops={
	NULL,				/* seek */
	tcfs_module_output,		/* read */
	tcfs_module_input,		/* write */
	NULL,				/* readdir */
	NULL,				/* poll */
	NULL,				/* ioctl */
	NULL,				/* mmap */
	tcfs_module_open,		/* open */
	NULL,				/* flush */
	tcfs_module_close,		/* release */
};

static struct inode_operations tcfs_inode_ops={
	&tcfs_file_ops,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tcfs_module_perm
};

struct proc_dir_entry tcfs_proc_file={
	0, /* Inode number, filled by the kernel */
	4, /* Lenght of the file name */
	"tcfs",
	S_IFREG|S_IRUGO|S_IWUGO, /* File mode */
	1, /* Num of links */
	0, /* uid of the file */
	0, /* gid of the file */
	0, /* File size */
	&tcfs_inode_ops,
	NULL
};

static ssize_t tcfs_module_output (struct file *file, char *buf,
				size_t len, loff_t *offset)
{
	static int done=0;
	char outmesg[2048];
	int i;

	if (done) {
		done=0;
		return 0;
	}

	sprintf (outmesg, "TCFS 3.0 statistics:\n\n");
	sprintf (outmesg+strlen(outmesg), "Modules loaded (via /proc): %d (last: %s)\n", modcount, tcfs_input);
	sprintf (outmesg+strlen(outmesg), "TCFS users key in memory: %d\n",
			tcfsusers);
	sprintf (outmesg+strlen(outmesg), "Num of read: %d\n", tcfs_numread);
	sprintf (outmesg+strlen(outmesg), "Num of write: %d\n", tcfs_numwrite);

	sprintf (outmesg+strlen(outmesg), "Num of readpage: %d\n",
		tcfs_numreadpage);
	sprintf (outmesg+strlen(outmesg), "Num of writepage: %d\n",
		tcfs_numwritepage);

	sprintf (outmesg+strlen(outmesg), "Num of get_header call: %d\n",
		tcfs_numofgetheader);
	sprintf (outmesg+strlen(outmesg), "Num of put_header call: %d\n",
		tcfs_numofputheader);

	sprintf (outmesg+strlen(outmesg), "Num of read in get_header: %d\n",
		tcfs_numreadget);
	sprintf (outmesg+strlen(outmesg), "Num of flags read in get_header: %d\n",
		tcfs_numreadflags);

	sprintf (outmesg+strlen(outmesg), "Num of write in put_header: %d\n",
		tcfs_numputwrite);
	sprintf (outmesg+strlen(outmesg), "Num of flags wrote in put_header: %d\n",
		tcfs_numputflags);

	for (i=0;i<len && outmesg[i]; i++) {
		put_user (outmesg[i], buf+i);
	}

	done=1;

	return i;
}

static ssize_t tcfs_module_input (struct file *file, const char *buf,
				size_t len, loff_t *offset)
{
	int i;

	for (i=0;i<1024 && i<len; i++)
		get_user(tcfs_input[i], buf+i);

	if (tcfs_input[i-1]=='\n')
		tcfs_input[i-1]='\0';

	tcfs_input[i]='\0';

	if (!strcmp (tcfs_input, "reset")) {
		tcfs_numread=0;
		tcfs_numwrite=0;
		tcfs_numreadget=0;
		tcfs_numofgetheader=0;
		tcfs_numreadflags=0;
		tcfs_numwritepage=0;
		tcfs_numreadpage=0;
		tcfs_numputwrite=0;
		tcfs_numputflags=0;
		tcfs_numofputheader=0;

		return i;
	}

	if (request_module (tcfs_input)==0) {
		modcount++;
	}

	return i;
}

static int tcfs_module_perm (struct inode *inode, int op)
{
	if (op==4||(op==2 && current->euid==0))
		return 0;

	return -EACCES;
}

int tcfs_module_open (struct inode *inode, struct file *file)
{
	MOD_INC_USE_COUNT;

	return 0;
}

int tcfs_module_close (struct inode *inode, struct file *file)
{
	MOD_DEC_USE_COUNT;

	return 0;
}

int init_procfs (void)
{
	modcount=0;
	return proc_register (&proc_root, &tcfs_proc_file);
}

int cleanup_procfs (void)
{
	return proc_unregister (&proc_root, tcfs_proc_file.low_ino);
}
