/*

  client_channel.c

  Author: Pekka Riikonen <priikone@poseidon.pspt.fi>

  Copyright (C) 1997 - 2001 Pekka Riikonen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

*/
/* $Id: client_channel.c,v 1.10 2001/04/12 22:00:21 priikone Exp $ */
/* This file includes channel message sending and receiving routines,
   channel key receiving and setting, and channel private key handling 
   routines. */

#include "clientlibincludes.h"
#include "client_internal.h"

/* Sends packet to the `channel'. Packet to channel is always encrypted
   differently from "normal" packets. SILC header of the packet is 
   encrypted with the next receiver's key and the rest of the packet is
   encrypted with the channel specific key. Padding and HMAC is computed
   with the next receiver's key. The `data' is the channel message. If
   the `force_send' is TRUE then the packet is sent immediately. */

void silc_client_send_channel_message(SilcClient client, 
				      SilcClientConnection conn,
				      SilcChannelEntry channel,
				      SilcChannelPrivateKey key,
				      SilcMessageFlags flags,
				      unsigned char *data, 
				      uint32 data_len, 
				      int force_send)
{
  int i;
  SilcSocketConnection sock = conn->sock;
  SilcBuffer payload;
  SilcPacketContext packetdata;
  SilcCipher cipher;
  SilcHmac hmac;
  unsigned char *id_string;
  uint32 iv_len;

  SILC_LOG_DEBUG(("Sending packet to channel"));

  /* Take the key to be used */
  if (channel->mode & SILC_CHANNEL_MODE_PRIVKEY) {
    if (key) {
      /* Use key application specified */
      cipher = key->cipher;
      hmac = key->hmac;
    } else if (channel->curr_key) {
      /* Use current private key */
      cipher = channel->curr_key->cipher;
      hmac = channel->curr_key->hmac;
    } else if (!channel->curr_key && channel->private_keys) {
      /* Use just some private key since we don't know what to use 
	 and private keys are set. */
      silc_dlist_start(channel->private_keys);
      key = silc_dlist_get(channel->private_keys);
      cipher = key->cipher;
      hmac = key->hmac;
      
      /* Use this key as current private key */
      channel->curr_key = key;
    } else {
      /* Use normal channel key generated by the server */
      cipher = channel->channel_key;
      hmac = channel->hmac;
    }
  } else {
    /* Use normal channel key generated by the server */
    cipher = channel->channel_key;
    hmac = channel->hmac;
  }

  if (!cipher || !hmac)
    return;

  /* Generate IV */
  iv_len = silc_cipher_get_block_len(cipher);
  if (channel->iv[0] == '\0')
    for (i = 0; i < iv_len; i++) channel->iv[i] = 
				   silc_rng_get_byte(client->rng);
  else
    silc_hash_make(client->md5hash, channel->iv, iv_len, channel->iv);

  /* Encode the channel payload. This also encrypts the message payload. */
  payload = silc_channel_message_payload_encode(flags, data_len, data, iv_len, 
						channel->iv, cipher, hmac);

  /* Get data used in packet header encryption, keys and stuff. */
  cipher = conn->send_key;
  hmac = conn->hmac_send;
  id_string = silc_id_id2str(channel->id, SILC_ID_CHANNEL);

  /* Set the packet context pointers. The destination ID is always
     the Channel ID of the channel. Server and router will handle the
     distribution of the packet. */
  packetdata.flags = 0;
  packetdata.type = SILC_PACKET_CHANNEL_MESSAGE;
  packetdata.src_id = conn->local_id_data;
  packetdata.src_id_len = SILC_ID_CLIENT_LEN;
  packetdata.src_id_type = SILC_ID_CLIENT;
  packetdata.dst_id = id_string;
  packetdata.dst_id_len = SILC_ID_CHANNEL_LEN;
  packetdata.dst_id_type = SILC_ID_CHANNEL;
  packetdata.truelen = payload->len + SILC_PACKET_HEADER_LEN + 
    packetdata.src_id_len + packetdata.dst_id_len;
  packetdata.padlen = SILC_PACKET_PADLEN((SILC_PACKET_HEADER_LEN +
					  packetdata.src_id_len +
					  packetdata.dst_id_len));

  /* Prepare outgoing data buffer for packet sending */
  silc_packet_send_prepare(sock, 
			   SILC_PACKET_HEADER_LEN +
			   packetdata.src_id_len + 
			   packetdata.dst_id_len,
			   packetdata.padlen,
			   payload->len);

  packetdata.buffer = sock->outbuf;

  /* Put the channel message payload to the outgoing data buffer */
  silc_buffer_put(sock->outbuf, payload->data, payload->len);

  /* Create the outgoing packet */
  silc_packet_assemble(&packetdata);

  /* Encrypt the header and padding of the packet. This is encrypted 
     with normal session key shared with our server. */
  silc_packet_encrypt(cipher, hmac, sock->outbuf, SILC_PACKET_HEADER_LEN + 
		      packetdata.src_id_len + packetdata.dst_id_len +
		      packetdata.padlen);

  SILC_LOG_HEXDUMP(("Packet to channel, len %d", sock->outbuf->len),
		   sock->outbuf->data, sock->outbuf->len);

  /* Now actually send the packet */
  silc_client_packet_send_real(client, sock, force_send, FALSE);
  silc_buffer_free(payload);
  silc_free(id_string);
}

/* Process received message to a channel (or from a channel, really). This
   decrypts the channel message with channel specific key and parses the
   channel payload. Finally it displays the message on the screen. */

void silc_client_channel_message(SilcClient client, 
				 SilcSocketConnection sock, 
				 SilcPacketContext *packet)
{
  SilcClientConnection conn = (SilcClientConnection)sock->user_data;
  SilcBuffer buffer = packet->buffer;
  SilcChannelMessagePayload payload = NULL;
  SilcChannelID *id = NULL;
  SilcChannelEntry channel;
  SilcChannelUser chu;
  SilcIDCacheEntry id_cache = NULL;
  SilcClientID *client_id = NULL;
  int found = FALSE;
  unsigned char *message;

  SILC_LOG_DEBUG(("Start"));

  /* Sanity checks */
  if (packet->dst_id_type != SILC_ID_CHANNEL)
    goto out;

  client_id = silc_id_str2id(packet->src_id, packet->src_id_len,
			     SILC_ID_CLIENT);
  if (!client_id)
    goto out;
  id = silc_id_str2id(packet->dst_id, packet->dst_id_len, SILC_ID_CHANNEL);
  if (!id)
    goto out;

  /* Find the channel entry from channels on this connection */
  if (!silc_idcache_find_by_id_one(conn->channel_cache, (void *)id,
				   SILC_ID_CHANNEL, &id_cache))
    goto out;

  channel = (SilcChannelEntry)id_cache->context;

  /* If there is no channel private key then just decrypt the message 
     with the channel key. If private keys are set then just go through
     all private keys and check what decrypts correctly. */
  if (!(channel->mode & SILC_CHANNEL_MODE_PRIVKEY)) {
    /* Parse the channel message payload. This also decrypts the payload */
    payload = silc_channel_message_payload_parse(buffer, channel->channel_key,
						 channel->hmac);
    if (!payload)
      goto out;
  } else if (channel->private_keys) {
    SilcChannelPrivateKey entry;

    silc_dlist_start(channel->private_keys);
    while ((entry = silc_dlist_get(channel->private_keys)) != SILC_LIST_END) {
      /* Parse the channel message payload. This also decrypts the payload */
      payload = silc_channel_message_payload_parse(buffer, entry->cipher,
						   entry->hmac);
      if (payload)
	break;
    }
    if (entry == SILC_LIST_END)
      goto out;
  } else {
    goto out;
  }

  message = silc_channel_message_get_data(payload, NULL);

  /* Find client entry */
  silc_list_start(channel->clients);
  while ((chu = silc_list_get(channel->clients)) != SILC_LIST_END) {
    if (!SILC_ID_CLIENT_COMPARE(chu->client->id, client_id)) {
      found = TRUE;
      break;
    }
  }

  /* Pass the message to application */
  client->ops->channel_message(client, conn, found ? chu->client : NULL,
			       channel, 
			       silc_channel_message_get_flags(payload),
			       message);

 out:
  if (id)
    silc_free(id);
  if (client_id)
    silc_free(client_id);
  if (payload)
    silc_channel_message_payload_free(payload);
}

/* Saves channel key from encoded `key_payload'. This is used when we
   receive Channel Key Payload and when we are processing JOIN command 
   reply. */

void silc_client_save_channel_key(SilcClientConnection conn,
				  SilcBuffer key_payload, 
				  SilcChannelEntry channel)
{
  unsigned char *id_string, *key, *cipher, hash[32];
  uint32 tmp_len;
  SilcChannelID *id;
  SilcIDCacheEntry id_cache = NULL;
  SilcChannelKeyPayload payload;

  payload = silc_channel_key_payload_parse(key_payload);
  if (!payload)
    return;

  id_string = silc_channel_key_get_id(payload, &tmp_len);
  if (!id_string) {
    silc_channel_key_payload_free(payload);
    return;
  }

  id = silc_id_str2id(id_string, tmp_len, SILC_ID_CHANNEL);
  if (!id) {
    silc_channel_key_payload_free(payload);
    return;
  }

  /* Find channel. */
  if (!channel) {
    if (!silc_idcache_find_by_id_one(conn->channel_cache, (void *)id,
				     SILC_ID_CHANNEL, &id_cache))
      goto out;
    
    /* Get channel entry */
    channel = (SilcChannelEntry)id_cache->context;
  }

  /* Save the key */
  key = silc_channel_key_get_key(payload, &tmp_len);
  cipher = silc_channel_key_get_cipher(payload, NULL);
  channel->key_len = tmp_len * 8;
  channel->key = silc_calloc(tmp_len, sizeof(*channel->key));
  memcpy(channel->key, key, tmp_len);

  if (!silc_cipher_alloc(cipher, &channel->channel_key)) {
    conn->client->ops->say(conn->client, conn,
		     "Cannot talk to channel: unsupported cipher %s", cipher);
    goto out;
  }

  /* Set the cipher key */
  silc_cipher_set_key(channel->channel_key, key, channel->key_len);

  /* Generate HMAC key from the channel key data and set it */
  if (!channel->hmac)
    silc_hmac_alloc("hmac-sha1-96", NULL, &channel->hmac);
  silc_hash_make(channel->hmac->hash, key, tmp_len, hash);
  silc_hmac_set_key(channel->hmac, hash, silc_hash_len(channel->hmac->hash));
  memset(hash, 0, sizeof(hash));

 out:
  silc_free(id);
  silc_channel_key_payload_free(payload);
}

/* Processes received key for channel. The received key will be used
   to protect the traffic on the channel for now on. Client must receive
   the key to the channel before talking on the channel is possible. 
   This is the key that server has generated, this is not the channel
   private key, it is entirely local setting. */

void silc_client_receive_channel_key(SilcClient client,
				     SilcSocketConnection sock,
				     SilcBuffer packet)
{
  SILC_LOG_DEBUG(("Received key for channel"));

  /* Save the key */
  silc_client_save_channel_key(sock->user_data, packet, NULL);
}

/* Adds private key for channel. This may be set only if the channel's mode
   mask includes the SILC_CHANNEL_MODE_PRIVKEY. This returns FALSE if the
   mode is not set. When channel has private key then the messages are
   encrypted using that key. All clients on the channel must also know the
   key in order to decrypt the messages. However, it is possible to have
   several private keys per one channel. In this case only some of the
   clients on the channel may know the one key and only some the other key.

   If `cipher' and/or `hmac' is NULL then default values will be used 
   (aes-256-cbc for cipher and hmac-sha1-96 for hmac).

   The private key for channel is optional. If it is not set then the
   channel messages are encrypted using the channel key generated by the
   server. However, setting the private key (or keys) for the channel 
   significantly adds security. If more than one key is set the library
   will automatically try all keys at the message decryption phase. Note:
   setting many keys slows down the decryption phase as all keys has to
   be tried in order to find the correct decryption key. However, setting
   a few keys does not have big impact to the decryption performace. 

   NOTE: that this is entirely local setting. The key set using this function
   is not sent to the network at any phase.

   NOTE: If the key material was originated by the SKE protocol (using
   silc_client_send_key_agreement) then the `key' MUST be the
   key->send_enc_key as this is dictated by the SILC protocol. However,
   currently it is not expected that the SKE key material would be used
   as channel private key. However, this API allows it. */

int silc_client_add_channel_private_key(SilcClient client,
					SilcClientConnection conn,
					SilcChannelEntry channel,
					char *cipher,
					char *hmac,
					unsigned char *key,
					uint32 key_len)
{
  SilcChannelPrivateKey entry;
  unsigned char hash[32];
  SilcSKEKeyMaterial *keymat;

  if (!(channel->mode & SILC_CHANNEL_MODE_PRIVKEY))
    return FALSE;

  if (!cipher)
    cipher = "aes-256-cbc";
  if (!hmac)
    hmac = "hmac-sha1-96";

  if (!silc_cipher_is_supported(cipher))
    return FALSE;

  if (!silc_hmac_is_supported(hmac))
    return FALSE;

  /* Produce the key material */
  keymat = silc_calloc(1, sizeof(*keymat));
  if (silc_ske_process_key_material_data(key, key_len, 16, 256, 16, 
					 client->md5hash, keymat) 
      != SILC_SKE_STATUS_OK)
    return FALSE;

  /* Remove the current key, if it exists. */
  if (channel->channel_key) {
    silc_cipher_free(channel->channel_key);
    memset(channel->key, 0, channel->key_len / 8);
    silc_free(channel->key);
    channel->channel_key = NULL;
    channel->key = NULL;
    channel->key_len = 0;
  }
  if (channel->hmac) {
    silc_hmac_free(channel->hmac);
    channel->hmac = NULL;
  }

  if (!channel->private_keys)
    channel->private_keys = silc_dlist_init();

  /* Save the key */
  entry = silc_calloc(1, sizeof(*entry));
  entry->key = silc_calloc(keymat->enc_key_len / 8, sizeof(*entry->key));
  memcpy(entry->key, keymat->send_enc_key, keymat->enc_key_len / 8);
  entry->key_len = keymat->enc_key_len / 8;

  /* Allocate the cipher and set the key*/
  silc_cipher_alloc(cipher, &entry->cipher);
  silc_cipher_set_key(entry->cipher, entry->key, keymat->enc_key_len);

  /* Generate HMAC key from the channel key data and set it */
  silc_hmac_alloc(hmac, NULL, &entry->hmac);
  silc_hash_make(entry->hmac->hash, entry->key, entry->key_len, hash);
  silc_hmac_set_key(entry->hmac, hash, silc_hash_len(entry->hmac->hash));
  memset(hash, 0, sizeof(hash));

  /* Add to the private keys list */
  silc_dlist_add(channel->private_keys, entry);

  if (!channel->curr_key)
    channel->curr_key = entry;

  /* Free the key material */
  silc_ske_free_key_material(keymat);

  return TRUE;
}

/* Removes all private keys from the `channel'. The old channel key is used
   after calling this to protect the channel messages. Returns FALSE on
   on error, TRUE otherwise. */

int silc_client_del_channel_private_keys(SilcClient client,
					 SilcClientConnection conn,
					 SilcChannelEntry channel)
{
  SilcChannelPrivateKey entry;

  if (!channel->private_keys)
    return FALSE;

  silc_dlist_start(channel->private_keys);
  while ((entry = silc_dlist_get(channel->private_keys)) != SILC_LIST_END) {
    silc_dlist_del(channel->private_keys, entry);
    memset(entry->key, 0, entry->key_len);
    silc_free(entry->key);
    silc_cipher_free(entry->cipher);
    silc_hmac_free(entry->hmac);
    silc_free(entry);
  }

  channel->curr_key = NULL;

  silc_dlist_uninit(channel->private_keys);
  channel->private_keys = NULL;

  return TRUE;
}

/* Removes and frees private key `key' from the channel `channel'. The `key'
   is retrieved by calling the function silc_client_list_channel_private_keys.
   The key is not used after this. If the key was last private key then the
   old channel key is used hereafter to protect the channel messages. This
   returns FALSE on error, TRUE otherwise. */

int silc_client_del_channel_private_key(SilcClient client,
					SilcClientConnection conn,
					SilcChannelEntry channel,
					SilcChannelPrivateKey key)
{
  SilcChannelPrivateKey entry;

  if (!channel->private_keys)
    return FALSE;

  silc_dlist_start(channel->private_keys);
  while ((entry = silc_dlist_get(channel->private_keys)) != SILC_LIST_END) {
    if (entry == key) {
      if (channel->curr_key == entry)
	channel->curr_key = NULL;

      silc_dlist_del(channel->private_keys, entry);
      memset(entry->key, 0, entry->key_len);
      silc_free(entry->key);
      silc_cipher_free(entry->cipher);
      silc_hmac_free(entry->hmac);
      silc_free(entry);

      if (silc_dlist_count(channel->private_keys) == 0) {
	silc_dlist_uninit(channel->private_keys);
	channel->private_keys = NULL;
      }

      return TRUE;
    }
  }

  return FALSE;
}

/* Returns array (pointers) of private keys associated to the `channel'.
   The caller must free the array by calling the function
   silc_client_free_channel_private_keys. The pointers in the array may be
   used to delete the specific key by giving the pointer as argument to the
   function silc_client_del_channel_private_key. */

SilcChannelPrivateKey *
silc_client_list_channel_private_keys(SilcClient client,
				      SilcClientConnection conn,
				      SilcChannelEntry channel,
				      uint32 *key_count)
{
  SilcChannelPrivateKey *keys = NULL, entry;
  uint32 count = 0;

  if (!channel->private_keys)
    return NULL;

  silc_dlist_start(channel->private_keys);
  while ((entry = silc_dlist_get(channel->private_keys)) != SILC_LIST_END) {
    keys = silc_realloc(keys, sizeof(*keys) * (count + 1));
    keys[count] = entry;
    count++;
  }

  if (key_count)
    *key_count = count;

  return keys;
}

/* Frees the SilcChannelPrivateKey array. */

void silc_client_free_channel_private_keys(SilcChannelPrivateKey *keys,
					   uint32 key_count)
{
  silc_free(keys);
}
