/* Module command interface.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include "vsd.h"
#include "module.h"
#include "user.h"
#include "mod_user.h"

static struct command_tag clist[] =
{
  {
    CMD_USER_ADD, ADMIN_VS,
    "Create a new user",
    "USER_ADD", 1, "<user name>",
    "user_add", 3, "<host svr> <vs> <user name>"
  },
  {
    CMD_USER_DEL, ADMIN_VS,
    "Delete a user",
    "USER_DEL", 1, "<user name>",
    "user_del", 3, "<host svr> <vs> <user name>"
  },
  {
    CMD_USER_MODIFY, ADMIN_VS,
    "Modify a user account",
    "USER_MOD", 1, "<user name>",
    "user_mod", 3, "<host svr> <vs> <user name>"
  },
  {
    CMD_USER_LIST, ADMIN_VS,
    "List all users on a virtual server",
    "USER_LIST", 0, "",
    "user_list", 2, "<host svr> <vs>"
  },
  {
    CMD_USER_CHPASS, ADMIN_VS,
    "Change the password for a user",
    "USER_CHPASS", 3, "<user> <old pass> <new pass>",
    "user_chpass", 5, "<host svr> <vs> <user> <old pass> <new pass>"
  },
  {
    CMD_USER_GRADD, ADMIN_VS,
    "Create a new group",
    "USER_GRADD", 1, "<group>",
    "user_gradd", 3, "<host svr> <vs> <group name>"
  },
  {
    CMD_USER_GRDEL, ADMIN_VS,
    "Delete a group",
    "USER_GRDEL", 1, "<group>",
    "user_grdel", 3, "<host svr> <vs> <group name>"
  },
  {
    CMD_USER_GRMOD, ADMIN_VS,
    "Add/remove a user from a group",
    "USER_GRMOD", 2, "<group> <+|-><user>",
    "user_grmod", 4, "<host svr> <vs> <group name> <+|-><user>"
  },
  {
    CMD_USER_GRLIST, ADMIN_VS,
    "List all groups on a virtual server",
    "USER_GRLIST", 0, "",
    "user_grlist", 2, "<host svr> <vs>"
  },
  {
    0, 0, NULL, NULL, 0, NULL, NULL, 0, NULL
  }
};

/* Return -1 if `cmd' does not exist.
   Return 1 if there is a syntax error.
   Return 0 if everything is cool.  */
int mod_user_check_cmd (int level, const char *cmd, int argc, char *argv[],
			struct command_tag **list)
{
  if (cmd == NULL)
    {
      /* Return a pointer to all commands.  */
      *list = clist;
      return 0;
    }

  /* Find command in clist.  */
  *list = module_lookup_command (clist, level, cmd);
  if (*list == NULL)
    return -1;  /* Command does not exist.  */

  /* Check command syntax.  Pretty basic at the moment.  */
#ifdef VSD_LIB
  if (argc < (*list)->vsdadm_args)
    return 1; /* Syntax error.  */
#else
  if (argc < (*list)->vsd_args)
    return 1; /* Syntax error.  */
#endif

  /* Syntax is ok.  */
  return 0;
}


#ifdef VSD_LIB
char *mod_user_cmd (void *io, int level, const char *cmd,
		    int argc, char **argv)
{
  struct command_tag *list;
  char *status = NULL;

  list = module_lookup_command (clist, level, cmd);
  if (! list)
    return "notexist";

  vsd_set_level (io, list->level);
  switch (list->token)
    {
    case CMD_USER_ADD:
      vsd_send_command (io, NULL, "USER_ADD %s", argv[2] /* user */);
      break;
    case CMD_USER_DEL:
      vsd_send_command (io, NULL, "USER_DEL %s", argv[2] /* user */);
      break;
    case CMD_USER_MODIFY:
      vsd_send_command (io, NULL, "USER_MOD %s", argv[2] /* user */);
      break;
    case CMD_USER_LIST:
      vsd_send_command (io, NULL, "USER_LIST");
      break; 
   case CMD_USER_CHPASS:
      vsd_send_command (io, NULL, "USER_CHPASS %s %s %s",
			argv[2] /* user */, argv[3] /* old pass */,
			argv[4] /* new pass */);
      break;
    case CMD_USER_GRADD:
      vsd_send_command (io, NULL, "USER_GRADD %s", argv[2] /* group */);
      break;
    case CMD_USER_GRDEL:
      vsd_send_command (io, NULL, "USER_GRDEL %s", argv[2] /* group */);
      break;
    case CMD_USER_GRMOD:
      vsd_send_command (io, NULL, "USER_GRMOD %s", argv[2] /* group */,
			argv[3] /* +|- user */);
      break;
    case CMD_USER_GRLIST:
      vsd_send_command (io, NULL, "USER_GRLIST");
      break;
    default:
      return "mod_user: command not recognised";
      break;
    }

  status = vsd_process_reply (io);
  return status;
}

#else /* ! VSD_LIB */
/* Return 0 if command processed, 1 if not recognised.  */
int mod_user_cmd (struct connection *vc,
		  int level, const char *command,
		  int argc, char *argv[], const char *data)
{
  int token;
  struct command_tag *tag;

  tag = module_lookup_command (clist, level, command);

  if (tag == NULL)
    return 1;

  token = tag->token;
  switch (token)
    {
    case CMD_USER_ADD:
      user_add (vc, argc, argv);
      break;
    case CMD_USER_DEL:
      user_del (vc, argc, argv);
      break;
    case CMD_USER_MODIFY:
      user_mod (vc, argc, argv);
      break;
    case CMD_USER_LIST:
      user_list (vc, argc, argv);
      break;
    case CMD_USER_CHPASS:
      user_chpass (vc, argc, argv);
      break;
    case CMD_USER_GRADD:
      user_group_add (vc, argc, argv);
      break;
    case CMD_USER_GRDEL:
      user_group_del (vc, argc, argv);
      break;
    case CMD_USER_GRMOD:
      user_group_mod (vc, argc, argv);
      break;
    case CMD_USER_GRLIST:
      user_group_list (vc, argc, argv);
      break;
    default:
      return 1;
      break;
    }

  return 0;
}
#endif /* VSD_LIB */

module user_module =
{
  mod_user_check_cmd,
  mod_user_cmd
};
