/* Modify user command.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include "libvsd.h"

int main (int argc, char *argv[])
{
  int c, status, uid = -1;
  char *comment = NULL;
  char *homedir = NULL;
  char *expire_date = NULL;
  char *inactive_time = NULL;
  char *shell = NULL;
  char *username = NULL;
  char *initial_group = NULL;
  char *login_name = NULL;
  char *additional_groups = NULL;
  struct passwd *pw;

  pw = getpwuid (getuid ());
  if (! pw
      || (strcmp (pw->pw_name, "admin") && strcmp (pw->pw_name, "root")))
    {
      fprintf (stderr, "%s: permission denied\n", argv[0]);
      return 1;
    }

  while ((c = getopt (argc, argv, "-c:mMonrd:e:f:G:l:g:k:s:u:")) != -1)
    switch (c)
      {
      case 'c':
	comment = optarg;
	break;
      case 'd':
	homedir = optarg;
	break;
      case 'e':
	expire_date = optarg;
	break;
      case 'f':
	inactive_time = optarg;
	break;
      case 'g':
	initial_group = optarg;
	break;
      case 'G':
	additional_groups = optarg;
	break;
      case 'l':
	login_name = optarg;
	break;
      case 's':
	shell = optarg;
	break;
      case 'u':
	uid = atoi (optarg);
	break;
      case '?':
	fprintf (stderr, "Unknown option `-%c'.\n", optopt);
	return 1;
      default:
        username = optarg;
	break;
      }

  if (username == NULL)
    {
      fprintf (stderr, "syntax: usermod [-u uid [-o]] [-g group] [-G group,...]\n");
      fprintf (stderr, "          [-d home [-m]] [-s shell] [-c comment] [-l new_name]\n");
      fprintf (stderr, "          [-f inactive] [-e expire] [-p passwd] [-L|-U] name\n");
      return 1;
    }

  status = vsd_usermod ("/", username,
			login_name, homedir, initial_group,
			additional_groups, shell, uid, comment);
  if (status == 13)
    printf ("%s: user `%s' contains invalid characters\n", argv[0], username);
  else if (status == 14)
    printf ("%s: there are no free uids/gids\n", argv[0]); 
  else if (status == 11)
    printf ("%s: cannot change %s: permission denied\n", argv[0], username);
  else if (status == 10)
    printf ("%s: couldn't access /etc/passwd: %s\n", argv[0],
	    strerror (errno));
  else if (status == 9)
    printf ("%s: user `%s' is already in use\n", argv[0], username);
  else if (status == 7)
    printf ("%s: user `%s' does not exist\n", argv[0], username);
  else if (status == 6)
    printf ("%s: group `%s' does not exist\n", argv[0], initial_group);
  else if (status == 4)
    printf ("%s: uid %d is already in use\n", argv[0], uid);
  else if (status == 2)
    printf ("%s: there is an error somewhere...\n", argv[0]);
  else
    {
      /* Rename user in /etc/vsd/priv.  */
      if (login_name)
	vsd_priv_rename_user ("/", username, login_name);
    }

  return status;
}
