/* Set user privileges.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include "libvsd.h"

int main (int argc, char *argv[])
{
  int x;
  char *username = NULL;
  struct vsd_file_line *priv_file;
  struct passwd *pw;

  pw = getpwuid (getuid ());
  if (! pw
      || (strcmp (pw->pw_name, "admin") && strcmp (pw->pw_name, "root")))
    {
      fprintf (stderr, "%s: permission denied\n", argv[0]);
      return 1;
    }

  /* setrights <username> [<+|->privilege] */
  if (argc <= 1)
    {
      fprintf (stderr, "syntax: setrights <username> [<+|-><privilege>] ...\n");
      return 1;
    }

  if (vsd_priv_load (&priv_file, "/"))
    {
      fprintf (stderr, "privileges file not found\n");
      return 1;
    }

  username = argv[1];
  pw = getpwnam (username);
  if (pw == NULL)
    {
      fprintf (stderr, "user %s does not exist\n", username);
      return 1;
    }

  for (x = 2; x < argc; x++)
    {
       char *priv = argv[x];

      if (*priv == '+')
	{
	  if (strcmp (priv + 1, "login") == 0)
	    {
	      fprintf (stderr, "You cannot grant the login privilege\n");
	      return 1;
	    }

	  /* Check admin user has the privilege.  */
	  if (strcmp (priv + 1, "chrtftp") != 0)
	    {
	      if (vsd_check_privilege (priv_file, "admin", priv + 1))
		{
		  fprintf (stderr, "User admin does not have %s privilege\n",
			   priv + 1);
		  return 1;
		}
	    }
	  else
	    {
	      /* Add ftp privilege if user hasn't already got it but then
	         only if the admin user has it.  */
	      if (vsd_check_privilege (priv_file, "admin", "ftp"))
		{
		  fprintf (stderr, "User admin does not have ftp privilege\n");
		  return 1;
		}
	      vsd_priv_add_member (priv_file, "ftp", username);
	    }

	  if (strcmp (priv + 1, "telnet") == 0
	      && vsd_check_privilege (priv_file, "admin", "login"))
	    {
	      fprintf (stderr,
		       "You need the login privilege to assign telnet\n");
	      return 1;
	    }

	  vsd_priv_add_member (priv_file, priv + 1, username);
	}
      else if (*priv == '-')
	{
	  vsd_priv_delete_member (priv_file, priv + 1, username);

	  /* Automatically remove the chrtftp privilege.  */
	  if (strcmp (priv + 1, "ftp") == 0)
	    vsd_priv_delete_member (priv_file, "chrtftp", username);
	}
    }

  vsd_priv_save (priv_file, "/");
  vsd_priv_free (priv_file);

  return 0;
}
