/* List user quotas.
   Copyright (c) 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/stat.h>
#include "libvsd.h"

#ifdef HAVE_SYS_QUOTA_H
#include <sys/quota.h>
#else
#ifdef HAVE_UFS_UFS_QUOTA_H
#include <ufs/ufs/quota.h>
#endif
#endif

/* DEV is passed to quotactl.  */
#if defined(HAVE_SYS_QUOTA_H)
#define DEV(map,vs) (map)->partition[(vs)->partition].device
#elif defined(HAVE_UFS_UFS_QUOTA_H)
#define DEV(map,vs) (map)->partition[(vs)->partition].pathname
#endif

static int diskquota (int cmd, const char *special, int uid, caddr_t addr)
{
#ifdef HAVE_SYS_QUOTA_H
  /* Linux style.  */
  return quotactl (cmd, special, uid, addr);
#else
#ifdef HAVE_UFS_UFS_QUOTA_H
  /* BSD style.  */
  return quotactl (special, cmd, uid, addr);
#endif
#endif
}

/* Get quota stats for all users on the virtual server.
   If uid is NULL then don't create a uid list.
   If puser is NULL then don't create user quota stats.
   If total is NULL then don't tally totals.  */
static int vs_user_stats (struct vsd_file_line *pw_file,
			  int **puid, void **puser_, void *total_,
			  const char *device)
{
  struct dqblk user_quota;
  int count, max, status;
  int *uid;
  struct dqblk *user;
  struct dqblk **puser = (struct dqblk **)puser_;
  struct dqblk *total = (struct dqblk *)total_;
  uid_t min_uid, max_uid;
  struct passwd pw;
  struct vsd_file_line *ent;

  if (puid == NULL && puser == NULL && total == NULL)
    return 0;

  vsd_get_uidgid ("/", &min_uid, &max_uid, NULL, NULL);

  max = 16;
  count = 0;
  uid = (int *) malloc (max * sizeof (int));
  user = (struct dqblk *) malloc (max * sizeof (struct dqblk));

  if (total != NULL)
    {
      total->dqb_bhardlimit = 0;
      total->dqb_bsoftlimit = 0;
      total->dqb_curblocks = 0;
    }

  /* Look at each /etc/passwd entry but only find quota information
     for users on the virtual server.  */
  ent = pw_file;
  do
    {
      ent = vsd_getpwent (ent, &pw);

      if (pw.pw_uid >= min_uid && pw.pw_uid <= max_uid)
	{
	  if (puid != NULL)
	    uid[count] = pw.pw_uid;
	  
	  status = diskquota (QCMD (Q_GETQUOTA, USRQUOTA), device,
			      pw.pw_uid, (caddr_t) &user_quota);
	  if (status == 0)
	    {
	      if (puser != NULL)
		user[count] = user_quota;
	      
	      if (count >= max - 1)
		{
		  max += 16;
		  if (puid != NULL)
		    uid = (int *) realloc (uid, max * sizeof (int));
		  if (puser != NULL)
		    user = (struct dqblk *) realloc (user,
						     max * sizeof (struct dqblk));
		}

	      if (total != NULL)
		{
		  total->dqb_bhardlimit += user_quota.dqb_bhardlimit;
		  total->dqb_bsoftlimit += user_quota.dqb_bsoftlimit;
		  total->dqb_curblocks += user_quota.dqb_curblocks;
		}
	    }
	  else
	    {
	      if (puser != NULL)
		memset (&user[count], 0, sizeof (struct dqblk));
	    }

	  count ++;
	}
    } while (ent != NULL);

  if (puid)
    *puid = uid;
  if (puser)
    *puser = user;

  return count;
}

int main (int argc, char *argv[])
{
  const char *username = NULL;
  int *uidv, count, fd, status;
  struct dqblk *userv, totalv;
  struct passwd pw;
  struct vsd_file_line *pw_file;
  struct stat sb;
  char template[32];

  {
    struct passwd *p = getpwuid (getuid ());
    if (! p
	|| (strcmp (p->pw_name, "admin") && strcmp (p->pw_name, "root")))
      {
	fprintf (stderr, "%s: permission denied\n", argv[0]);
	return 1;
      }
  }

  if (argc != 1 && argc != 2)
    {
      fprintf (stderr, "syntax: quotastats [<username>]\n");
      return 1;
    }

#ifdef TEST
  if (stat ("./", &sb))
#else
  if (stat ("/etc/passwd", &sb))
#endif
    {
      fprintf (stderr, "error: cannot stat /etc/passwd: %s\n",
	       strerror (errno));
      return 1;
    }

  if (argc == 2)
    username = argv[1];

  vsd_passwd_load (&pw_file, "/");

  if (username == NULL)
    {
      printf ("=================+========+=============+============\n");
      printf ("User name        | Uid    | Limit       | Current use\n");
      printf ("=================+========+=============+============\n");
    }

  /* Get disk quota information from the kernel.  We have to create
     a special block file with the same device number as the hard
     disk that the virtual server is located on.  The stat of /etc/passwd
     is a good enough way of determining the device number.  */
  strcpy (template, "/tmp/hdXXXXXX");
  fd = mkstemp (template);
  close (fd);
  unlink (template);
  mknod (template, 0600 | S_IFBLK, sb.st_dev);
  count = vs_user_stats (pw_file, &uidv, (void **) &userv, &totalv, template);
  unlink (template);

  status = 0;
  if (username == NULL)
    {
      unsigned int serverquota = vsd_quota_get ("/") * 1024;
      while (--count >= 0)
	if (! vsd_getpwuid (pw_file, uidv[count], &pw))
	  printf ("%17s|%8d|%12dK|%11dK\n", pw.pw_name, uidv[count],
		  userv[count].dqb_bhardlimit, userv[count].dqb_curblocks);
      printf ("=================+========+=============+============\n");
      printf ("Allocated/used   |        |%12dK|%11dK\n",
	      totalv.dqb_bhardlimit, totalv.dqb_curblocks);
      printf ("Max/available    |        |%12uK|%11dK\n",
	      serverquota, serverquota - totalv.dqb_curblocks);
      printf ("=================+========+=============+============\n");
    }
  else
    {
      if (! vsd_getpwnam (pw_file, username, &pw))
	{
	  while (--count >= 0 && uidv[count] != pw.pw_uid)
	    ;
	  printf ("Disk quota for user %s (uid %d)\n", username, pw.pw_uid);
	  printf ("        Limit : %dK\n", userv[count].dqb_bhardlimit);
	  printf ("  Current use : %dK\n", userv[count].dqb_curblocks);
	}
      else
	{
	  fprintf (stderr, "user %s does not exist\n", username);
	  status = 1;
	}
    }

  vsd_passwd_free (pw_file);
  free (uidv);
  free (userv);
  return status;
}
