/* Virtualise inetd services.
   Copyright (c) 2000 Idaya Ltd.
   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* This is based on the idea presented in the Virtual Services HOWTO:
     http://metalab.unc.edu/pub/Linux/docs/HOWTO/Virtual-Services-HOWTO  */

#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/syslog.h>
#include "libvsd.h"

/* Map an IP address to a virtual server name.
   Return NULL on error and "localhost" if no translation could be found.  */
static struct vsd_vs *find_virtual_server (struct vsd_vs_map *map,
					   int *error)
{
  struct sockaddr_in addr;
  struct vsd_vs *vs;
  int namelen;
  char *ip;

  *error = 0;
  namelen = sizeof (addr);
  if (getsockname (0, (struct sockaddr *)&addr, &namelen) < 0)
    {
      syslog (LOG_ERR, "getsockname failed: %m");
      *error = 1;
      return NULL;
    }

  ip = inet_ntoa (addr.sin_addr);

  /* Find virtual server.  */
  vs = vsd_getmapent (map, NULL, ip);
  if (vs != NULL)
    {
      /* Is the virtual server account disabled ?  */
      if (vs->status != 0)
        {
          syslog (LOG_INFO, "attempted access to disabled account %s",
                  vs->name);
	  *error = 2;
          return NULL;
        }
      return vs;
    }

  /* Address not found. Assume that it is an attempt to access the
     hosting server.  */
  return NULL;
}

int main (int argc, char **argv)
{
  struct vsd_vs *vs;
  struct vsd_vs_map *map;
  int error;

  openlog ("virtuald", LOG_PID, LOG_DAEMON);
  map = vsd_map_read ();
  if (map == NULL)
    {
      syslog (LOG_ERR, "can't read vsd.conf: %m");
      return 1;
    }

  if (argc <= 1)
    {
      syslog (LOG_ERR, "no arguments");
      return 1;
    }

#ifdef DEBUG
  syslog (LOG_INFO, "exec %s", argv[1]);
#endif

  vs = find_virtual_server (map, &error);
  if (vs)
    {
      char *path = vsd_map_server_root (map, vs->name);
      if (! path || chroot (path))
	{
	  syslog (LOG_ERR, "chroot to %s failed: %m", path);
	  return 1;
	}

      if (chdir ("/"))
	{
	  syslog (LOG_ERR, "chdir failed: %m");
	  return 1;
	}
    }

  if (error)
    return 1;

  execvp (argv[1], argv + 1);
  syslog (LOG_ERR, "exec %s failed: %m", argv[1]);
  closelog();
  return 127;
}
