/* Become a virtual server.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* This application has been written to remove the requirement for
   support staff to have root access to a hosting server in order
   to perform administrative tasks on a virtual server for which
   they have no password access.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>
#include "libvsd.h"

/* bevs takes `-r' as an optional command line argument.  This will
   allow a unprivileged user to login as the super-user on a virtual
   server.  This could be considered a security flaw and a better
   option would be to ask for the super-user password before allowing
   this.  */
int main (int argc, char *argv[])
{
  const char *vs, *vsroot, *temp;
  char buff[64];
  struct vsd_vs_map *map;
  struct vsd_vs *ent;
  const char *new_argv[6], *new_env[6];
  struct passwd *pw;
  int superuser = 0;
  int targetuid = 0;

  if (argc != 2 && argc != 3)
    {
      printf ("syntax: bevs [-r] <virtual-server>\n");
      return 1;
    }

  if (argc == 3 && strcmp (argv[1], "-r") == 0)
    {
      superuser = 1;
      vs = argv[2];
    }
  else if (argc == 3 && strcmp (argv[2], "-r") == 0)
    {
      superuser = 1;
      vs = argv[1];
    }
  else if (argc == 2)
    vs = argv[1];
  else if (argc == 3)
    {
      printf ("syntax: bevs [-r] <virtual-server>\n");
      return 1;
    }
  
  /* Get virtual server details.  */
  map = vsd_map_read ();
  if (map == NULL)
    {
      printf ("-- cannot read server map file: %s\n", strerror (errno));
      return 1;
    }

  ent = vsd_getmapent (map, vs, NULL);
  if (ent == NULL)
    {
      printf ("-- cannot find virtual server `%s'\n", vs);
      return 1;
    }

  /* Obtain the pathname to the virtual server root directory.  */
  vsroot = vsd_map_server_root (map, vs);

  if (chroot (vsroot))
    {
      printf ("-- cannot chroot to virtual server: %s\n",
	      strerror (errno));
      _exit (127);
    }
  
  chdir ("/");
  pw = getpwuid (ent->startuid);
  targetuid = (superuser) ? 0 : ent->startuid;
  if (superuser)
    initgroups ("root", targetuid);
  else
    initgroups ("admin", targetuid);

  setgid (targetuid);
  setuid (targetuid);

  new_argv[0] = "sh";
  new_argv[1] = "-i";
  new_argv[2] = "-c";
  new_argv[3] = (char *) strdup (pw->pw_shell);
  new_argv[4] = NULL;

  /* Create standard environment variables.  */

  /* Preserve TERM from caller's terminal settings.  */
  temp = getenv ("TERM");
  if (! temp)
    temp = "dumb";
  sprintf (buff, "TERM=%s", temp);
  new_env[0] = strdup (buff);

  /* Add HOME.  */
  sprintf (buff, "HOME=%s", pw->pw_dir);
  new_env[1] = strdup (buff);

  /* Add USER.  */
  sprintf (buff, "USER=%s", pw->pw_name);
  new_env[2] = strdup (buff);

  new_env[3] = strdup ("PS1=[\\u@\\h \\W]\\$ ");
  new_env[4] = strdup ("PATH=/usr/local/bin:/usr/local/sbin:/sbin:/bin:/usr/sbin:/usr/bin");
  new_env[5] = NULL;
  execve ("/bin/sh", (char *const *) new_argv, (char *const *) new_env);

  return 0;
}
