/* User privilege and general validation checking.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>

#include "libvsd.h"

/* Get minimum and maximum uid and gid ranges for a virtual server.
   If server_root is NULL then assume we are chroot to the VS.
   Accept uid/gid pointers as NULL, returning values only for those
   pointers that are not NULL (obviously).
   Return 0 on success, 1 on failure.  */
int vsd_get_uidgid (const char *server_root,
		    uid_t *uid_min, uid_t *uid_max,
		    gid_t *gid_min, gid_t *gid_max)
{
  struct vsd_file_line *logindefs, *line;
  char *p, key[128];
  int x;

  if (! server_root)
    server_root = "/";

  if (vsd_load_file (&logindefs, server_root, "/etc/login.defs"))
    return 1;

  for (line = logindefs; line; line = line->next)
    {
      /* Skip white space.  */
      for (p = line->line; isspace (*p); p++)
	;

      /* Ignore comment lines and skip empty lines.  */
      if (*p == '#' || *p == '\0')
	continue;

      for (x = 0; ! isspace (*p); x++, p++)
	key[x] = *p;
      key[x] = '\0';

      if (uid_min && ! strcmp (key, "UID_MIN"))
	*uid_min = (uid_t) atoi (p);
      if (uid_max && ! strcmp (key, "UID_MAX"))
	*uid_max = (uid_t) atoi (p);
      if (gid_min && ! strcmp (key, "GID_MIN"))
	*gid_min = (gid_t) atoi (p);
      if (gid_max && ! strcmp (key, "GID_MAX"))
	*gid_max = (gid_t) atoi (p);
    }
  vsd_free_file (logindefs);
  return 0;
}

/* Is `uid' within the range available to the virtual server.
   Return -1 on error, 0 on failure and 1 on success.  */
int vsd_valid_uid (uid_t uid)
{
  uid_t uid_min, uid_max;

  if (vsd_get_uidgid (NULL, &uid_min, &uid_max, NULL, NULL))
    return -1;

  if (uid < uid_min || uid > uid_max)
    {
      errno = ERANGE;
      return 0;
    }

  return 1;
}

/* Is `gid' within the range available to the virtual server.
   Return -1 on error, 0 on failure and 1 on success.  */
int vsd_valid_gid (gid_t gid)
{
  gid_t gid_min, gid_max;

  if (vsd_get_uidgid (NULL, NULL, NULL, &gid_min, &gid_max))
    return -1;

  if (gid < gid_min || gid > gid_max)
    {
      errno = ERANGE;
      return 0;
    }

  return 1;
}

/* Check uid is a valid user on the virtual server.
   Return 1 if valid, 0 if not valid and -1 on error.  */
int vsd_validate_user_uid (uid_t uid)
{
  struct passwd *pw;

  if (uid == 0) /* Allow root user */
    return 1;

  /* Check user is defined in the password list.  */
  pw = getpwuid (uid);
  if (pw == NULL)
    {
      errno = ENOENT;
      return 0;
    }

  /* Check uid is in the range defined for the virtual server.  */
  return vsd_valid_uid (uid);
}

/* Check `username' is a valid user on the virtual server.
   Return 1 if valid, 0 if not valid and -1 on error.  */
int vsd_validate_user_name (const char *username)
{
  struct passwd *pw;

  if (strcmp (username, "root") == 0)
    return 1;

  if (username == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  pw = getpwnam (username);
  if (pw == NULL)
    {
      errno = ENOENT;

      return 0;
    }

  return vsd_valid_uid (pw->pw_uid);
}

/* Return 1 if `uid' belongs to the admin user. 0 if not.  */
int vsd_admin_user (uid_t uid)
{
  struct passwd *pw;

  pw = getpwuid (uid);
  if (pw == NULL)
    {
      errno = ENOENT;
      return -1;
    }

  if (strcmp (pw->pw_name, "admin") == 0)
    return 1;

  return 0;
}

/* Return 0 on success, 1 on failure, -1 on error.  */
int vsd_check_path (const char *path, int check_file, int access_rights)
{
  struct stat b;
  char *tmp, *p;

  if (check_file)
    if (access (path, access_rights))
      {
	if (errno == EACCES)
	  {
	    if (stat (path, &b))
	      return 1;
	    
	    if (vsd_valid_uid (b.st_uid) == 1
		&& vsd_valid_gid (b.st_gid) == 1)
	      return 0;
	  }
	
	if (errno != ENOENT)
	  /* Access denied.  */
	  return 1;
      }

  /* Destination exists and we are allowed to write to it.  */

  /* Strip file from path.  */
  tmp = alloca (strlen (path) + 1);

  strcpy (tmp, path);
  p = strrchr (tmp, '/');
  if (p == NULL)
    strcpy (tmp, "./");
  else
    *++p = '\0';

  if (access (tmp, access_rights))
    {
      if (errno == EACCES)
	{
	  if (stat (tmp, &b))
	    return 1;
	  /* access says that we don't (as the ordinary user) have permission
	     to access the path. But we say the admin user should be granted
	     permission provided the destination uid/gid is within the
	     ranges of his virtual server.  */
	  if (vsd_valid_uid (b.st_uid) == 1 && vsd_valid_gid (b.st_gid) == 1)
	    return 0;
	}

      /* Access denied.  */
      return 1;
    }

  /* Access granted.  */
  return 0;
}

/* Is pid a member of our Virtual Server ?
   Return 1 if yes, 0 if no, -1 on error.  */
int vsd_owner_pid (pid_t pid)
{
  char tmp[128], buff[256];

  /* If the process is not a member of our virtual server then
     readlink will return a permission denied error.  */
  sprintf (tmp, "/proc/%d/root", pid);
  memset (buff, 0, sizeof (buff));
  if (readlink (tmp, buff, sizeof (buff)) == -1)
    return -1;

  if (buff[0] == '/' && buff[1] == '\0')
    return 1;

  return 0;
}

/* Return NULL if the username valid.  Otherwise return a string
   indicating the error.  */
const char *vsd_valid_username (const char *user)
{
  if (user == NULL)
    return "is_username_valid: passed NULL argument";

  if (strspn (user, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_") != strlen (user))
    return "contains unsuitable characters";

  if (isdigit (user[0]))
    return "cannot start with a digit";

  /* The shadow password suite restricts us to 16 characters.  */
  if (strlen (user) > 16)
    return "maximum length is 16 characters";

  return NULL;
}

/* Return NULL if the username is valid.  Otherwise return a string
   indicating the error.  */
const char *vsd_valid_vsname (const char *vs)
{
  if (vs == NULL)
    return "is_vsname_valid: passed NULL argument";

  if (strspn (vs, "abcdefghijklmnopqrstuvwxyz0123456789-_") != strlen (vs))
    return "contains unsuitable characters";

  if (isdigit (vs[0]))
    return "cannot start with a digit";

  /* Restrict virtual server names to 20 characters.  */
  if (strlen (vs) > 20)
    return "name is too long";

  return NULL;
}

/* Simplify a path name in place, deleting redundant components.  This
   reduces OS overhead and guarantees that equivalent paths compare
   the same (modulo symlinks).

   Transforms made:
   foo/bar/../quux      foo/quux
   foo/./bar            foo/bar
   foo//bar             foo/bar
   /../quux             /quux
   //quux               //quux  (POSIX allows leading // as a namespace escape)

   Guarantees no trailing slashes. All transforms reduce the length
   of the string.

   This function is shamelessly taken from cpplib in the GNU GCC source
   tree.  */
static void simplify_pathname (char *path)
{
  char *from = path, *to = path;
  char *base;
  int absolute = 0;

  /* Remove redundant initial /s.  */
  if (*from == '/')
    {
      absolute = 1;
      to++;
      from++;
      if (*from == '/')
        {
	  if (*++from == '/')
	    /* 3 or more initial /s are equivalent to 1 /.  */
	    while (*++from == '/');
	  else
	    /* On some hosts // differs from /; Posix allows this.  */
	    to++;
        }
    }
  base = to;
  
  for (;;)
    {
      while (*from == '/')
	from++;
      
      if (from[0] == '.' && from[1] == '/')
	from += 2;
      else if (from[0] == '.' && from[1] == '\0')
	goto done;
      else if (from[0] == '.' && from[1] == '.' && from[2] == '/')
        {
	  if (base == to)
            {
	      if (absolute)
		from += 3;
	      else
                {
		  *to++ = *from++;
		  *to++ = *from++;
		  *to++ = *from++;
		  base = to;
                }
            }
	  else
            {
	      to -= 2;
	      while (to > base && *to != '/') to--;
	      if (*to == '/')
		to++;
	      from += 3;
            }
        }
      else if (from[0] == '.' && from[1] == '.' && from[2] == '\0')
        {
	  if (base == to)
            {
	      if (!absolute)
                {
		  *to++ = *from++;
		  *to++ = *from++;
                }
            }
	  else
            {
	      to -= 2;
	      while (to > base && *to != '/') to--;
	      if (*to == '/')
		to++;
            }
	  goto done;
        }
      else
	/* Copy this component and trailing /, if any.  */
	while ((*to++ = *from++) != '/')
	  if (!to[-1])
	    {
	      to--;
	      goto done;
	    }
    }

 done:
  /* Trim trailing slash */
  if (to[0] == '/' && (!absolute || to > path+1))
    to--;
  
  /* Change the empty string to "." so that stat() on the result
     will always work. */
  if (to == path)
    *to++ = '.';
  
  *to = '\0';
}

/* Check that the path `path' exists with the virtual server given by
   `server_root'.  If the path doesn't exist then check each element of
   the tree and ensure that the user has write access at each relevant
   level.  If the user does have write access and a path element doesn't
   exist then create it.  Return NULL upon successful completion and
   a pointer to an error string upon encountering an error.

   `path' can be terminated by either NULL or whitespace.
   `server_root' is the path to the root of the virtual server.
   `dir' if set then the path should point to a directory.  */
const char *vsd_check_and_make_path (const char *server_root, char *path,
				     int dir)
{
  int ret, arglen;
  char *q, *p, *argp;
  uid_t min_uid, max_uid;
  struct stat sb;

  /* Get uid boundaries for the virtual server.  */
  vsd_get_uidgid (server_root, &min_uid, &max_uid, NULL, NULL);

  /* Find end of path - might be NULL or space terminated.  */
  q = path;
  while (*q && ! isspace (*q))
    q++;
  arglen = q - path;
  argp = (char *) malloc (arglen + 1 + strlen (server_root) + 1);
  if (argp == NULL)
    return "Out of memory";

  strcpy (argp, server_root);
  strncat (argp, path, arglen);
  argp[arglen + strlen (server_root)] = '\0';

  /* The fast case. The sensible user has already set the
     paths up.  */
  ret = stat (argp, &sb);

  /* Chop path to virtual server root.  */
  strncpy (argp, path, arglen);
  argp[arglen] = '\0';

  if (ret == 0)
    {
      /* Something exists but we need to ensure it is of the right type. */
      if (dir && ! S_ISDIR (sb.st_mode))
	{
	  free (argp);
	  return "Path exists but points to a file rather than a directory";
	}
      else if (! dir && S_ISDIR (sb.st_mode))
	{
	  free (argp);
	  return "Path exists but points to a directory rather than a file";
	}
    }
  else if (*argp == '/')
    {
      /* Currently we can only cope with absolute pathnames.  */
      char *path;
      uid_t uid = 0;
      gid_t gid = 0;
      int pathlen;
      
      /* An element, or many elements, of the path do not exist.  Create the
	 missing pieces and set make them owned by the user of the last piece
	 of existing directory.  We have to be careful of hacker types who
	 try to abuse the system and create directories anywhere on the VS.
	 In this case we make sure that directories can only be created
	 within directories already owned by a user on the VS.  */
      simplify_pathname (argp);
      pathlen = strlen (argp);
      path = (char *) malloc (pathlen + 1);
      path[0] = '.'; path[1] = '\0';
      chdir (server_root);

      /* Don't create the final element as a directory if the path is
	 supposed to point to a file.  */
      if (! dir)
	pathlen = strrchr (argp, '/') - argp;
      
      /* Walk the path.  Skip multiple `/'.  */
      p = argp;
      while (*p && *p != '/')
	p++;
      p++;
      while (pathlen > (p - argp))
	{
	  /* Skip leading slashes.  */
	  q = p;
	  while (*q && *q != '/')
	    q++;
	  
	  strcat (path, "/");
	  strncat (path, p, q - p);
	  p = q + 1;
	  if (stat (path, &sb) && errno == ENOENT)
	    {
	      if (uid < min_uid || uid > max_uid)
		{
		  free (argp);
		  return "Cannot use path: permission denied";
		  /* io_buffer_store (vc, "-ERR Cannot use path `%s': permission denied on segment `%s'", argp, path); */
		}
	      else
		{
		  /* Create the directory and set the ownership to
		     the same as the parent directory.  */
		  mkdir (path, 0755);
		  chown (path, uid, gid);
		}
	    }
	  else
	    {
	      /* Remember the uid and gid of this directory. We will
		 refer to it when creating new directories.  */
	      uid = sb.st_uid;
	      gid = sb.st_gid;
	    }
	}
    }
  else
    {
      free (argp);
      return "vsd_check_and_make_path: unsupported";
    }

  free (argp);
  return NULL;
}


/* Do some simple syntax checking on an IP address.
   Return NULL if IP address is valid.
   Return a pointer to an error string if invalid.  */
const char *vsd_check_ip_address (const char *ip)
{ 
  int len, x, y, i1, i2, i3, i4;

  if (ip == NULL)
    return "invalid argument";

  /* IP addresses can only contain numbers and dots.  */
  if (strcspn (ip, "0123456789."))
    return "address contains invalid characters";

  len = strlen (ip);
  /* There should be three dots. */
  for (y = 0, x = 0; x < len; x++)
    if (ip[x] == '.')
      {
	y++;
	if (ip[x + 1] == '.')
	  return "address contains `..' sequence";
      }     

  if (y != 3)
    return "there should be three dots in an IP address";

  /* It should start and end with a digit.  */
  if (! isdigit (ip[0]) && ! isdigit (ip[len]))
    return "address should start and end with a digit";

  /* Numbers should range 0 - 255.  */
  sscanf (ip, "%d.%d.%d.%d", &i1, &i2, &i3, &i4);
  if (i1 < 0 || i1 > 255 || i2 < 0 || i2 > 255
      || i3 < 0 || i3 > 255 || i4 < 0 || i4 > 255)
    return "address has an invalid value";

  return NULL;
}
