/* Obtain information from the configuration files.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* This module provides a interface that all clients should use for
   querying or changing the information in /etc/vsd.conf.  */

/* FIXME: This whole file could do with a bit of re-writing.  The
   format of /etc/vsd.conf is terrible and is very prone to error if
   the user does some hand editing.  */

/* The format of /etc/vsd.conf is:
      <Global>
      PrimaryNS ns1.dsvr.net
      SecondaryNS ns2.dsvr.net
      ErrorLog /var/log/vsd.log
      UIDRange 1000 30000
      UIDAllocChunk 300
      BWLimit 32Kbit
      BWBurst 256Kbit
      </Global>

      <Partition 0>
      Mount /virtual/disk0
      MaxVS 40
      </Partition>

      <Partition 1>
      Mount /virtual/disk1
      MaxVS 50
      </Partition>

      <VirtualServer vsname>
      IP 1.2.3.4
      StartUID 1000
      Users 200
      Quota 300
      Partition 0
      Status active
      PrimaryNS our-ns1.com
      SecondaryNS our-ns2.com
      AliasIP 1.2.3.5
      AliasIP 1.2.3.6
      BWLimit 6Kbit
      BWBurst 12KBit
      </VirtualServer>

   Within a virtual server decl, name server arguments are optional.  */


#include "config.h"
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <unistd.h>

#include "libvsd.h"

/* Convert an IP network address to a standard numbers-and-dots string.  */
char *vsd_ip_ntoa (void *ip)
{
  return inet_ntoa (*((struct in_addr *) ip));
}

/* Convert a standard numbers-and-dots string into an IP network address.  */
int vsd_ip_aton (const char *ip, void *inp)
{
  return inet_aton (ip, inp);
}


/* Allocate memory of size `bytes' bytes.  If ptr is not NULL, then assume
   we are re-allocating memory.  */
static void *xalloc (void *ptr, size_t bytes)
{
  if (! ptr)
    return calloc (bytes, sizeof (char));
  return realloc (ptr, bytes);
}

/* Shamelessly stolen from iproute2.
   Parse a string such as 2Mbit, 2Kbit and return a value
   in bytes per second.  */
static unsigned int get_rate (char *s)
{
  char *p;
  double bps = strtod (s, &p);

  if (p == s)
    return 0;

  if (*p)
    {
      if (strcasecmp (p, "kbps") == 0)
	bps *= 1024.0;
      else if (strcasecmp (p, "mbps") == 0)
	bps *= 1024.0 * 1024.0;
      else if (strcasecmp (p, "mbit") == 0)
	bps *= 1024.0 * 1024.0 / 8.0;
      else if (strcasecmp (p, "kbit") == 0)
	bps *= 1024.0 / 8.0;
      else if (strcasecmp (p, "bps") == 0)
	return (unsigned int) bps;
    } else
      bps /= 8.0;

  return (unsigned int) bps;
}

/* Parse a <Partition>...</Partition> construct.  */
static struct vsd_file_line *read_partition (struct vsd_vs_map *map,
					     struct vsd_file_line *vsconf,
					     int part)
{
  struct vsd_file_line *line = vsconf;
  char *p, **argv;
  int partition, argc;

  memset (&map->partition[part], 0, sizeof (struct vsd_disk));

  /* Get <arg> argument in <Partition <arg>>.  First skip all spaces
     and then find the space preceeding the argument.  */
  for (p = line->line; *p && isspace (*p); p++)
    ;

  if (strncmp (p, "<Partition ", 11) != 0)
    {
      vsd_log_error (VSD_LOG_ERROR,
		     "vsd.conf: read_partition called on non-<Partition>");
      return line;
    }

  for (; *p != ' '; p++)
    ;
  partition = strtol (p, NULL, 10);

  line = line->next;
  for (; line; line = line->next)
    {
      /* Skip white space.  */
      for (p = line->line; *p && isspace (*p); p++)
	;
      /* Skip comments and blank lines.  */
      if (*p == '\0' || *p == '#')
	continue;

      if (strncmp (p, "</Partition>", 12) == 0)
	break;

      argv = vsd_argv_parse (' ', p, &argc);
      if (argc != 2)
	{
	  /* Invalid number of arguments.  */
	  vsd_log_error (VSD_LOG_ERROR,
			 "vsd.conf: <Partition %d>: invalid number of args `%.128s'",
			 partition, p);
	  if (argv)
	    free (argv);
	  continue;
	}

      if (strcmp (argv[0], "Mount") == 0)
	map->partition[partition].mount = strdup (argv[1]);
      else if (strcmp (argv[0], "MaxVS") == 0)
	map->partition[partition].maxvs = strtol (argv[1], NULL, 10);
      else
	vsd_log_error (VSD_LOG_ERROR,
		       "vsd.conf: <Partition %d>: unrecognised command `%.128s'",
		       partition, argv[0]);
      free (argv);
    }

  return line;
}

/* Parse a <Global>...</Global> construct.  */
static struct vsd_file_line *read_global (struct vsd_vs_map *map,
					  struct vsd_file_line *vsconf)
{
  struct vsd_file_line *line = vsconf;
  char *p, **argv;
  int argc;

  for (p = vsconf->line; *p && isspace (*p); p++)
    ;
  if (strncmp (p, "<Global>", 8) != 0)
    {
      vsd_log_error (VSD_LOG_ERROR,
		     "vsd.conf: read_global called on non-<Global>");
      return line;
    }

  for (line = vsconf->next; line; line = line->next)
    {
      /* Skip white space.  */
      for (p = line->line; *p && isspace (*p); p++)
	;
      /* Skip comments and blank lines.  */
      if (*p == '\0' || *p == '#')
	continue;

      if (strncmp (p, "</Global>", 9) == 0)
	break;

      argv = vsd_argv_parse (' ', p, &argc);
      if (argc != 2
	  && (strcmp (argv[0], "UIDRange") == 0 && argc != 3))
	{
	  /* Invalid number of arguments.  */
	  vsd_log_error (VSD_LOG_ERROR,
			 "vsd.conf: <Global>: invalid number of args `%.128s'", p);
	  if (argv)
	    free (argv);
 	  continue;
	}

      if (strcmp (argv[0], "PrimaryNS") == 0)
	strcpy (map->global.ns1, argv[1]);
      else if (strcmp (argv[0], "SecondaryNS") == 0)
	strcpy (map->global.ns2, argv[1]);
      else if (strcmp (argv[0], "ErrorLog") == 0)
	map->global.errorlog = strdup (argv[1]);
      else if (strcmp (argv[0], "UIDRange") == 0)
	{
	  map->global.minuid = strtol (argv[1], NULL, 10);
	  map->global.maxuid = strtol (argv[2], NULL, 10);
	}
      else if (strcmp (argv[0], "UIDAllocChunk") == 0)
	map->global.uidchunk = strtol (argv[1], NULL, 10);
      else if (strcmp (argv[0], "BWLimit") == 0)
	map->global.bwlimit = get_rate (argv[1]);
      else if (strcmp (argv[0], "BWBurst") == 0)
	map->global.bwburst = get_rate (argv[1]);
      else
	vsd_log_error (VSD_LOG_ERROR,
		       "vsd.conf: <Global>: unrecognised command `%.128s'",
		       argv[0]);
      free (argv);
    }

  return line;
}

/* Parse a <VirtualServer>...</VirtualServer> construct.  */
static struct vsd_file_line *read_virtualserver (struct vsd_vs_map *map,
						 struct vsd_file_line *vsconf,
						 int vs)
{
  struct vsd_file_line *line = vsconf;
  char *p, **argv;
  int argc;

  memset (&map->server[vs], 0, sizeof (struct vsd_vs));

  /* Get <arg> argument in <VirtualServer <arg>>.  First skip all spaces
     and then find the space preceeding the argument.  */
  for (p = line->line; *p && isspace (*p); p++)
    ;

  /* Check it is a VirtualServer decl.  */
  if (strncmp (p, "<VirtualServer ", 15) != 0)
    {
      vsd_log_error (VSD_LOG_ERROR,
		     "vsd.conf: read_virtualserver called on non-<VirtualServer>");
      return line;
    }
  for (; *p != ' '; p++)
    ;

  /* Extract argument.  */
  {
    char *q = map->server[vs].name;
    
    for (; *p && isspace (*p); p++)
      ;
    while (*p && *p != '>')
      *q++ = *p++;
    *q = '\0';
  }

  line = line->next;
  for (; line; line = line->next)
    {
      /* Skip white space.  */
      for (p = line->line; *p && isspace (*p); p++)
	;
      /* Skip comments and blank lines.  */
      if (*p == '\0' || *p == '#')
	continue;

      if (strncmp (p, "</VirtualServer>", 16) == 0)
	break;

      argv = vsd_argv_parse (' ', p, &argc);
      if (argc != 2)
	{
	  /* Invalid number of arguments.  */
	  vsd_log_error (VSD_LOG_ERROR,
			 "vsd.conf: <Default>: invalid number of args `%.128s'", p);
	  if (argv)
	    free (argv);
	  continue;
	}

      if (strcmp (argv[0], "IP") == 0)
	{
	  map->server[vs].ip = malloc (sizeof (struct in_addr));
	  vsd_ip_aton (argv[1], map->server[vs].ip);
	}
      else if (strcmp (argv[0], "StartUID") == 0)
	map->server[vs].startuid = strtol (argv[1], NULL, 10);
      else if (strcmp (argv[0], "Users") == 0)
	map->server[vs].users = strtol (argv[1], NULL, 10);
      else if (strcmp (argv[0], "Quota") == 0)
	map->server[vs].quota = strtol (argv[1], NULL, 10);
      else if (strcmp (argv[0], "Partition") == 0)
	map->server[vs].partition = strtol (argv[1], NULL, 10);
      else if (strcmp (argv[0], "Status") == 0)
	{
	  if (strcmp (argv[1], "active") == 0)
	    map->server[vs].status = S_ACTIVE;
	  else
	    map->server[vs].status = S_DISABLED;
	}
      else if (strcmp (argv[0], "PrimaryNS") == 0)
	strcpy (map->server[vs].ns1, argv[1]);
      else if (strcmp (argv[0], "SecondaryNS") == 0)
	strcpy (map->server[vs].ns2, argv[1]);
      else if (strcmp (argv[0], "AliasIP") == 0)
	{
	  /* FIXME: Add IPv6 support.  */
	  int x = ++map->server[vs].ipalias_total;
	  if (x == 1)
	    map->server[vs].ipalias = malloc (sizeof (struct in_addr));
	  else
	    map->server[vs].ipalias = realloc (map->server[vs].ipalias,
					       sizeof (struct in_addr) * x);
	  vsd_ip_aton (argv[1], &map->server[vs].ipalias[x - 1]);
	}
      else if (strcmp (argv[0], "BWLimit") == 0)
	map->server[vs].bwlimit = get_rate (argv[1]);
      else if (strcmp (argv[0], "BWBurst") == 0)
	map->server[vs].bwburst = get_rate (argv[1]);
      else
	vsd_log_error (VSD_LOG_ERROR,
		       "vsd.conf: <VirtualServer>: unrecognised command `%.128s'",
		       p);
      free (argv);
    }

  return line;
}

/* Populate a vsd_vs_map structure with the contents of /etc/vsd.conf.
   If an error occurs in parsing then return a string containing
   that error, otherwise return NULL.  */
struct vsd_vs_map *vsd_map_read (void)
{
  struct vsd_file_line *vsconf, *line;
  struct vsd_vs_map *map;
  char *p;

  if (vsd_load_file (&vsconf, "/", VSD_CONF))
    return NULL;

  map = (struct vsd_vs_map *) malloc (sizeof (struct vsd_vs_map));
  if (map == NULL)
    {
      vsd_free_file (vsconf);
      return NULL;
    }

  memset (map, 0, sizeof (struct vsd_vs_map));
  for (line = vsconf; line; line = line->next)
    {
      /* Skip white space.  */
      for (p = line->line; isspace (*p); p++)
	;
      /* Ignore comment lines and skip empty lines.  */
      if (*p == '#' || *p == '\0')
	continue;

      /* All configuration directives are contained within a <foo>...</foo>
	 construct.  */
      if (*p == '<')
	{
	  if (strncmp (p, "<Partition ", 11) == 0)
	    {
	      map->partitions ++;
	      map->partition =
		(struct vsd_disk *) xalloc (map->partition,
					    sizeof (struct vsd_disk)
					    * map->partitions);
	      if (map->partition == NULL)
		{
		  vsd_log_error (VSD_LOG_ERROR,
				 "vsd.conf: out of memory allocating vsd_disk");
		  vsd_free_file (vsconf);
		  return NULL;
		}
	      line = read_partition (map, line, map->partitions - 1);
	    }
	  else if (strncmp (p, "<Global>", 8) == 0)
	    line = read_global (map, line);
	  else if (strncmp (p, "<VirtualServer ", 15) == 0)
	    {
	      map->servers ++;
	      map->server =
		(struct vsd_vs *) xalloc (map->server,
					  sizeof (struct vsd_vs)
					  * map->servers);
	      if (map->server == NULL)
		{
		  vsd_log_error (VSD_LOG_ERROR,
				 "vsd.conf: out of memory allocating vsd_vs");
		  vsd_free_file (vsconf);
		  return NULL;
		}
	      line = read_virtualserver (map, line, map->servers - 1);
	    }
	  else
	    vsd_log_error (VSD_LOG_ERROR,
			   "vsd.conf: unrecognised line `%.128s'", p);
	}
      else
	vsd_log_error (VSD_LOG_ERROR,
		       "vsd.conf: unrecognised line `%.128s'", p);
    }

  vsd_free_file (vsconf);
  return map;
}

int vsd_map_save (struct vsd_vs_map *map)
{
  FILE *stream;
  int x;

  stream = fopen (VSD_CONF, "w");
  if (stream == NULL)
    return 1;

  fprintf (stream, "# /etc/vsd.conf - FreeVSD configuration file\n");
  fprintf (stream, "# This file is automatically generated\n\n");

  fprintf (stream, "<Global>\n");
  fprintf (stream, "UIDRange %d %d\n", map->global.minuid,
	   map->global.maxuid);
  fprintf (stream, "UIDAllocChunk %d\n", map->global.uidchunk);
  fprintf (stream, "PrimaryNS %s\n", map->global.ns1);
  fprintf (stream, "SecondaryNS %s\n", map->global.ns2);
  fprintf (stream, "ErrorLog %s\n", map->global.errorlog);
  fprintf (stream, "BWBurst %dbps\n", map->global.bwburst);
  fprintf (stream, "BWLimit %dbps\n", map->global.bwlimit);
  fprintf (stream, "</Global>\n\n");

  for (x = 0; x < map->partitions; x++)
    {
      fprintf (stream, "<Partition %d>\n", x);
      fprintf (stream, "# Mount point of disk device\n");
      fprintf (stream, "Mount %s\n", map->partition[x].mount);
      fprintf (stream, "# Maximum number of virtual servers\n");
      fprintf (stream, "MaxVS %d\n", map->partition[x].maxvs);
      fprintf (stream, "</Partition>\n\n");
    }

  fprintf (stream, "# Virtual server account details\n");
  for (x = 0; x < map->servers; x++)
    {
      char *status;
      int y;

      if (map->server[x].name[0] == '\0')
	continue; /* Skip deleted entries.  */
      
      fprintf (stream, "<VirtualServer %s>\n", map->server[x].name);
      fprintf (stream, "IP %s\n", vsd_ip_ntoa (map->server[x].ip));
      fprintf (stream, "StartUID %d\n", map->server[x].startuid);
      fprintf (stream, "Users %d\n", map->server[x].users);
      switch (map->server[x].status)
	{
	case S_ACTIVE: status = "active"; break;
	case S_DISABLED: status = "disabled"; break;
	default: status = "disabled"; break;
	}
      fprintf (stream, "Status %s\n", status); 
      fprintf (stream, "Quota %d\n", map->server[x].quota);
      fprintf (stream, "Partition %d\n", map->server[x].partition);

      if (*map->server[x].ns1)
	fprintf (stream, "PrimaryNS %s\n", map->server[x].ns1);
      if (*map->server[x].ns2)
	fprintf (stream, "SecondaryNS %s\n", map->server[x].ns2);

      for (y = 0; y < map->server[x].ipalias_total; y++)
	fprintf (stream, "AliasIP %s\n",
		 vsd_ip_ntoa (&map->server[x].ipalias[y]));

      if (map->server[x].bwburst)
	fprintf (stream, "BWBurst %dbps\n", map->server[x].bwburst);
      if (map->server[x].bwlimit)
	fprintf (stream, "BWLimit %dbps\n", map->server[x].bwlimit);
      fprintf (stream, "</VirtualServer>\n\n");
    }

  fclose (stream);
  return 0;
}

/* Free the internal memory structures created by vsd_map_read to hold
   the data of vsd.conf.  */
void vsd_map_free (struct vsd_vs_map *map)
{
  int x;

  if (map == NULL)
    return;

  for (x = 0; x < map->partitions; x++)
    free (map->partition[x].mount);
  free (map->partition);

  for (x = 0; x < map->servers; x++)
    {
      if (map->server[x].ipalias)
	free (map->server[x].ipalias);
      if (map->server[x].ip)
	free (map->server[x].ip);
    }

  if (map->servers >= 1)
    free (map->server);

  if (map->global.errorlog)
    free (map->global.errorlog);

  free (map);
}

/* Lock the vsd.conf map configuration file.  Return 0 on success.  */
int vsd_map_lock (struct vsd_vs_map *map)
{
  struct flock fl;

  map->lock_fd = open (VSD_CONF, O_RDWR, 0);
  if (map->lock_fd == -1)
    return -1;

  fl.l_type = F_WRLCK;
  fl.l_start = 0;
  fl.l_whence = SEEK_SET;
  fl.l_len = 0;

  return fcntl (map->lock_fd, F_SETLKW, &fl);
}

/* Unlock the vsd.conf map configuration file.  Return 0 on success.  */
int vsd_map_unlock (struct vsd_vs_map *map)
{
  struct flock fl;
  int s;

  fl.l_type = F_UNLCK;
  fl.l_start = 0;
  fl.l_whence = SEEK_SET;
  fl.l_len = 0;

  s = fcntl (map->lock_fd, F_SETLK, &fl);
  close (map->lock_fd);
  return s;
}

/* Choose a partition suitable for a virtual server.  If a partition can
   be found, return the partition number, otherwise return -1.  */
static int allocate_partition (struct vsd_vs_map *map)
{
  int x, *count;

  count = (int *) alloca (map->partitions * sizeof (int));
  memset (count, 0, map->partitions * sizeof (int));
  for (x = 0; x < map->servers; x++)
    count[map->server[x].partition]++;

  /* Find the first partition that has a spare space.  */
  for (x = 0; x < map->partitions; x++)
    if (count[x] < map->partition[x].maxvs)
      return x;

  /* All partitions are full.  */
  return -1;
}

/* Add a new virtual server to the vsd.conf configuration file.
   `server_name' is the name of the virtual server.
   `ip' is the IP address of the virtual server
   `start_uid' is the start of the uid range allocated to the server
   `num_users' is the number of uids to allocate to the server
   `quota' is the initial disk quota limit imposed on the server.
   `ns1' - if NULL, then use default nameservers otherwise use `ns1'
   `ns2' - if NULL, then use default nameservers otherwise use `ns2'
   Return 0 on success and 1 on failure.  */
int vsd_map_add_server (struct vsd_vs_map *map,
			const char *server_name, const char *ip,
			int start_uid, int num_users, int quota,
			const char *ns1, const char *ns2)
{
  int partition = allocate_partition (map);
  int x;
  
  if (partition == -1)
    {
      /* All partitions are full.  */
      errno = EMFILE;
      return 1;
    }

  map->servers ++;
  map->server = (struct vsd_vs *) xalloc (map->server,
					  sizeof (struct vsd_vs)
					  * map->servers);
  if (map->server == NULL)
    return 1;

  x = map->servers - 1;

  strcpy (map->server[x].name, server_name);
  map->server[x].ip = malloc (sizeof (struct in_addr));
  vsd_ip_aton (ip, map->server[x].ip);
  map->server[x].startuid = start_uid;
  map->server[x].users = num_users;
  map->server[x].quota = quota;
  map->server[x].partition = partition;
  map->server[x].status = S_ACTIVE;
  if (ns1)
    strcpy (map->server[x].ns1, ns1);
  else
    map->server[x].ns1[0] = '\0';
  if (ns2)
    strcpy (map->server[x].ns2, ns2);
  else
    map->server[x].ns2[0] = '\0';

  map->server[x].bwburst = 0;
  map->server[x].bwlimit = 0;
  map->server[x].ipalias_total = 0;
  map->server[x].ipalias = NULL;
  return 0;
}

/* Delete virtual server `server'.
   Return 0 on success and 1 if the server doesn't exist.  */
int vsd_map_delete_server (struct vsd_vs_map *map, const char *server)
{
  int x;
  int found = 0;

  /* Delete virtual server.  */
  for (x = 0; x < map->servers; x++)
    if (map->server[x].name[0] != '\0'
	&& strcmp (map->server[x].name, server) == 0)
      {
	map->server[x].name[0] = '\0';
	found = 1;
      }
 
  return 1;
}

/* Rename virtual server `oldvs' to `newvs'.  Return 0 on success,
   1 if `oldvs' doesn't exist, or 2 if `newvs' does exist.  */
int vsd_map_rename_server (struct vsd_vs_map *map, const char *oldvs,
			   const char *newvs)
{
  int x;

  /* See if `newvs' already exists.  */
  for (x = 0; x < map->servers; x++)
    if (map->server[x].name[0] != '\0'
	&& strcmp (map->server[x].name, newvs) == 0)
      return 2;

  for (x = 0; x < map->servers; x++)
    if (map->server[x].name[0] != '\0'
	&& strcmp (map->server[x].name, oldvs) == 0)
      {
        strcpy (map->server[x].name, newvs);
	return 0;
      }
 
  return 1;
}

/* Change the IP address of virtual server `vs' from `oldip' to `newip'.
   If appropriate then change IP aliases.  Return 0 on success, 1 if not
   found, 2 if `newip' is already in use.  */
int vsd_map_change_server_ip (struct vsd_vs_map *map, const char *vs,
			      const char *oldip, const char *newip)
{
  int x, y;
  struct in_addr o_ip, n_ip;

  /* FIXME: All this type casting is a joke. I suspect there is a much
     cleaner way.  */

  inet_aton (oldip, &o_ip);
  inet_aton (newip, &n_ip);

  /* See if `newip' already exists as a virtual server or an IP address.  */
  for (x = 0; x < map->servers; x++)
    {
      if (map->server[x].name[0] != '\0'
	  && (((struct in_addr *) map->server[x].ip)->s_addr == n_ip.s_addr))
	return 2;
      for (y = 0; y < map->server[x].ipalias_total; y++)
	if (((struct in_addr *) &map->server[x].ipalias[y])->s_addr == n_ip.s_addr)
	  return 2;
    }

  /* Change virtual server IP address (if appropriate).  */
  for (x = 0; x < map->servers; x++)
    {
      if (map->server[x].name[0] != '\0'
	  && strcmp (map->server[x].name, vs) == 0)
	{
	  if (((struct in_addr *) map->server[x].ip)->s_addr == o_ip.s_addr)
	    {
	      ((struct in_addr *) map->server[x].ip)->s_addr = n_ip.s_addr;
	      return 0;
	    }
	  else
	    for (y = 0; y < map->server[x].ipalias_total; y++)
	      if (((struct in_addr *) &map->server[x].ipalias[y])->s_addr == o_ip.s_addr)
		{
		  ((struct in_addr *) &map->server[x].ipalias[y])->s_addr = n_ip.s_addr;
		  return 0;
		}
	}
    }

  return 1;
}


/* Using either the `virtual_server' or `ip_addr' as an argument, return
   a pointer to a vsd_vs structure that corresponds to the given arguments.
   Return NULL if there is an error, or the virtual server or IP address
   does not exist.  */
struct vsd_vs *vsd_getmapent (struct vsd_vs_map *map,
			      const char *virtual_server,
			      const char *ip_addr)
{
  int x;

  if ((virtual_server == NULL || *virtual_server == '\0')
      && (ip_addr == NULL || *ip_addr == '\0'))
    {
      errno = EINVAL;
      return NULL;
    }

  if (virtual_server == NULL || *virtual_server == '\0')
    {
      int y;
      struct in_addr ipaddr;
      
      /* Search for an IP.  */
      inet_aton (ip_addr, &ipaddr);
      
      for (x = 0; x < map->servers; x++)
	{
	  if (map->server[x].name[0] == '\0')
	    continue;
	  
	  if (((struct in_addr *) map->server[x].ip)->s_addr == ipaddr.s_addr)
	    return &map->server[x];
	  for (y = 0; y < map->server[x].ipalias_total; y++)
	    if (((struct in_addr *) &map->server[x].ipalias[y])->s_addr == ipaddr.s_addr)
	      return &map->server[x];
	}
    }
  else
    {
      /* Search for a virtual server name.  */
      for (x = 0; x < map->servers; x++)
	if (strcmp (virtual_server, map->server[x].name) == 0)
	  return &map->server[x];
    }

  /* No match was found.  */
  errno = ENOENT;
  return NULL;
}

/* Look for a virtual server called `server_name' and if successful
   return a malloc'ed buffer that contains the full pathname to the
   virtual server root directory.  Return NULL if there is an error.  */
char *vsd_map_server_root (struct vsd_vs_map *map, const char *server_name)
{
  struct vsd_vs *vs = vsd_getmapent (map, server_name, NULL);
  static char *temp = NULL;
  int len;

  if (! vs)
    return NULL;

  len = (strlen (map->partition[vs->partition].mount)
	 + strlen (vs->name) + 6);

  /* Allocate some storage space for the pathname.  */
  if (temp == NULL)
    temp = (char *) malloc (len);
  else
    temp = (char *) realloc (temp, len);

  if (temp != NULL)
    sprintf (temp, "%s/vs/%s",
	     map->partition[vs->partition].mount, vs->name);

  return temp;
}

/* Change the active status of a virtual server called `server' to
   that as set in `status'.  */
int vsd_map_set_status (struct vsd_vs_map *map, const char *server,
			const char *status)
{
  struct vsd_vs *vs = vsd_getmapent (map, server, NULL);

  if (vs == NULL)
    return 1;

  if (strcmp (status, "active") == 0)
    vs->status = S_ACTIVE;
  else if (strcmp (status, "disabled") == 0)
    vs->status = S_DISABLED;

  return 0;
}

/* Return the minimum and maximum allowable UIDs available to a virtual
   server `server'.  */
void vsd_map_uid_range (struct vsd_vs_map *map, const char *server,
			int *uid_min, int *uid_max)
{
  struct vsd_vs *vs = vsd_getmapent (map, server, NULL);

  *uid_min = *uid_max = 0;
  if (vs != NULL)
    {
      *uid_min = vs->startuid;
      *uid_max = *uid_min + vs->users;
    }
}

/* Return -1 on error or a start UID on success.  */
int vsd_alloc_uid (struct vsd_vs_map *map, int uid_size)
{
  int start_uid, x, found, range_size;
  char *uid_range;

  if (uid_size < 0)
    {
      errno = EINVAL;
      return -1;
    }

  /* No server map so assume we are the first.  */
  if (map == NULL)
    return map->global.minuid;

  range_size = ((map->global.maxuid - map->global.minuid)
		/ map->global.uidchunk);
  uid_range = (char *) alloca (range_size + 1);
  if (uid_range == NULL)
    return -1;

  memset (uid_range, 0, range_size);

  /* Round up uid_size to the next chunk size.  */
  uid_size = ((uid_size + map->global.uidchunk) / map->global.uidchunk);

  /* Create an map of allocated uid chunks.  */
  for (x = 0; x < map->servers; x++)
    memset (uid_range + ((map->server[x].startuid - map->global.minuid)
			 / map->global.uidchunk),
	    1, map->server[x].users / map->global.uidchunk);

  found = 0;
  start_uid = 0;
  while (! found && start_uid <= range_size)
    {
      /* Search for an unallocated chunk.  */
      if (uid_range[start_uid] == 0)
	{
	  /* Found one. Is it large enough ?  */
	  found = 1;
	  for (x = 0; x <= uid_size - 1; x++)
	    if (uid_range[start_uid + x] == 1)
	      {
		/* Free chunk is not large enough.  */
		found = 0;
		break;
	      }
	}

      start_uid ++;
    }

  return (! found) ? -1 : (((start_uid - 1) * map->global.uidchunk)
			   + map->global.minuid);
}

/* Add an IP alias to an existing virtual server entry.
   Return 0 on success
   Return 1 if `virtual_server' does not exist.
   Return 2 if `ipalias' is already defined by another virtual server.
   Return 3 if `ipalias' is already defined by `virtual_server'.
   Return 4 if memory allocation failed.  */
int vsd_map_add_ipalias (struct vsd_vs_map *map,
			 const char *virtual_server, const char *ipalias)
{
  struct vsd_vs *vs, *vs1;
  int x;

  vs = vsd_getmapent (map, virtual_server, NULL);
  if (vs == NULL)
    return 1;

  vs1 = vsd_getmapent (map, NULL, ipalias);
  if (vs1 != NULL && vs != vs1)
    return 2;

  if (vs == vs1)
    return 3;

  /* FIXME: Add IPv6 support.  */
  x = ++vs->ipalias_total;
  if (x == 1)
    vs->ipalias = malloc (sizeof (struct in_addr));
  else
    vs->ipalias = realloc (vs->ipalias, sizeof (struct in_addr) * x);
  if (vs->ipalias == NULL)
    return 4;

  vsd_ip_aton (ipalias, &vs->ipalias[x - 1]);

  return 0;
}

/* Add an IP alias to an existing virtual server entry.
   Return 0 on success
   Return 1 if `virtual_server' does not exist.
   Return 2 if `ipalias' does not exist.
   Return 3 if `ipalias' does not belong to `virtual_server'.
   Return 4 if memory allocation failed.  */
int vsd_map_delete_ipalias (struct vsd_vs_map *map,
			    const char *virtual_server, const char *ipalias)
{
  struct vsd_vs *vs, *vs1;
  char *ip;

  vs = vsd_getmapent (map, virtual_server, NULL);
  if (vs == NULL)
    return 1;

  vs1 = vsd_getmapent (map, NULL, ipalias);
  if (vs1 == NULL)
    return 2;

  if (vs != vs1)
    return 3;

  /* Quick case - 1 IP alias.  */
  if (vs->ipalias_total == 1)
    {
      free (vs->ipalias);
      vs->ipalias = NULL;
      vs->ipalias_total = 0;
    }
  else
    {
      int x, y;
      void **ipaliases;

      /* Re-construct IP alias array without `ipalias'.  */
      ipaliases = malloc (sizeof (struct in_addr) * (vs->ipalias_total - 1));
      if (ipaliases == NULL)
	return 4;
      for (x = y = 0; x < vs->ipalias_total; x++)
	{
	  ip = vsd_ip_ntoa (&vs->ipalias[x]);
	  if (strcmp (ip, ipalias) != 0)
	    ipaliases[y++] = vs->ipalias[x];
	}
      vs->ipalias_total --;
      free (vs->ipalias);
      vs->ipalias = ipaliases;
    }

  return 0;
}


#ifdef TEST
int main (void)
{
  struct vsd_vs_map *map;
  struct vsd_vs *vs;
  int x;

#if 1
  for (x = 0; x < 1000000; x++)
    {
      map = vsd_map_read ();
      vsd_map_free (map);
    }
#else
  map = vsd_map_read ();
  vs = vsd_getmapent (map, NULL, "10.1.2.3");
  vsd_map_free (map);
#endif

  return 0;
}
#endif
