/* Declarations for functions defined by libvsd.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#ifndef LIBVSD_H
#define LIBVSD_H 1

#include <stdio.h>
#include <sys/types.h>
#include <grp.h>
#include <pwd.h>

#ifdef __cplusplus
extern "C" {
#endif

/* C library compatibility functions.  It would be much better if we
   didn't declare these in here.  */
#ifndef HAVE_FGETPWENT
extern struct passwd *fgetpwent (FILE *stream);
#endif
#ifndef HAVE_FGETGRENT
extern struct group *fgetgrent (FILE *stream);
#endif
#ifndef HAVE_STPCPY
extern char *stpcpy (char *s1, const char *s2);
#endif

/* argv.c */
extern char **vsd_argv_add (char **argv, int *argc, char *line);
extern void vsd_argv_free (char **argv, int argc);
extern char **vsd_argv_parse (const char sep, char *arguments,
                              int *argvec_count);

/* dyn-string.c  */
typedef struct dstring
{
  /* Memory allocated for the string.  */
  int buffer_size;
  /* Length of the stored string.  */
  int length;
  /* Pointer to the start of the string.  */
  char *s;
} *dstring_t;

extern dstring_t dstring_new (int size);
extern void dstring_free (dstring_t ds);
extern void dstring_append (dstring_t ds, const char *s);
extern void dstring_fmt_append (dstring_t ds, const char *fmt, ...);
extern char *dstring_yield (dstring_t ds);
extern char *dstring_ptr (dstring_t ds);

/* file.c */
struct vsd_file_line
{
  char *line;
  struct vsd_file_line *prev;
  struct vsd_file_line *next;
};

extern int vsd_load_file (struct vsd_file_line **list, const char *server,
			  const char *filename);
extern int vsd_save_file (struct vsd_file_line *head, const char *server,
			  const char *filename);
extern void vsd_free_file (struct vsd_file_line *head);
extern int vsd_file_insert_line (struct vsd_file_line *head, const char *line);
extern struct vsd_file_line *vsd_file_delete_line (struct vsd_file_line *head,
						   struct vsd_file_line *line);
extern struct vsd_file_line *vsd_file_find_line (struct vsd_file_line *head,
						 const char *line);
extern const char *vsd_file_skip_spaces (const char *s);
extern struct vsd_file_line *vsd_file_tail (struct vsd_file_line *head);
extern struct vsd_file_line *vsd_file_find_entry (struct vsd_file_line *head,
						  const char *entry);
extern struct vsd_file_line *vsd_file_walk (struct vsd_file_line *ent);
extern int vsd_file_str_replace (struct vsd_file_line *line,
				 const char *oldstr, const char *newstr);
extern struct vsd_file_line *vsd_file_str_search (struct vsd_file_line *head,
						  const char *string);
extern void vsd_file_str_search_replace (struct vsd_file_line *head,
					 const char *old_str,
					 const char *new_str);

/* group.c  */
extern int vsd_group_load (struct vsd_file_line **list,
			    const char *server_root);
extern int vsd_group_save (struct vsd_file_line *list,
			    const char *server_root);
extern void vsd_group_free (struct vsd_file_line *list);
extern int vsd_groupadd (const char *server_root, const char *group_name,
			 gid_t *gid);
extern int vsd_groupdel (const char *server_root, const char *group_name);
extern int vsd_groupmod (const char *server_root, const char *group_name,
			 gid_t gid, const char *new_group);
extern struct vsd_file_line *vsd_groupgetnam (struct vsd_file_line *file,
					      const char *group_name);
extern struct group *vsd_group_decode (struct vsd_file_line *line,
				       struct group *gr);
extern void vsd_group_struct_free (struct group *gr);
extern void vsd_group_add_user (struct vsd_file_line *line,
				const char *user_name);
extern void vsd_group_del_user (struct vsd_file_line *line,
				const char *user_name);
extern void vsd_group_remove_user (struct vsd_file_line *head,
				   const char *user_name);
extern void vsd_group_rename_user (struct vsd_file_line *head,
				   const char *old_user, const char *new_user);

/* groupent.c  */
extern struct group *vsd_getgrgid (FILE *stream, gid_t gid);
extern struct group *vsd_getgrnam (FILE *stream, const char *name);
extern void vsd_getgrmem (FILE *stream, const char *user, char **buf);
extern FILE *vsd_opengrp (const char *virtual_server);
extern void vsd_closegrp (FILE *stream);

/* io.c */
char *io_getline (const char delim, const char **ptr);

/* priv.c */
extern int vsd_priv_load (struct vsd_file_line **list, const char *server);
extern int vsd_priv_save (struct vsd_file_line *list, const char *server);
extern void vsd_priv_free (struct vsd_file_line *list);
extern int vsd_check_privilege (struct vsd_file_line *priv_file,
				const char *username, const char *privilege);
extern int vsd_priv_add_member (struct vsd_file_line *priv_file,
				const char *priviledge, const char *username);
extern int vsd_priv_delete_member (struct vsd_file_line *priv_file,
				   const char *priviledge,
				   const char *username);
extern char **vsd_priv_get_members (struct vsd_file_line *priv_file,
				    const char *username, int *membcount);
extern char **vsd_priv_get_group_members (const char *server,
					  const char *username,
					  int *membcount);
extern char **vsd_priv_list_privs (struct vsd_file_line *head, int *argcount);
extern int vsd_priv_access (const char *username, const char *priv);
extern int vsd_priv_delete_user (const char *server_root, const char *user);
extern int vsd_priv_rename_user (const char *server_root,
				 const char *old_user, const char *new_user);

/* process.c */
extern int *vsd_enum_procs (const char *vs);
extern int *vsd_enum_procs_vs (const char *vs);
extern pid_t vsd_getppid (pid_t p);

/* user.c */
extern int vsd_get_uidgid (const char *server_root,
			   uid_t *uid_min, uid_t *uid_max,
			   gid_t *gid_min, gid_t *gid_max);
extern int vsd_valid_uid (uid_t uid);
extern int vsd_valid_gid (gid_t gid);
extern int vsd_validate_user_uid (uid_t uid);
extern int vsd_validate_user_name (const char *username);
extern int vsd_admin_user (uid_t uid);
extern int vsd_check_path (const char *path, int check_file,
			   int access_rights);
extern int vsd_owner_pid (pid_t pid);
extern const char *vsd_valid_username (const char *user);
extern const char *vsd_valid_vsname (const char *vs);
extern const char *vsd_check_and_make_path (const char *server_root,
					    char *path, int dir);
extern const char *vsd_check_ip_address (const char *ip);

/* map.c */
struct vsd_disk
{
  char *mount;  /* Mount point of the disk device.  */
  int maxvs;  /* Max number of virtual servers that can be created.  */
};

struct vsd_vs
{
  char name[32]; /* Name of the virtual server.  */
  void *ip; /* IP address.  */
  int startuid; /* Starting UID (lowest uid usable by the virtual server.  */
  int users; /* Number of users available to the virtual server.  */
  int status; /* Is the virtual server active or disabled.  */
  int quota; /* Disk quota available (in megabytes).  */
  int partition; /* Partition number that the virtual server is on.  */
  char ns1[32];  /* Domain name of the primary name server.  */
  char ns2[32];  /* Domain name of the secondary name server.  */
  unsigned int bwburst;  /* Bandwidth limit - burstable max.  */
  unsigned int bwlimit;  /* Bandwidth limit.  */
  int ipalias_total; /* Number of IP aliases.  */
  void **ipalias; /* IP aliases.  */
};

struct vsd_global
{
  int minuid; /* Lowest UID that can be allocated to a VS.  */
  int maxuid; /* Maximum UID that can be allocated to a VS.  */
  int uidchunk; /* VSes have UIDs allocated in chunks of this size.  */
  char ns1[32];  /* Domain name of the primary name server.  */
  char ns2[32];  /* Domain name of the secondary name server.  */
  char *errorlog;  /* Path to a file for recording all errors with vsd.  */
  unsigned int bwburst;  /* Bandwidth limit - burstable max.  */
  unsigned int bwlimit;  /* Bandwidth limit.  */
};

struct vsd_vs_map
{
  int lock_fd;
  int partitions;  /* Number of partitions defined.  */
  int servers;  /* Number of virtual servers defined.  */
  struct vsd_global global;
  struct vsd_disk *partition;
  struct vsd_vs *server;
};

#define S_ACTIVE 0
#define S_DISABLED 1

extern struct vsd_vs_map *vsd_map_read (void);
extern int vsd_map_save (struct vsd_vs_map *map);
extern void vsd_map_free (struct vsd_vs_map *map);
extern int vsd_map_lock (struct vsd_vs_map *map);
extern int vsd_map_unlock (struct vsd_vs_map *map);

extern int vsd_map_add_server (struct vsd_vs_map *map,
			       const char *server_name, const char *ip,
			       int start_uid, int num_users, int quota,
			       const char *ns1, const char *ns2);
extern int vsd_map_delete_server (struct vsd_vs_map *map, const char *server);
extern int vsd_map_rename_server (struct vsd_vs_map *map, const char *oldvs,
				  const char *newvs);
extern struct vsd_vs *vsd_getmapent (struct vsd_vs_map *map,
				     const char *virtual_server,
				     const char *ip_addr);
extern char *vsd_map_server_root (struct vsd_vs_map *map,
					const char *server_name);
extern int vsd_alloc_uid (struct vsd_vs_map *srvmap, int uid_size);
extern int vsd_map_set_status (struct vsd_vs_map *map, const char *server,
			       const char *status);
extern void vsd_map_uid_range (struct vsd_vs_map *map, const char *server,
			       int *uid_min, int *uid_max);
extern int vsd_map_change_server_ip (struct vsd_vs_map *map, const char *vs,
				     const char *oldip, const char *newip);
extern char *vsd_ip_ntoa (void *ip);
extern int vsd_ip_aton (const char *ip, void *inp);

extern int vsd_map_add_ipalias (struct vsd_vs_map *map,
				const char *virtual_server,
				const char *ipalias);
extern int vsd_map_delete_ipalias (struct vsd_vs_map *map, 
				   const char *virtual_server,
				   const char *ipalias);

/* passwd.c  */
extern int vsd_passwd_load (struct vsd_file_line **list,
			    const char *server_root);
extern int vsd_passwd_save (struct vsd_file_line *list,
			    const char *server_root);
extern void vsd_passwd_free (struct vsd_file_line *list);
extern int vsd_userdel (const char *server_root, const char *user_name);
extern int vsd_useradd (const char *server_root, const char *user_name,
			const char *homedir, const char *initial_group,
			const char *additional_groups, const char *shell,
			uid_t *uid, const char *gecos);
extern struct passwd *vsd_passwd_decode (struct vsd_file_line *line,
					 struct passwd *pw);
extern int vsd_usermod (const char *server_root, const char *user_name,
			const char *login_name, const char *homedir,
			const char *initial_group,
			const char *additional_groups, const char *shell,
			uid_t uid, const char *gecos);
extern int vsd_getpwnam (struct vsd_file_line *file, const char *name,
			 struct passwd *pw);
extern int vsd_getpwuid (struct vsd_file_line *file, uid_t uid,
			 struct passwd *pw);
extern struct vsd_file_line *vsd_getpwent (struct vsd_file_line *ent,
					   struct passwd *pw);

/* chpasswd.c */
extern int vsd_chpass (const char *server_root, const char *user_name,
		       int admin, const char *old_password,
		       const char *new_password);

/* server.c */
extern int vsd_getvirtualservername (char *buffer, int size);

/* string.c */
extern size_t string_lenc (const char *s, char term);
extern char *string_cpyc (char *d, const char *s, char term);
extern const char *string_member (const char *list, const char *token,
				  const char sep);
extern int string_member_delete (char *list, const char *token,
				 const char sep);
extern int string_member_add (char **list, char *search_from,
			      const char *token, const char sep);

/* log.c  */
#define VSD_LOG_INFO 0
#define VSD_LOG_DEBUG 1
#define VSD_LOG_ERROR 2
#define VSD_LOG_PROTOCOL 3
extern void vsd_log_error (int severity, const char *fmt, ...);

/* quota.c */
extern void vsd_quota_set (const char *server_root, int newquota);
extern unsigned int vsd_quota_get (const char *server_root);

#ifdef __cplusplus
}
#endif

#endif
