/* Group file manipulation.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* Group file manipulation functions where the file is not necessarily
   /etc/group.  */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <grp.h>

#include "libvsd.h"

struct group *vsd_getgrgid (FILE *stream, gid_t gid)
{
  struct group *grp;

  if (stream == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  /* Return to the file beginning.  */
  fseek (stream, 0L, SEEK_SET);

  /* Search for the group.  */
  while ((grp = fgetgrent (stream)))
    {
      if (grp->gr_gid == gid)
	return grp;
    }

  /* Not found.  */
  return NULL;
}

struct group *vsd_getgrnam (FILE *stream, const char *name)
{
  struct group *grp;

  if (stream == NULL || name == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  fseek (stream, 0L, SEEK_SET);

  while ((grp = fgetgrent (stream)))
    {
      if (strcmp (grp->gr_name, name) == 0)
	return grp;
    }

  return NULL;
}

static int on_list (char **list, const char *member)
{
  while (*list)
    {
      if (strcmp (*list, member) == 0)
	return 1;

      list++;
    }

  return 0;
}

/* Return a comma-separated list of groups that `user' is a member of.
   `buf' contains the resultant list in a malloc'd buffer.  */
void vsd_getgrmem (FILE *stream, const char *user, char **buf)
{
  struct group *grp;
  char *list = NULL;

  /* Start at the beginning.  */
  fseek (stream, 0L, SEEK_SET);

  while ((grp = fgetgrent (stream)))
    {
      /* Look at each member of a group.  */
      if (on_list (grp->gr_mem, user))
	{
	  /* User is a member of group.  */
	  if (list == NULL)
	    {
	      list = (char *)malloc (strlen (grp->gr_name) + 1);
	      strcpy (list, grp->gr_name);
	    }
	  else
	    {
	      list = (char *)realloc (list,
				      strlen (list)
				      + strlen (grp->gr_name)
				      + 3);
	      strcat (list, ",");
	      strcat (list, grp->gr_name);
	    }
	}
    }
  *buf = list;
}

/* If `virtual_server' is NULL then open the host server's
   /etc/group file.  */
FILE *vsd_opengrp (const char *server_root)
{
  char filename[256];

  if (server_root == NULL)
    strcpy (filename, "/etc/group");
  else
    sprintf (filename, "%s/etc/group", server_root);

  return fopen (filename, "r");
}

void vsd_closegrp (FILE *stream)
{
  fclose (stream);
}
