/* Load a text file into memory and store in a double-linked list.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "libvsd.h"

/* Load `filename' into memory and store in a doubly-linked list, `list'.
   If `server' is not NULL, then `filename' is to be loaded from the
   virtual server.

   Return 0 on success.
   Return -1 on error and errno will be set as following:
       ESRCH  - The home directory for the virtual server `server' could
                not be found. 
       ENOMEM - Memory allocation failure.
       ENOENT - File `filename' does not exist.  */
int vsd_load_file (struct vsd_file_line **list, const char *server_root,
		   const char *filename)
{
  struct vsd_file_line *prev_entry, *first_entry, *entry;
  FILE *stream;
  char *temp;

  if (list == NULL || server_root == NULL || filename == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  /* Filename temporary.  */
  temp = (char *) malloc (strlen (server_root) + strlen (filename) + 2);
  if (temp == NULL)
    return -1;

  strcpy (temp, server_root);
  strcat (temp, filename);

  /* Attempt to open file.  */
  stream = fopen (temp, "r");
  if (stream == NULL)
    return -1;

  /* Free filename temporary.  */
  free (temp);

  /* Some temporary storage for loading the file.  */
  temp = (char *) malloc (16384);
  if (temp == NULL)
    goto nomem;

  /* Create a temporary NULL header node.  */
  first_entry = (struct vsd_file_line*) malloc (sizeof (struct vsd_file_line));
  if (first_entry == NULL)
    goto nomem;
  first_entry->line = NULL;
  first_entry->next = NULL;

  prev_entry = first_entry;
  /* FIXME: Get rid of this stupid buffer.  */
  while (fgets (temp, 16383, stream))
    {
      int linelen = strlen (temp);

      entry = (struct vsd_file_line *) malloc (sizeof (struct vsd_file_line));
      if (entry == NULL)
	goto nomem;

      entry->line = (char *) malloc (linelen);
      if (entry->line == NULL)
	goto nomem;

      if (linelen > 1)
	strncpy (entry->line, temp, linelen - 1);
      entry->line[linelen - 1] = '\0';
      entry->next = NULL;
      entry->prev = prev_entry;

      prev_entry->next = entry;
      prev_entry = entry;
    }
      
  fclose (stream);

  /* Unlink the NULL head node.  */
  if (first_entry->next != NULL)
    first_entry->next->prev = NULL;
  *list = first_entry->next;
  free (first_entry);

  /* Free the file temporary buffer.  */
  free (temp);
  return 0;

nomem:
  if (stream != NULL)
    fclose (stream);

  /* Free memory allocated to the file.  */
  if (first_entry)
    vsd_free_file (first_entry);

  /* Free the file temporary buffer.  */
  free (temp);
  errno = ENOMEM;
  return -1;
}

/* Save the file stored in `head' to disk with the name `filename'.
   If `server' is not NULL, then `filename' is to be saved to the
   virtual server.

   Return 0 on success.
   Return -1 on error and errno will be set as following:
       ESRCH  - The home directory for the virtual server `server' could
                not be found. 
       ENOMEM - Memory allocation failure.
       ENOENT - File `filename' does not exist.  */
int vsd_save_file (struct vsd_file_line *head, const char *server_root,
		   const char *filename)
{
  struct vsd_file_line *entry;
  FILE *stream;
  char *temp;

  if (head == NULL || server_root == NULL || filename == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  temp = (char *) malloc (strlen (server_root) + strlen (filename) + 2);
  strcpy (temp, server_root);
  strcat (temp, filename);

  stream = fopen (temp, "w");
  if (stream == NULL)
    {
      free (temp);
      return -1;
    }

  /* Write file out, line at a time.  */
  entry = head;
  while (entry)
    {
      if (entry->line != NULL)
	fprintf (stream, "%s\n", entry->line);
      entry = entry->next;
    }

  fclose (stream);
  free (temp);

  return 0;
}

/* Free memory allocated to the file structure initially by vsd_load_file
   but also by other functions in this source file.  */
void vsd_free_file (struct vsd_file_line *head)
{
  while (head)
    {
      struct vsd_file_line *entry = head->next;

      /* Line could be NULL if there was a memory allocation error
	 while loading the file.  */
      if (head->line != NULL)
	free (head->line);

      free (head);
      head = entry;
    }
}

/* Return a pointer to the last line in the file.  */
struct vsd_file_line *vsd_file_tail (struct vsd_file_line *head)
{
  if (head == NULL)
    return NULL;

  while (head->next)
    head = head->next;

  return head;
}

/* Insert the zero-terminated string `line' as the next node
   after `head'. Return 0 on success and -1 on error.  */
int vsd_file_insert_line (struct vsd_file_line *head, const char *line)
{
  struct vsd_file_line *entry;

  if (line == NULL)
    {
      errno = EINVAL;
      return -1;
    }

  entry = (struct vsd_file_line *) malloc (sizeof (struct vsd_file_line));
  if (entry == NULL)
    return -1;

  entry->line = (char *) malloc (strlen (line) + 1);
  if (entry->line == NULL)
    {
      free (entry);
      return -1;
    }

  strcpy (entry->line, line);
  entry->next = head->next;
  entry->prev = head;

  head->next = entry;

  return 0;
}

/* Unlink `line' from the linked list and free memory allocated to it.
   Return 0 on success, -1 on error.  */
struct vsd_file_line *vsd_file_delete_line (struct vsd_file_line *head,
					    struct vsd_file_line *line)
{
  struct vsd_file_line *prev, *next;

  if (head == NULL || line == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  if (line->prev == NULL)
    {
      /* `line' is the first entry on the list.  */
      prev = head->next;
      free (head);

      return prev;
    }

  /* Unlink `line' from the list.  */
  prev = line->prev;
  next = line->next;

  prev->next = next;
  if (next)
    next->prev = prev;
  
  free (line->line);
  free (line);

  return head;
}

/* Search the file `head' looking for an exact match to line `line'.
   If a match is found, return a pointer to the line. Otherwise
   return NULL.  */
struct vsd_file_line *vsd_file_find_line (struct vsd_file_line *head,
					  const char *line)
{
  if (line != NULL)
    for (; head; head = head->next)
      if (strcmp (head->line, line) == 0)
	return head;

  return NULL;
}

/* Skip spaces in a line.  Return pointer to the first non-whitespace
   character.  */
const char *vsd_file_skip_spaces (const char *s)
{
  if (s != NULL)
    while (isspace (*s))
      s++;

  return s;
}

/* For files that are in the format:
       <token>:<stuff>...
       <token>:<stuff>...

   then this function can be used to return a pointer to the line where
   `entry' matches `token'.

   Return NULL and EINVAL if the parameters are invalid.
   Return NULL and ENOENT if `entry' cannot be matched to a token.  */
struct vsd_file_line *vsd_file_find_entry (struct vsd_file_line *head,
					   const char *entry)
{
  struct vsd_file_line *line;

  if (head == NULL || entry == NULL || (entry && *entry == '\0'))
    {
      errno = EINVAL;
      return NULL;
    }

  for (line = head; line; line = line->next)
    {
      char *p;

      /* Skip blank lines.  */
      if (line->line[0] == '\0')
	continue;

      p = line->line;
      /* Skip initial whitespace.  */
      while (isspace (*p))
	p++;

      /* Skip comment lines.  */
      if (*p == '#')
	continue;

      /* Is this the token we are looking for ? */
      if (strncmp (entry, p, strlen (entry)) == 0
	  && p[strlen (entry)] == ':')
	return line;
    }

  /* Token was not found.  */
  errno = ENOENT;
  return NULL;
}

/* Walk though a file a line at a time, return a pointer to the next
   line in the file.  Return NULL if there are no more lines.  */
struct vsd_file_line *vsd_file_walk (struct vsd_file_line *ent)
{
  if (ent == NULL)
    return NULL;

  return ent->next;
}

/* Replace all occurances of `oldstr' with `newstr' on the line `line'.
   If the match and replace is successful, return 0.  Otherwise return -1.  */
int vsd_file_str_replace (struct vsd_file_line *line, const char *oldstr,
			  const char *newstr)
{
  char *newline, *ptr;

  /* Find the old string.  */
  ptr = strstr (line->line, oldstr);
  if (ptr == NULL)
    return -1;

  /* Allocate space for the new line.  */
  newline = (char *) malloc (strlen (line->line) + 1
			     + strlen (newstr)
			     - strlen (oldstr));

  /* Copy the characters leading upto the replaced string.  */
  strncpy (newline, line->line, ptr - line->line);
  newline[ptr - line->line] = '\0';

  /* Copy the replaced string.  */
  strcat (newline, newstr);
  /* Copy the characters after the replaced string.  */
  strncat (newline, ptr + strlen (oldstr),
	   strlen (line->line) - strlen (oldstr));
  free (line->line);
  line->line = newline;

  return 0;
}

/* Search for a given string forward in the file `head'.  If found, return
   a pointer to the line.  If not found, return NULL.  */
struct vsd_file_line *vsd_file_str_search (struct vsd_file_line *head,
					   const char *string)
{
  while (head != NULL)
    {
      char *result = strstr (head->line, string);

      if (result != NULL)
	return head; /* Found a match.  */

      head = head->next;
    }

  return NULL;
}

/* Replace all occurances of string `oldstr' with string `newstr' in the
   file `head'.  */
void vsd_file_str_search_replace (struct vsd_file_line *head,
				  const char *oldstr, const char *newstr)
{
  while ((head = vsd_file_str_search (head, oldstr)))
    {
      while (! vsd_file_str_replace (head, oldstr, newstr))
	;
      head = head->next;
    }
}


#ifdef TEST
int main (int argc, char *argv[])
{
  struct vsd_file_line *head, *ent;

  /* Create some data.  */
  head = (struct vsd_file_line*) malloc (sizeof (struct vsd_file_line));
  head->line = strdup ("this is the first line");
  head->next = NULL;
  vsd_file_insert_line (vsd_file_tail (head), "this is the this second line");
  vsd_file_insert_line (vsd_file_tail (head), "this is the third line");
  vsd_file_insert_line (vsd_file_tail (head), "this is the fourth line");
  vsd_file_insert_line (vsd_file_tail (head), "this is the fifth line");
  vsd_file_insert_line (vsd_file_tail (head), "blah blah blah blah");
  vsd_file_insert_line (vsd_file_tail (head), "blafblah blah blaf");

  ent = vsd_file_str_search (head, "fourth");
  printf ("search for 'fourth': ent->line = '%s'\n", ent->line);

  ent = vsd_file_str_search (head, "first");
  printf ("search for 'first': ent->line = '%s'\n", ent->line);

  ent = vsd_file_str_search (head, "second");
  printf ("search for 'second': ent->line = '%s'\n", ent->line);

  ent = vsd_file_str_search (head, "blah");
  printf ("search for 'blah': ent->line = '%s'\n", ent->line);

  ent = vsd_file_str_search (head, "third");
  printf ("search for 'third': ent->line = '%s'\n", ent->line);

  ent = vsd_file_str_search (head, "blaf");
  printf ("search for 'blaf': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "blaf", "odd"))
    printf ("replace `blaf' with `odd': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "blah", "foobar"))
    printf ("replace `blah' with `foobar': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "blah", "foobar"))
    printf ("replace `blah' with `foobar': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "blaf", "z"))
    printf ("replace `blaf' with `z': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "foobar", ""))
    printf ("replace `foobar' with `': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "z", ""))
    printf ("replace `z' with `': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, " ", ""))
    printf ("replace ` ' with `': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, " ", ""))
    printf ("replace ` ' with `': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "odd", ""))
    printf ("replace `odd' with `': ent->line = '%s'\n", ent->line);

  if (! vsd_file_str_replace (ent, "foobar", ""))
    printf ("replace `foobar' with `': ent->line = '%s'\n", ent->line);

  vsd_file_str_search_replace (head, "this", "that");
  printf ("line1: '%s'\nline2: '%s'\nline3: '%s'\nline4: '%s'\n",
	  head->line, head->next->line, head->next->next->line,
	  head->next->next->next->line);
  return 0;
}
#endif


