/* Get password/group entries from a stream.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* Compatibility function written with FreeBSD in mind.  */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef HAVE_FGETPWENT
#include <pwd.h>
#endif
#ifndef HAVE_FGETGRENT
#include <grp.h>
#endif
#include "libvsd.h"

#if !defined (HAVE_FGETPWENT) || !defined (HAVE_FGETGRENT)
static char buf[1024];

static char *cpy (char **ptr)
{
  char *p = *ptr, *q = *ptr;

  while (*q && *q != '\0' && *q != '\n' && *q != ':')
    q++;

  if (*q == ':')
    *q = '\0';
  *ptr = q + 1;

  return p;
}

#ifndef HAVE_FGETPWENT
struct passwd *fgetpwent (FILE *stream)
{
  static struct passwd pw;
  char *p;

  do
  {
    if (fgets (buf, sizeof (buf) - 1, stream) == 0)
      return NULL;
    p = strchr (buf, '\n');
  } while (buf[0] == '#' && ! p);

  /* Remove terminating new line.  */
  *p = '\0';
  p = buf;
  pw.pw_name = cpy (&p);
  pw.pw_passwd = cpy (&p);
  pw.pw_uid = atoi (cpy (&p));
  pw.pw_gid = atoi (cpy (&p));
  pw.pw_gecos = cpy (&p);
  pw.pw_dir = cpy (&p);
  pw.pw_shell = cpy (&p);

  return &pw;
}
#endif /* HAVE_FGETPWENT */

#ifndef HAVE_FGETGRENT
struct group *fgetgrent (FILE *stream)
{
  static char **bufv = NULL;
  static int bufc = 0;
  static struct group gr;
  char *p;

  /* Free memory allocated to an earlier group vector.  */
  if (bufv != NULL)
    free (bufv);

  do
    {
      if (fgets (buf, sizeof (buf) - 1, stream) == 0)
        return NULL;
      p = strchr (buf, '\n');
    } while (buf[0] == '#' && ! p);

  /* Remove terminating new line.  */
  *p = '\0';
  p = buf;
  gr.gr_name = cpy (&p);
  (void) cpy (&p);
  gr.gr_gid = atoi (cpy (&p));
  
  bufv = vsd_argv_parse (':', p, &bufc);
  gr.gr_mem = bufv;

  return &gr;
}
#endif /* HAVE_FGETPWENT */

#ifdef TEST
int main (void)
{
  FILE *stream;
  struct passwd *pw;
  struct group *gr;
  int x;
      
  stream = fopen ("/etc/passwd", "r");
  while ((pw = fgetpwent (stream)))
    printf ("name='%s', passwd='%s', uid=%d, gid=%d, gecos='%s', dir='%s', shell='%s'\n",
            pw->pw_name, pw->pw_passwd, pw->pw_uid, pw->pw_gid,
            pw->pw_gecos, pw->pw_dir, pw->pw_shell);
  fclose (stream);

  stream = fopen ("/etc/group", "r");
  while ((gr = fgetgrent (stream)))
    {
      printf ("name='%s', gid='%d': groups = ", gr->gr_name, gr->gr_gid);
      for (x = 0; gr->gr_mem[x]; x++)
        printf ("%s,", gr->gr_mem[x]);
      printf ("\n");
    } 
  fclose (stream);

  return 0;
}
#endif

#endif /* !defined */
