/* Change password in user password file.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* To save us having to link against libcrypt, we intentionally keep
   this function in a seperate file.  */

/* We require these to import the crypt and strdup function declarations.  */
#include <errno.h>
#define __USE_XOPEN
#define __USE_XOPEN_EXTENDED
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include "libvsd.h"

static int i64c (int i)
{
  if (i <= 0) return '.';
  if (i == 1) return '/';
  if (i >= 2 && i < 12) return '0' - 2 + i;
  if (i >= 12 && i < 38) return 'A' - 12 + i;
  if (i >= 38 && i < 63) return 'a' - 38 + i;

  return 'z';
}

/* Random salt generator.  */
static char *make_salt (void)
{
  time_t now;
  unsigned long x;
  static char result[3];

  time (&now);
  x += now + getpid () + clock ();
  result[0] = i64c (((x >> 18) ^ (x >> 6)) & 077);
  result[1] = i64c (((x >> 12) ^ x) & 077);
  result[2] = 0;

  return result;
}

/* Change password for user `user_name'.
   If `admin' is zero, then compare `old_password' before changing
   for `new_password'.
   Return 0 on success
   Return 1 on error.
   Return 2 if options are invalid.
   Return 3 if crypt not supported.
   Return 4 if password doesn't match.
   Return 6 if the user doesn't exist.
   Return 7 if the uid is out of range.  */
int vsd_chpass (const char *server_root, const char *user_name,
		int admin, const char *old_password, const char *new_password)
{
  struct vsd_file_line *file, *line;
  char *cp;
  struct passwd pw;
  dstring_t ds;
  uid_t min, max;

  vsd_passwd_load (&file, server_root);
  /* Match the line and the user.  */
  line = vsd_file_find_entry (file, user_name);
  if (line == NULL)
    {
      vsd_passwd_free (file);
      return (errno == ENOENT) ? 6 : 1;
    }

  vsd_passwd_decode (line, &pw);
  /* Ensure the user account that we are changing the password
     for is within the virtual server range.  */
  vsd_get_uidgid (server_root, &min, &max, NULL, NULL);
  if (pw.pw_uid < min || pw.pw_uid > max)
    {
      vsd_passwd_free (file);
      return 7;
    }

  if (! admin)
    {
      if (old_password == NULL)
	{
	  errno = EINVAL;
	  vsd_passwd_free (file);
	  return 1;
	}

      /* Check the old password.  */
      cp = crypt (old_password, pw.pw_passwd);
      if (! cp)
	{
	  /* Crypt may return a null pointer. This usually means that crypt
	     isn't supported on the system.  */
	  vsd_passwd_free (file);
	  return 3;
	}

      /* If not the administrator then verify the old password matches before
	 changing for the new.  */
      if (strcmp (cp, pw.pw_passwd) != 0)
	{
	  vsd_passwd_free (file);
	  return 4;
	}
    }

  /* Encrypt the new password.  */
  cp = crypt (new_password, make_salt ());
  free (pw.pw_passwd);
  pw.pw_passwd = strdup (cp);

  /* Replace the line in the user file with the new password.  */
  ds = dstring_new (64);
  dstring_fmt_append (ds, "%s:%s:%d:%d:%s:%s:%s",
		      pw.pw_name, pw.pw_passwd,
		      pw.pw_uid, pw.pw_gid,
		      pw.pw_gecos, pw.pw_dir, pw.pw_shell);
  vsd_file_insert_line (line, dstring_ptr (ds));
  /* Delete the old line.  */
  vsd_file_delete_line (file, line);
  /* Save.  */
  vsd_passwd_save (file, server_root);
  vsd_passwd_free (file);
  dstring_free (ds);

  return 0;
}
