/* Argument vector manipulation.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "libvsd.h"

/* Add a new entry to the argv vector. If `argv' is NULL, then create
   a new vector.  Return NULL on error.  */
char **vsd_argv_add (char **argv, int *argc, char *line)
{
  if (argv == NULL)
    {
      argv = (char **) malloc (sizeof (char *));
      *argc = 1;
    }
  else
    {
      *argc = *argc + 1;
      argv = (char **) realloc (argv, (*argc + 1) * sizeof (char *));
    }

  if (argv == NULL)
    return NULL;

  argv[*argc - 1] = (char *) malloc (strlen (line) + 1);
  strcpy (argv[*argc - 1], line);

  return argv;
}

/* Free memory allocated to an argument vector.  */
void vsd_argv_free (char **argv, int argc)
{
  int x;

  if (argv)
    {
      for (x = 0; x < argc; x++)
        if (argv[x])
          free (argv[x]);

      free (argv);
    }
}

/* Split the `arguments' string into an argument vector. Use
   `sep' as the characters to seperate arguments.  Returns
   the argument vector and the vector size.  */
char **vsd_argv_parse (const char sep, char *arguments,
                       int *argvec_count)
{
  char **argvec = NULL, *p;
  int arg_count;

  if (arguments == NULL)
    {
      *argvec_count = 0;
      return argvec;
    }

  /* Count the number of arguments.  */
  arg_count = 1;
  p = arguments;
  /* Skip initial whitespace.  */
  while (*p && isspace (*p))
    p++;
  while (*p)
    {
      if (*p == sep)
	arg_count++;
      p++;
    }

  argvec = (char **) malloc ((arg_count + 1) * sizeof (char *));
  arg_count = 0;
  if (argvec == NULL)
    {
      *argvec_count = 0;
      return NULL;
    }

  p = arguments;

  /* Skip initial whitespace.  */
  while (*p && isspace (*p))
    p++;
  argvec[arg_count ++] = p;

  while (*p != '\0')
    {
      /* Skip chars until we find the seperator or a quote.  */
      while (*p && *p != sep)
	p++;
      if (! *p)
	break;
      *p++ = '\0';

      /* Skip any extra seperators.  */
      while (*p && *p == sep)
	p++;

      if (*p == '\"')
	{
	  argvec[arg_count++] = ++p;
	  while (*p && *p != '\"')
	    p++;
	  *p++ = '\0';
	}
      else
	{
	  /* Set pointer for the next command line argument.  */
	  argvec[arg_count++] = p;
	}
    }

  *argvec_count = arg_count;
  return argvec;
}
