/* Command line interface for the virtual server administrator.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <pwd.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdarg.h>
#include <sys/syslog.h>
#include <unistd.h>
#include <netdb.h>
#include <pwd.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include "libvsd.h"
#include "module.h"

#ifdef WITH_OPENSSL
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#endif

struct io_svr
{
  int fd; /* file descriptor.  */
  int data; /* amount of data in the buffer.  */
  char *buffer; /* data buffer */
  int buffer_size; /* size of data buffer */
  int dp; /* index into data buffer.  */
  char virtual_server[32];
  int level;
#ifdef WITH_OPENSSL
  SSL_CTX *ctx;
  SSL *ssl;
#endif
};

extern char *stpcpy (char *dest, const char *src);
static void error (int usage,
		   const char *fmt, ...) __attribute__ ((__noreturn__));

/* #define DEBUG 1 */

/* If no data has been read in READ_TIMEOUT seconds then timeout.  */
#define READ_TIMEOUT 20

/* IO abstraction.  Provides a common interface for communicating with
   or without SSL to the hosting server.  */

/* Output `bufsiz' bytes from `buffer' to the outgoing file descriptor,
   which may be the terminal, a file or a socket.  */
static int adm_io_read (struct io_svr *io, char *buffer, int bufsiz)
{
  int ret;
#ifdef WITH_OPENSSL
  ret = SSL_read (io->ssl, buffer, bufsiz);
#else
  while ((ret = read (io->fd, buffer, bufsiz)) == -1 && errno == EINTR)
    ;
#endif
  return ret;
}

/* Read `length' bytes from the incoming file descriptor (either terminal,
   file or socket) into `buffer'.  */
static int adm_io_write (struct io_svr *io, const char *buffer, int length)
{
  int ret;
#ifdef WITH_OPENSSL
  ret = SSL_write (io->ssl, buffer, length);
#else
  while ((ret = write (io->fd, buffer, length) == -1) && errno == EINTR)
    ;
#endif
  return ret;
}

/* Open a connection to the virtual server adminstrator hosted on
   'hostsvr'.  If compiled with SSL support then this function will
   try and negotiate an SSL session.  Certificates are expected to be
   found under the SSL_BASE directory.  For example, if SSL_BASE is
   set to /etc/vsd and we are connecting as virtual server foobar, then
   /etc/vsd/client/foobar.key is foobar's private key and
   /etc/vsd/client/foobar.crt if foobar's certificate.

   VSD uses the Common Name (CN) field of the private key to work out
   which virtual server the client is trying to connect as.  The CN
   must match the virtual server directory name e.g. in the example
   above, CN is `foobar'.

   Return 0 on success or 1 on error.  */
static int adm_io_initialise (struct io_svr *io, const char *hostsvr,
			      const char *virtual_server)
{
  struct hostent *hostent;
  struct sockaddr_in addr;
#ifdef WITH_OPENSSL
  char key[256], buf[4096];

  /* Initialise the SSL library, read and verify that the private key
     and certificate keys are valid.  */
  SSLeay_add_ssl_algorithms ();
  SSL_load_error_strings ();
  io->ctx = SSL_CTX_new (SSLv23_method ());
  if (io->ctx == NULL)
    {
      fprintf (stderr, "error: %s\n",
	       ERR_error_string (ERR_get_error (), buf));
      return 1;
    }

  sprintf (key, SSL_BASE "/client/%s.key", virtual_server);
  if (SSL_CTX_use_PrivateKey_file (io->ctx, key, SSL_FILETYPE_PEM) <= 0)
    {
      fprintf (stderr, "error: %s: (%s/client/%s.key)\n",
	       ERR_error_string (ERR_get_error (), buf),
	       SSL_BASE, virtual_server);
      return 1;
    }

  sprintf (key, SSL_BASE "/client/%s.crt", virtual_server);
  if (SSL_CTX_use_certificate_file (io->ctx, key, SSL_FILETYPE_PEM) <= 0)
    {
      fprintf (stderr, "error: %s: (%s/client/%s.crt)\n",
	       ERR_error_string (ERR_get_error (), buf),
	       SSL_BASE, virtual_server);
      return 1;
    }
    
  /* Check the private key is valid.  */
  if (! SSL_CTX_check_private_key (io->ctx))
    {
      printf ("error: Private key/certificate mismatch\n");
      return 1;
    }
#endif

  /* Establish a connection to the remote host.  */
  hostent = gethostbyname (hostsvr);
  if (hostent == NULL)
    {
      printf ("could not resolve host %s\n", hostsvr);
      return 1;
    }

  memset (&addr, 0, sizeof (addr));
  addr.sin_family = AF_INET;
  addr.sin_port = htons (WITH_VSD_PORT);
  addr.sin_addr = *(struct in_addr *) hostent->h_addr;

  io->fd = socket (PF_INET, SOCK_STREAM, 0);
  if (io->fd == -1)
    {
      printf ("socket create failed: %s\n", strerror (errno));
      return 1;
    }

  if (connect (io->fd, (struct sockaddr *) &addr, sizeof (addr)) == -1)
    {
      printf ("connection to %s could not be established: %s\n",
	      hostsvr, strerror (errno));
      return 1;
    }

  io->data = 0;
  io->buffer = NULL;
  io->dp = 0;

#ifdef WITH_OPENSSL
  /* Connection is now established, so attach an SSL communications layer.  */

  /* SSL_CTX_set_verify (ctx, SSL_VERIFY_PEER, NULL); */
  io->ssl = SSL_new (io->ctx);
  SSL_set_fd (io->ssl, io->fd);
  SSL_connect (io->ssl);
#ifdef DEBUG
  {
    X509 *server_cert;
    printf ("SSL connection using %s\n", SSL_get_cipher (io->ssl));
    server_cert = SSL_get_peer_certificate (io->ssl);
    if (server_cert != NULL)
      {
	char *s, buf[256];
	
	printf ("server certificate:\n");
	s = X509_NAME_oneline (X509_get_subject_name (server_cert),
			       buf, sizeof (buf));
	if (s)
	  printf ("  subject: %s\n", s);
	s = X509_NAME_oneline (X509_get_issuer_name (server_cert),
			       buf, sizeof (buf));
	if (s)
	  printf ("  issuer: %s\n", s);
	
	X509_free (server_cert);
      }
    else
      printf ("server does not have a certificate\n");
  }
#endif /* DEBUG */
#endif /* WITH_OPENSSL */

  if (vsd_reply_success (io))
    return 1;

  return 0;
}

/* Close the connection to the virtual server administrator.  Shutdown
   the SSL session, if SSL support was compiled in.  */
static void adm_io_finalise (struct io_svr *io)
{
#ifdef WITH_OPENSSL
  SSL_shutdown (io->ssl);
  close (io->fd);
  SSL_free (io->ssl);
  SSL_CTX_free (io->ctx);
#else
  close (io->fd);
#endif
}

/* Read a line from the virtual server.  A line is terminated by a single
   line-feed character.  If a successful read was performed, a pointer
   to the line will be returned, otherwise NULL will be returned.  */
char *vsd_get_line (void *io_)
{
  char *cp;
  time_t start_time, end_time;
  struct io_svr *io = (struct io_svr *) io_;

  /* Does any data exist in the buffer ? */
  if (io->data)
    {
      /* Look for a newline character in the buffer.  */
      cp = strchr (io->buffer + io->dp, '\n');

      if (cp)
	{
	  int found;

	  /* A newline was found.  */
	  *cp = '\0'; /* Terminate the string.  */
	  found = io->dp; /* Mark the start of the string.  */
	  /* Reduce the amount of data left in the buffer.  */
	  io->data -= (cp + 1 - (io->buffer + io->dp));

	  /* Move the data pointer to the start of the next string.  */
	  io->dp = (cp + 1) - io->buffer;

#ifdef DEBUG
	  printf ("get_line: '%s'\n", io->buffer + found);
#endif
	  return io->buffer + found;
	}
      else
	{
	  /* No newline was found. Move remaining data to the beginning
	     of the buffer.  */
	  memmove (io->buffer, io->buffer + io->dp, io->data);
	  io->dp = 0; /* Reset line index to the beginning.  */
	}
    }
  else
    io->dp = 0;

  if (! io->buffer)
    {
      /* Initially allocate 256 bytes for the receive buffer.  */
      io->buffer_size = 256;
      io->buffer = (char *) malloc (io->buffer_size + 1);
    }

  start_time = time (0);
  while (1)
    /*  while (io->data < io->buffer_size) */
    {
      int ret;

#ifdef DEBUG
      printf ("-- buffer_size = %d, data = %d\n", 
	      io->buffer_size, io->data);
#endif

      if (io->buffer_size - io->data < 16)
	{
#ifdef DEBUG
	  printf ("increase buffer from %d by 256\n", io->buffer_size);
#endif
	  /* Getting tight on memory, increase the read buffer size.  */
	  io->buffer_size += 256;
	  io->buffer = (char *) realloc (io->buffer, io->buffer_size + 1);
	}

      /* Retry read if failure was from an interrupt. */
      memset (io->buffer + io->data, 0, io->buffer_size - io->data);
      ret = adm_io_read (io, &io->buffer[io->data],
			 io->buffer_size - io->data);
      /* io->buffer_size - (io->data + 1) */

#ifdef DEBUG
      {
	int x;
	printf ("-- ret = %d, buffer = ", ret);
	for (x = 0; x <= ret; x++)
	  printf ("%c", io->buffer[io->data + x]);
	printf ("[the end]\n");
      }
#endif

      if (ret == -1)
	break; /* Usually an error.  */

      if (ret == 0)
	{
	  struct timespec period, remain;
	  /* Timeout if read failure for READ_TIMEOUT seconds.  */
	  end_time = time (0);
	  if (end_time - start_time > READ_TIMEOUT)
	    {
	      printf ("Read timeout\n");
	      return NULL;
	    }
	  /* Sleep for 0.2 seconds to stop read timeouts taking up
	     too much CPU.  */
	  period.tv_sec = 0;
	  period.tv_nsec = 200000000;
	  nanosleep (&period, &remain);
	}

      io->data += ret;
      cp = strchr (io->buffer, '\n');
      if (cp)
	{
	  *cp = '\0';
	  io->data -= (cp + 1 - (io->buffer + io->dp));
	  io->dp = (cp + 1) - io->buffer;

#ifdef DEBUG
	  printf ("get_line: '%s'\n", io->buffer);
#endif
	  return io->buffer;
	}
    }

  return NULL;
}

/* Send 'line' to the virtual server administrator.
   Return 0 on success, -1 on failure.  */
int vsd_send_line (void *io, const char *fmt, ...)
{
  va_list ap;
  int ret, len, nchars, size;
  char *line;

  va_start (ap, fmt);

  size = 256;
  line = (char *) malloc (size);
  nchars = vsnprintf (line, size, fmt, ap);
  if (nchars >= size)
    {
      size = nchars + 2;
      line = (char *) realloc (line, size);
      vsnprintf (line, size, fmt, ap);
    }

#ifdef DEBUG
  printf ("vsd_send_line: '%s'\n", line);
#endif

  strcat (line, "\n");
  len = strlen (line);
  ret = adm_io_write ((struct io_svr *) io, line, len);

  free (line);
  if (ret == len)
    return 0;

  return -1;
}

/* Receive a reply from the virtual server administrator.
   If it is a +OK message then return 0.
   If it is a -ERR message, return 1 (i.e. failure).  */
int vsd_reply_success (void *io)
{
  char *line;

  line = vsd_get_line (io);
  if (line == NULL)
    return 1;

  if (strncmp (line, "+OK", 3) == 0)
    return 0;

  /* Error messages are of the form: -ERR<space><error msg>  */
  printf ("%s\n", line + sizeof ("-ERR"));
  return 1;
}

/* Retrieve all data that VSD has returned.  This is meant to be used
   as the receiving part of a message exchange i.e. the client has
   previously sent a command request.  This function reads data a line
   at a time until the string `EOF' is returned.  Data is extracted
   from the result.

   Upon successful completion, a pointer to a buffer containing the data
   will be returned, otherwise NULL will be returned.  */
char *vsd_process_reply (void *io)
{
  char *line, *in;
  char **vec = NULL;
  int vecc = 0;
  dstring_t buf = dstring_new (32);
  int level = 0;
  int flag = 0;

  while (1)
    {
      /* Get a line.  */
      line = vsd_get_line (io);
      if (line == NULL)
	break;

      in = line;
      if (level != 2)
	{
	  /* Skip blanks.  */
	  while (*in && isspace (*in))
	    in++;
	  if (! *in)
	    continue;
	  if (in[0] == 'E' && in[1] == 'O' && in[2] == 'F')
	    break;

	  /* Split arguments into an argument vector.  */
	  vec = vsd_argv_parse (' ', in, &vecc);
	  if (vec == NULL)
	    continue;
	}

      if (level == 0)
	{
	  /* Expect `virtual server {'.  */
	  level ++;
	  free (vec);
	  flag = 1;
	}
      else if (level == 1)
	{
	  /* Expect a command or a `}'.  */
	  if (*in == '}')
	    {
	      level --;
	    }
	  else
	    {
	      if (*vec[vecc - 1] == '{')
		{
		  level ++;
		  vecc --;
		}
	    }
	}
      else if (level == 2)
	{
	  if (strchr (in, '}'))
	    {
	      /* End of data segment.  */
	      level --;
	    }
	  else
	    dstring_fmt_append (buf, "%s\n", in);
	}
    }

  if (buf->length)
    {
      char *s = strdup (buf->s);
      dstring_free (buf);
      return s;
    }

  if (! flag)
    return "-ERR expected result but got nothing";

  return NULL;
}

/* Receive data from vsd (just like vsd_process_reply), but return the raw
   protocol, rather than processing it.  This is used only by mod_raw.  */
char *vsd_raw_reply (void *io)
{
  dstring_t buf = dstring_new (32);

  while (1)
    {
      char *line = vsd_get_line (io);
      if (line == NULL)
	break;

      if (line[0] == '+' && line[1] == 'O' && line[2] == 'K' && line[3] == ' ')
	break;

      dstring_fmt_append (buf, "%s\n", line);
    }

  if (buf->length)
    {
      char *s = strdup (buf->s);
      dstring_free (buf);
      return s;
    }

  return NULL;
}

/* Read contents of `filename' into a malloc buffer and return
   a pointer to it.  */
char *vsd_read_file (void *io, const char *filename)
{
  char *buffer = NULL;
  struct stat sb;

  if (! stat (filename, &sb))
    {
      buffer = (char *) malloc (sb.st_size + 1);
      if (buffer)
	{
	  FILE *stream = fopen (filename, "r");
	  fread (buffer, sb.st_size, 1, stream);
	  fclose (stream);
	}
    }
  return buffer;
}

/* Send a command to VSD.  The command and its arguments are supplied
   in `fmt' and the variable number of arguments that follow.  If `data'
   is not NULL, then it points to a buffer that contains further data
   to be sent along with the command.   Return 0 on success, anything
   else is failure.  */
int vsd_send_command (void *io_, const char *data, const char *fmt, ...)
{
  va_list ap;
  int status, nchars, size;
  char *line;
  struct io_svr *io = (struct io_svr *) io_;

  va_start (ap, fmt);
  status = vsd_send_line (io_, "%s {",
			  ((io->level & ADMIN_HOST)
			   ? "<root>" : io->virtual_server));

  size = 256;
  line = (char *) malloc (size);
  nchars = vsnprintf (line, size, fmt, ap);
  if (nchars >= size)
    {
      size = nchars + 2;
      line = (char *) realloc (line, size);
      vsnprintf (line, size, fmt, ap);
    }

  if (data)
    status = vsd_send_line (io_, "%s {\n%s\n}", line, data);
  else
    status = vsd_send_line (io_, "%s", line);
  status = vsd_send_line (io_, "}\nEOF");

  free (line);
  va_end (ap);
  return status;
}

/* Send command `cmd' to VSD.  The arguments are passed as in vector
   format in `argv' and the argument count is `argc'.  If `data' is
   not NULL, then it points to a buffer that contains further data
   to be sent along with the command.  Return 0 on success, anything
   else is failure.  */
int vsd_send_vcommand (void *io_, const char *data, const char *cmd,
		       int argc, char *argv[])
{
  int status, nchars, size, x;
  char *line, *p;
  struct io_svr *io = (struct io_svr *) io_;

  status = vsd_send_line (io_, "%s {",
			  ((io->level & ADMIN_HOST)
			   ? "<root>" : io->virtual_server));

  /* Calculate command line size.  */
  for (nchars = x = 0; x < argc; x++)
    nchars += strlen (argv[x]) + 1;
  nchars += strlen (cmd) + 1;
  line = (char *) malloc (nchars + 1);

  /* Build command line.  */
  p = stpcpy (line, cmd);
  for (x = 0; x < argc ; x++)
    {
      *p++ = ' ';
      p = stpcpy (p, argv[x]);
    }
  *p = '\0';

  if (data)
    status = vsd_send_line (io_, "%s {\n%s\n}", line, data);
  else
    status = vsd_send_line (io_, "%s", line);
  free (line);
  status = vsd_send_line (io_, "}\nEOF");

  return status;
}

/* Send a command to VSD.  The command and its arguments are supplied
   in `fmt' and the variable number of arguments that follow.  If `fname'
   is not NULL, then it points to a file that will be read and its
   contents sent as additional data for the command.  Return 0 on success,
   anything else is failure.  */
int vsd_send_command_file (void *io_,
			   const char *fname, const char *fmt, ...)
{
  va_list ap;
  int status, nchars, size;
  char *line;
  struct io_svr *io = (struct io_svr *) io_;

  va_start (ap, fmt);
  status = vsd_send_line (io_, "%s {", io->virtual_server);

  size = 256;
  line = (char *) malloc (size);
  nchars = vsnprintf (line, size, fmt, ap);
  if (nchars >= size)
    {
      size = nchars + 2;
      line = (char *) realloc (line, size);
      vsnprintf (line, size, fmt, ap);
    }

  if (fname)
    {
      FILE *stream = fopen (fname, "r");

      status = vsd_send_line (io_, "%s {\n", line);
      if (stream != NULL)
	{
	  while (fgets (line, sizeof (line), stream))
	    vsd_send_line (io_, "%s", line);

	  fclose (stream);
	}
      vsd_send_line (io_, "}\n");
    }
  else
    status = vsd_send_line (io_, "%s", line);
  status = vsd_send_line (io_, "}\nEOF");
  free (line);

  va_end (ap);
  return status;
}

/* There are two types of command in VSD - ADMIN_HOST and ADMIN_VS.
   These levels affect the type of commands available to the client.
   If the client were to connect to VSD using the VS `<root>' then
   only ADMIN_HOST commands would be available.  If the client were
   to connect to VSD with a proper VS name, then only ADMIN_VS commands
   would be available.

   There are a couple of occasions where a single command is available
   for both ADMIN_VS and ADMIN_HOST.  It is up-to the module to decide
   which is the most appropriate.  This function is provided so that
   the module can set the appropriate level, without having intimate
   knowledge of the io_svr structure.  */
void vsd_set_level (void *io_, int level)
{
  struct io_svr *io = (struct io_svr *) io_;
  io->level = level;
}

static char *program_name;

/* Display an error message `fmt' with an optional number of arguments.
   Then terminate the application.  If `usage' is set to 1, then prepend
   the message with `usage: '.  */
static void error (int usage, const char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  if (usage == 1)
    fprintf (stdout, "usage: %s ", program_name);

  vfprintf (stdout, fmt, ap);
  va_end (ap);
  fputc ('\n', stdout);

  exit (1);
}

int main (int argc, char *argv[])
{
  struct io_svr io;
  char host_server[256], virtual_server[256];
  char *operation;
  const char *status;
  struct command_tag *tag;

  /* Check that the 'operation' argument has not been supplied with */
  /* a prepended '-'. Print error if it is */
  if ((argc > 1) && ((operation = strrchr (argv[1], '-')) != NULL))
    {
      error (0, "syntax: vsdadm %s ...", ++operation);
    }
  operation = argv[1];

  /* We can be sure that there should be at least two arguments.  Print
     full command list, if less are supplied.  */
  if (operation == NULL || argc <= 2)
    {
      module_help (ADMIN_VS | ADMIN_HOST, NULL);
      return 1;
    }

  /* Log command line arguments to syslog.  */
  openlog ("vsdadm", LOG_PID, LOG_DAEMON);
  switch (argc)
    {
    case 2:
      syslog (LOG_WARNING, "cmd: %s", argv[1]);
      break;
    case 3:
      syslog (LOG_WARNING, "cmd: %s %s", argv[1], argv[2]);
      break;
    case 4:
      syslog (LOG_WARNING, "cmd: %s %s %s", argv[1], argv[2], argv[3]);
      break;
    case 5:
      syslog (LOG_WARNING, "cmd: %s %s %s %s", argv[1], argv[2], argv[3],
	      argv[4]);
      break;
    case 6:
      syslog (LOG_WARNING, "cmd: %s %s %s %s %s", argv[1],
	      argv[2], argv[3], argv[4], argv[5]);
      break;
    case 7:
      syslog (LOG_WARNING, "cmd: %s %s %s %s %s %s", argv[1],
	      argv[2], argv[3], argv[4], argv[5], argv[6]);
      break;
    case 8:
      syslog (LOG_WARNING, "cmd: %s %s %s %s %s %s %s", argv[1],
	      argv[2], argv[3], argv[4], argv[5], argv[6], argv[7]);
      break;
    }
  
  program_name = argv[0];

  /* Check the syntax of command.  */
  if (module_check_cmd (ADMIN_VS | ADMIN_HOST, operation,
			argc - 2, argv + 1, &tag) == 1)
    error (0, "syntax: %s %s", operation, tag->vsdadm_syntax);

  /* FIXME++  This chunk of code should be in the syntax argument
     checking of each VSD module.  */
  if (argc >= 4)
    {
      if (strlen (argv[3]) > 31)
	{
	  printf ("virtual server %s is too long\n", argv[3]);
	  return 1;
	}
      strcpy (virtual_server, argv[3]);
    }
  else
    strcpy (virtual_server, "<root>");
  /* FIXME-- */

  /* Get the host server and virtual server names.  */
  strcpy (host_server, argv[2]);

  /* Open a connection to the host server.  */
  if (strcmp (virtual_server, "<root>") == 0 || tag->level == ADMIN_HOST)
    {
      if (adm_io_initialise (&io, host_server, "root"))
	return 1;
    }
  else
    {
      if (adm_io_initialise (&io, host_server, virtual_server))
	return 1;
    }

  strcpy (io.virtual_server, virtual_server);
  /* Perform the action on the virtual server.  */
  status = module_command (&io, ADMIN_VS | ADMIN_HOST, operation,
			   argc - 1, argv + 2);
  adm_io_finalise (&io);

  if (status && strncmp (status, "-ERR", 4) == 0)
    {
      printf ("error: %s\n", status);
      return 1;
    }

  if (status)
    printf ("%s", status);
  return 0;
}

