/* User account manipulation.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <sys/syslog.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include "vsd.h"
#include "libvsd.h"

#define USE_PROFTPD 1

extern char *stpcpy (char *, const char *);

static const char *is_homedir_valid (const char *home)
{
  if (home == NULL)
    return "is_homedir_valid: passed NULL argument";

  if (strspn (home, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-./_") != strlen (home))
    return "Path for home directory contains invalid characters";

  return NULL;
}

/* List all users on a virtual server and various details about them.
   This command takes no arguments.  */
void user_list (struct connection *vc, int argc, char *argv[])
{
  struct passwd pw;
  uid_t min_uid, max_uid;
  struct vsd_file_line *pw_file, *ent;

  if (vsd_passwd_load (&pw_file, vc->server_root))
    io_buffer_store (vc, "-ERR Could not open %s password file",
		     vc->virtual_server);
  else
    {
      /* Obtain a list of all users on the virtual server.  */
      vsd_get_uidgid (vc->server_root, &min_uid, &max_uid, NULL, NULL);
      ent = pw_file;
      do
	{
	  ent = vsd_getpwent (ent, &pw);

	  if (pw.pw_uid >= min_uid && pw.pw_uid <= max_uid)
	    io_buffer_store (vc, "name=%s uid=%d home=\"%s\" shell=\"%s\"\n",
			     pw.pw_name, pw.pw_uid, pw.pw_dir,
			     pw.pw_shell);
	} while (ent != NULL);
      vsd_passwd_free (pw_file);
    }
}

/* List all user groups on a virtual server and various details about them.
   This command takes no arguments.  */
void user_group_list (struct connection *vc, int argc, char *argv[])
{
  struct group *grp;
  FILE *gr_stream;
  gid_t min_uid, max_uid;

  gr_stream = vsd_opengrp (vc->server_root);
  if (gr_stream == NULL)
    io_buffer_store (vc, "-ERR Could not open %s group file",
		     vc->virtual_server);
  else
    {
      /* Obtain a list of all users on the virtual server.  */
      vsd_get_uidgid (vc->server_root, NULL, NULL, &min_uid, &max_uid);
      
      while ((grp = fgetgrent (gr_stream)))
	if (grp->gr_gid >= min_uid && grp->gr_gid <= max_uid)
	  {
	    char **list = grp->gr_mem;
	    io_buffer_store (vc, "name=%s gid=%d members=\"",
			     grp->gr_name, grp->gr_gid);
	    while (*list)
	      {
		io_buffer_store (vc, "%s", *list);
		if (list[1])
		  io_buffer_store (vc, ",");

		list++;
	      }
	    io_buffer_store (vc, "\"\n");
	  }	    

      vsd_closegrp (gr_stream);
    }
}

/* Add a user to a virtual server.  */
void user_add (struct connection *vc, int argc, char *argv[])
{
  char line[256];
  const char *username = argv[0], *password = NULL;
  const char *homedir = NULL;
  uid_t uid = 0;
  gid_t gid = 0;
  int status;
  const char *st;

  if ((st = vsd_valid_username (username)))
    {
      io_buffer_store (vc, "-ERR Cannot add user: %s", st);
      return;
    }

  /* Parse optional command line arguments.  */
  if (argc >= 1)
    {
      int x;

      for (x = 0; x <= argc - 1; x++)
	{
	  if (! password)
	    password = get_option ("pass", argv[x]);
	  if (! homedir)
	    homedir = get_option ("home", argv[x]);
	}
    }

  if (homedir && (st = is_homedir_valid (homedir)))
    {
      io_buffer_store (vc, "-ERR Cannot add user: %s", st);
      return;
    }

  if (! homedir)
    {
      /* The default home directory is /home/user.  */
      sprintf (line, "/home/%s", username);
      homedir = line;
    }

  /* The default group is the same as the user name.  */
  status = vsd_groupadd (vc->server_root, username, &gid);
  /* Add the user to the virtual server.  */
  status = vsd_useradd (vc->server_root, username, homedir,
			username, NULL, "/bin/bash", &uid, NULL);
  if (status == 9)
    io_buffer_store (vc, "-ERR User is already defined");
  else
    {
      /* Create the user's home directory.  */
      if (homedir == NULL)
	sprintf (line, "%s/home/%s", vc->server_root, username);
      else
	sprintf (line, "%s/%s", vc->server_root, homedir);
      mkdir (line, (mode_t) 0755);
      chown (line, uid, gid);

      if (password != NULL
	  && vsd_chpass (vc->server_root, username, 1, NULL, password))
	io_buffer_store (vc, "-ERR User added but couldn't set password");
    }
}

/* Delete a user from a virtual server.  */
void user_del (struct connection *vc, int argc, char *argv[])
{
  int x, membc, status;
  int flags_delete_home, flags_delete_group, flags_delete_files;
  int flags_change_uids;
  struct vsd_file_line *priv_file;
  char **membv;
  const char *st = vsd_valid_username (argv[0]);

  if (st)
    {
      io_buffer_store (vc, "-ERR Cannot delete user: %s", st);
      return;
    }

  if (strcmp (argv[0], "admin") == 0 || strcmp (argv[0], "web") == 0
      || strcmp (argv[0], "ftp") == 0 || strcmp (argv[0], "mail") == 0)
    {
      io_buffer_store (vc, "-ERR Deleting the %s user is silly");
      return;
    }

  flags_delete_home = flags_delete_group = 0;
  flags_delete_files = flags_change_uids = 0;
  flags_delete_group = 1;
  for (x = 1; x < argc; x++)
    {
      if (strcmp (argv[x], "-r") == 0)
	flags_delete_home = 1; /* Delete the user's home directory.  */
      if (strcmp (argv[x], "-g") == 0)
	flags_delete_group = 1; /* Delete the user's associated group.  */
      if (strcmp (argv[x], "-f") == 0)
	flags_delete_files = 1; /* Delete all files belonging to the user.  */
      if (strcmp (argv[x], "-c") == 0)
	flags_change_uids = 1; /* Make all user's files owned by admin.  */
    }

  /* Does the user already exist ? */
  status = vsd_userdel (vc->server_root, argv[0]);
  if (status == 6)
    {
      io_buffer_store (vc, "-ERR User %s does not exist", argv[0]);
      return;
    }

  /* Delete the privileges for this user.  */
  if (vsd_priv_load (&priv_file, vc->server_root))
    {
      io_buffer_store (vc, "-ERR Could not delete privileges for user: %m");
      /* Limp on and attempt to delete the user's group, if requested.  */
    }
  else
    {
      /* Get a list of privileges that `old_user' is a member of.  */
      membv = vsd_priv_get_members (priv_file, argv[0], &membc);

      /* Delete `old_user' from the privileges and add `new_user' to
	 the privilegs.  */
      for (x = 0; x < membc; x++)
	vsd_priv_delete_member (priv_file, membv[x], argv[0]);

      vsd_priv_save (priv_file, vc->server_root);
      vsd_priv_free (priv_file);
      vsd_argv_free (membv, membc);
    }

  if (flags_delete_group)
    {
      status = vsd_groupdel (vc->server_root, argv[0]);
      if (status == 6)
	io_buffer_store (vc, "-ERR Group %s does not exist", argv[0]);
    }
}

/* Modify a user.  */
void user_mod (struct connection *vc, int argc, char *argv[])
{
  const char *username = argv[0], *password = NULL, *homedir = NULL;
  const char *shell = NULL, *new_username = NULL, *gecos = NULL;
  int status;
  const char *st = vsd_valid_username (username);

  if (st)
    {
      io_buffer_store (vc, "-ERR Cannot modify user: %s", st);
      return;
    }

  /* Command syntax explanation:
       --pass=<password> - new password for the user
       --home=<home dir> - the user's new login directory
       --shell=<default shell> - the name of the user's new login shell
       --login=<login-name> - change the user's account name to <login-name>
  */

  /* Parse optional command line arguments.  */
  if (argc >= 1)
    {
      int x;

      for (x = 0; x <= argc - 1; x++)
	{
	  if (! password)
	    password = get_option ("pass", argv[x]);
	  if (! homedir)
	    homedir = get_option ("home", argv[x]);
	  if (! shell)
	    shell = get_option ("shell", argv[x]);
	  if (! new_username)
	    new_username = get_option ("login", argv[x]);
	  if (! gecos)
	    gecos = get_option ("gecos", argv[x]);
	}
    }

  if (homedir && is_homedir_valid (homedir))
    return;

  if (new_username)
    {
      if (strcmp (new_username, "admin") == 0
	  || strcmp (new_username, "web") == 0
	  || strcmp (new_username, "ftp") == 0
	  || strcmp (new_username, "mail") == 0)
	{
	  io_buffer_store (vc,
			   "-ERR Modifying the %s user login name is silly",
			   new_username);
	  return;
	}

      st = vsd_valid_username (new_username);
      if (st)
	{
	  io_buffer_store (vc,
			   "-ERR Supplied new login name %s is invalid: %s",
			   new_username, st);
	  return;
	}
    }

  status = vsd_usermod (vc->server_root, username, new_username, homedir,
			NULL /*username */, NULL, shell, -1, gecos);
  if (status == 10)
    io_buffer_store (vc, "-ERR Couldn't access /etc/passwd: %m");
  else if (status == 9)
    io_buffer_store (vc, "-ERR User %s is already in use", username);
  else if (status == 7)
    io_buffer_store (vc, "-ERR User %s does not exist", username);
  else if (status == 6)
   io_buffer_store (vc, "-ERR Group %s does not exist", username);
  /*   else if (status == 4)
     io_buffer_store (vc, "-ERR Uid %d is already in use", uid); */
  else if (status == 2)
    io_buffer_store (vc, "-ERR There is an error somewhere...");

  if (new_username)
    vsd_priv_rename_user (vc->server_root, username, new_username);

#if 0
  if (homedir)
    {
      char line[256];
      /* Create the new user home directory.  */
      if (snprintf (line, sizeof (line), "%s/%s", vc->server_root, homedir) < sizeof (line))
	{
	  mkdir (line, (mode_t) 0755);
	  chown (line, uid, gid);
	}
    }
#endif

  if (password != NULL
      && vsd_chpass (vc->server_root, username, 1, NULL, password))
    io_buffer_store (vc, "-ERR User modified but password not set");
}

/* Change password for a user on a virtual server.  */
void user_chpass (struct connection *vc, int argc, char *argv[])
{
  const char *user = argv[0], *old_pass = argv[1], *new_pass = argv[2];
  int status;
  const char *st = vsd_valid_username (user);

  /* Format:
       CHUSERPASS <username> <old password> <new password>

     where
       <username> - user on a virtual server
       <old password> - current password for the user
       <new password> - new password for the user
  */
  if (st)
    {
      io_buffer_store (vc, "-ERR Cannot change user password: %s", st);
      return;
    }

  /* Does the user already exist ? */
  status = vsd_chpass (vc->server_root, user, 0, old_pass, new_pass);
  if (status == 6)
    io_buffer_store (vc, "-ERR User %s does not exist", user);
  else if (status == 7)
    io_buffer_store (vc, "-ERR Cannot change password for user %s: permission denied", user);
  else if (status == 4)
    io_buffer_store (vc, "-ERR Incorrect password supplied for user %s", user);
  else if (status != 0)
    io_buffer_store (vc, "-ERR Password change failed with status %d", status);
}

/* Add a new group.  */
void user_group_add (struct connection *vc, int argc, char *argv[])
{
  const char *st = vsd_valid_username (argv[0]);

  /* Check command line validity and arguments.  */
  if (st)
    io_buffer_store (vc, "-ERR Cannot add group: %s", st);
  else
    {
      gid_t gid = 0;
      int status = vsd_groupadd (vc->server_root, argv[0], &gid);
      if (status == 9)
	io_buffer_store (vc, "-ERR Group %s is already defined", argv[0]);
    }
}

/* Delete an existing group.  */
void user_group_del (struct connection *vc, int argc, char *argv[])
{
  const char *st = vsd_valid_username (argv[0]);

  /* Check command line validity and arguments.  */
  if (st)
    io_buffer_store (vc, "-ERR Cannot delete group: %s", st);
  else
    {
      if (strcmp (argv[0], "admin") == 0 || strcmp (argv[0], "web") == 0
	  || strcmp (argv[0], "ftp") == 0 || strcmp (argv[0], "mail") == 0)
	io_buffer_store (vc, "-ERR Deleting the %s group is silly");
      else
	{
	  int status = vsd_groupdel (vc->server_root, argv[0]);
	  if (status == 9)
	    io_buffer_store (vc, "-ERR Group %s does not exist", argv[0]);
	}
    }
}

/* Modify an existing group.  */
void user_group_mod (struct connection *vc, int argc, char *argv[])
{
  const char *st = vsd_valid_username (argv[0]);

  /* Check command line validity and arguments.  */
  if (st)
    io_buffer_store (vc, "-ERR Cannot modify group: %s", st);
  else
    {
    }
}

