/* Quota command implementation.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Written by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD).

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/syslog.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef HAVE_SYS_QUOTA_H
#include <sys/quota.h>
#else
#ifdef HAVE_UFS_UFS_QUOTA_H
#include <ufs/ufs/quota.h>
#endif
#endif

#include "vsd.h"
#include "libvsd.h"
#include "quota.h"

/* DEV is passed to quotactl.  */
#if defined(HAVE_SYS_QUOTA_H)
#define DEV(map,vs) (map)->partition[(vs)->partition].device
#elif defined(HAVE_UFS_UFS_QUOTA_H)
#define DEV(map,vs) (map)->partition[(vs)->partition].pathname
#endif

static int diskquota (int cmd, const char *serverroot, int uid, caddr_t addr)
{
  int status, fd;
  char template[32];
  struct stat sb;

  if (stat (serverroot, &sb))
    return 1;

  /* Get disk quota information from the kernel.  We have to create
     a special block file with the same device number as the hard
     disk that the virtual server is located on.  The stat of /etc/passwd
     is a good enough way of determining the device number.  */
  strcpy (template, "/tmp/hdXXXXXX");
  fd = mkstemp (template);
  close (fd);
  unlink (template);
  mknod (template, 0600 | S_IFBLK, sb.st_dev);

#ifdef HAVE_SYS_QUOTA_H
  /* Linux style.  */
  status = quotactl (cmd, template, uid, addr);
#else
#ifdef HAVE_UFS_UFS_QUOTA_H
  /* BSD style.  */
  status = quotactl (template, cmd, uid, addr);
#endif
#endif
  unlink (template);
  return status;
}

/* Get quota stats for all users on the virtual server.
   If uid is NULL then don't create a uid list.
   If puser is NULL then don't create user quota stats.
   If total is NULL then don't tally totals.  */
static int vs_user_stats (struct connection *vc,
			  int **puid, void **puser_, void *total_)
{
  struct dqblk user_quota;
  int count, max, status;
  int *uid;
  struct dqblk *user;
  struct dqblk **puser = (struct dqblk **)puser_;
  struct dqblk *total = (struct dqblk *)total_;
  uid_t min_uid, max_uid;
  struct passwd pw;
  struct vsd_file_line *pw_file, *ent;

  if (puid == NULL && puser == NULL && total == NULL)
    return 0;

  if (vsd_passwd_load (&pw_file, vc->server_root))
    return 0;

  vsd_map_uid_range (vc->map, vc->virtual_server, &min_uid, &max_uid);

  max = 16;
  count = 0;
  uid = (int *) malloc (max * sizeof (int));
  user = (struct dqblk *) malloc (max * sizeof (struct dqblk));

  if (total != NULL)
    {
      total->dqb_bhardlimit = 0;
      total->dqb_bsoftlimit = 0;
      total->dqb_curblocks = 0;
    }

  /* Look at each /etc/passwd entry but only find quota information
     for users on the virtual server.  */
  ent = pw_file;
  do
    {
      ent = vsd_getpwent (ent, &pw);

      if (pw.pw_uid >= min_uid && pw.pw_uid <= max_uid)
	{
	  if (puid != NULL)
	    uid[count] = pw.pw_uid;
	  
	  status = diskquota (QCMD (Q_GETQUOTA, USRQUOTA), vc->server_root,
			      pw.pw_uid, (caddr_t) &user_quota);
	  if (status == 0)
	    {
	      if (puser != NULL)
		user[count] = user_quota;
	      
	      if (count >= max - 1)
		{
		  max += 16;
		  if (puid != NULL)
		    uid = (int *) realloc (uid, max * sizeof (int));
		  if (puser != NULL)
		    user = (struct dqblk *) realloc (user,
						     max * sizeof (struct dqblk));
		}

	      if (total != NULL)
		{
		  total->dqb_bhardlimit += user_quota.dqb_bhardlimit;
		  total->dqb_bsoftlimit += user_quota.dqb_bsoftlimit;
		  total->dqb_curblocks += user_quota.dqb_curblocks;
		}
	    }
	  else
	    {
	      if (puser != NULL)
		memset (&user[count], 0, sizeof (struct dqblk));
	    }

	  count ++;
	}
    } while (ent != NULL);
  vsd_passwd_free (pw_file);

  if (puid)
    *puid = uid;
  if (puser)
    *puser = user;

  return count;
}

/* Return quota limits for a user.  */
void quota_userget (struct connection *vc, int argc, char *argv[])
{
  struct dqblk quota;
  int status;
  struct vsd_file_line *pw_file;
  struct passwd pw;

  /* Check user exists.  */
  if (vsd_passwd_load (&pw_file, vc->server_root))
    {
      io_buffer_store (vc, "-ERR Cannot load /etc/passwd");
      return;
    }

  status = vsd_getpwnam (pw_file, argv[0], &pw);
  vsd_passwd_free (pw_file);
  if (status)
    {
      io_buffer_store (vc, "-ERR User %s does not exist", argv[0]);
      return;
    }

  /* should be a /dev */
  status = diskquota (QCMD (Q_GETQUOTA, USRQUOTA),
		      vc->server_root, pw.pw_uid, (caddr_t) &quota);

  if (status == -1)
    {
      syslog (LOG_ERR, "Get quota stats failed: %m");
      io_buffer_store (vc, "-ERR Could not get quota stats");
    }
  else
    io_buffer_store (vc, "uid=%d hardb=%d softb=%d curb=%d", pw.pw_uid,
		     quota.dqb_bhardlimit, quota.dqb_bsoftlimit,
		     quota.dqb_curblocks);
}

/* Set quota limits for a user.  */
void quota_userset (struct connection *vc, int argc, char *argv[])
{
  struct dqblk user_quota, total_quota;
  int status, bytes;
  const char *user, *size, *p;
  struct vsd_file_line *pw_file;
  struct passwd pw;

  user = argv[0];
  size = argv[1];

  /* Check user exists.  */
  if (vsd_passwd_load (&pw_file, vc->server_root))
    {
      io_buffer_store (vc, "-ERR Cannot load /etc/passwd");
      return;
    }

  status = vsd_getpwnam (pw_file, argv[0], &pw);
  vsd_passwd_free (pw_file);
  if (status)
    {
      io_buffer_store (vc, "-ERR User %s does not exist", argv[0]);
      return;
    }

  /* Get quota stats for the virtual server.  We require this to ensure
     that an individual quota limit is not set greater than the quota
     limit of the virtual server.  */

  /* Get quota stats for all users on the virtual server.  */
  vs_user_stats (vc, NULL, NULL, &total_quota);

  /* Get quota stats for the user.  */
  status = diskquota (QCMD (Q_GETQUOTA, USRQUOTA), vc->server_root,
		      pw.pw_uid, (caddr_t) &user_quota);
  if (status == -1 && errno == ESRCH)
    memset (&user_quota, 0, sizeof (user_quota));
  else if (status == -1 && errno != ESRCH)
    {
      syslog (LOG_ERR, "set_quota: get user quota stats failed: %m");
      io_buffer_store (vc, "-ERR Could not get user quota stats");
      return;
    }

  /* Subtract the user's quota from the server's total. We can then
     calculate a maximum allocation for the user.  */
  total_quota.dqb_bhardlimit -= user_quota.dqb_bhardlimit;
  total_quota.dqb_bsoftlimit -= user_quota.dqb_bsoftlimit;
  total_quota.dqb_curblocks -= user_quota.dqb_curblocks;

  /* Parse the size argument.  */
  p = size;
  if (*p == '+' || *p == '-')
    p++; /* Skip the +/- sign for now.  */

  bytes = 0;
  while (isdigit (*p))
    bytes = (bytes * 10) + (*p++ - '0');

  /* Handle a kilobyte/megabyte expression.  */
  if (toupper (*p) == 'M')
    bytes *= 1024;

  /* Now handle the +/- sign.  */
  if (size[0] == '+')
    user_quota.dqb_bsoftlimit += bytes;
  else if (size[0] == '-')
    user_quota.dqb_bsoftlimit -= bytes;
  else if (isdigit (size[0]))
    user_quota.dqb_bsoftlimit = bytes;

  /* Make the user's quota hard limit equal to it's soft limit.  */
  user_quota.dqb_bhardlimit = user_quota.dqb_bsoftlimit;

  /* Ensure user quota lies within the boundaries of the virtual server's
     disk quota.  */
  if (user_quota.dqb_bsoftlimit + total_quota.dqb_bsoftlimit
      > vc->vs->quota * 1024)
    {
      /* We could do this but I think it is a better idea to report an
	 error to the user.
	 user_quota.bsoftlimit = server_quota.bsoftlimit;  */
      io_buffer_store (vc, "-ERR Quota would exceed virtual server allocation");
      return;
    }

  if (user_quota.dqb_bsoftlimit < 0)
    {
      /* User's quota has been set far too low.  Flag as an error.  */
      io_buffer_store (vc, "-ERR User quota set too low");
      return;
    }

  if (user_quota.dqb_bsoftlimit == 0)
    {
      /* Don't let a user set the quota to unlimited.  */
      io_buffer_store (vc, "-ERR User quota cannot be set to 0K");
      return;
    }

  if (user_quota.dqb_bsoftlimit <= user_quota.dqb_curblocks)
    {
      /* Don't let a user set the quota below a user's current use.
         Doing this allows the user to circumvent the quota restrictions.  */
      io_buffer_store (vc,
		       "Quota cannot be set below user's current use of %dK",
		       user_quota.dqb_curblocks);
      return;
    }

  status = diskquota (QCMD (Q_SETQUOTA, USRQUOTA), vc->server_root,
		      pw.pw_uid, (caddr_t) &user_quota);
  if (status == -1)
    {
      syslog (LOG_ERR, "Set quota stats failed: %m");
      io_buffer_store (vc, "-ERR Could not set quota. Are quotas enabled ?");
    }
  else
    /* Synchronise the new quota to disk.  */
    if (diskquota (QCMD (Q_SYNC, 0), vc->server_root, 0, (caddr_t) 0) != 0)
      /* Don't report this as an error to the user.  */
      syslog (LOG_ERR, "set_quota: sync to disk failed: %m");
}

/* Get quota statistics for a virtual server and all users on it.  */
void quota_stats (struct connection *vc, int argc, char *argv[])
{
  int *uidv, count;
  struct dqblk *userv, totalv;
  struct passwd pw;
  struct vsd_file_line *pw_file;

  count = vs_user_stats (vc, &uidv, (void **) &userv, (void **) &totalv);

  if (vsd_passwd_load (&pw_file, vc->server_root))
    {
      io_buffer_store (vc, "-ERR Cannot load /etc/passwd");
      return;
    }

  while (--count >= 0)
    if (! vsd_getpwuid (pw_file, uidv[count], &pw))
      /* Strip the virtual server name off the user name.  */
      io_buffer_store (vc, "user=%s uid=%d hardb=%d softb=%d curb=%d\n",
		       pw.pw_name, uidv[count],
		       userv[count].dqb_bhardlimit,
		       userv[count].dqb_bsoftlimit,
		       userv[count].dqb_curblocks);

  io_buffer_store (vc, "hardb=%d softb=%d curb=%d",
		   totalv.dqb_bhardlimit, totalv.dqb_bsoftlimit,
		   totalv.dqb_curblocks);

  vsd_passwd_free (pw_file);
  free (uidv);
  free (userv);
}

/* Return quota limits for a virtual server.  */
void quota_vsget (struct connection *vc, int argc, char *argv[])
{
  struct vsd_vs *vs;
  int quota;

  vs = vsd_getmapent (vc->map, vc->virtual_server, NULL);
  quota = vs->quota * 1024;
  io_buffer_store (vc, "vsquota=%d", quota);
}

/* Set quota limits for a virtual server.  This call is restricted
   to a host server administrator.  */
void quota_vsset (struct connection *vc, int argc, char *argv[])
{
  struct vsd_vs *vs;
  struct dqblk total_quota;
  int quota, bytes;
  char *p, *size = argv[0];

  vs = vsd_getmapent (vc->map, vc->virtual_server, NULL);
  quota = vs->quota * 1024;

  /* Parse the size argument.  */
  p = size;
  if (*p == '+' || *p == '-')
    p++; /* Skip the +/- sign for now.  */

  bytes = 0;
  while (isdigit (*p))
    bytes = (bytes * 10) + (*p++ - '0');

  /* Handle a kilobyte/megabyte expression.  */
  if (toupper (*p) == 'M')
    bytes *= 1024;

  if (size[0] == '+')
    quota += bytes;
  else if (size[0] == '-')
    quota -= bytes;
  else
    quota = bytes;

  /* Get total quota allocation for all users on the virtual server.  */
  if (vs_user_stats (vc, NULL, NULL, &total_quota))
    {
      if (quota < (total_quota.dqb_bsoftlimit / 1024))
	/* Server's quota has been set lower than the total allocation
	   for all users on the server.  */
	io_buffer_store (vc, "-ERR Total user quota allocation exceeds VS quota");
      else if (quota < 0)
	/* Server's quota has been set far too low.  */
	io_buffer_store (vc, "-ERR VS quota is too low");
      else
	{
	  /* Set the quota in /etc/vsd.conf.  */
	  vs->quota = quota / 1024;
	  vsd_map_save (vc->map);
	  /* Set the quota in /etc/vsd/quota on the virtual server.  */
	  vsd_quota_set (vc->server_root, vs->quota);
	}
    }
  else
    io_buffer_store (vc, "-ERR Cannot get user quota usage");
}

