/* User privileges manipulation.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <sys/syslog.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>

#ifdef HAVE_LIBPAM
#include <security/pam_appl.h>
#endif

#include "vsd.h"
#include "libvsd.h"
#include "priv.h"

#define USE_PROFTPD 1

#ifdef HAVE_LIBPAM
struct checkpw_cred
{
  const char *uname;
  const char *pass;
};
#endif

extern char *stpcpy (char *, const char *);

/* Return 1 on success, -1 on error.
   If `admin_check' is true then the admin user must have privilege
   `privilege' to be able to assign it to `user_name'.  */
static int common_set_rights (struct connection *vc, const char *user_name,
			      const char *privilege, int admin_check)
{
  int status;
  struct vsd_file_line *list;
#ifdef USE_WUFTPD
  char *line;
  FILE *stream;
  struct passwd *pwd;
#endif

  if (privilege[0] != '-' && privilege[0] != '+')
    {
      io_buffer_store (vc, "-ERR Argument format error");
      return 0;
    }

  /* Other privileges are written to a special file and run-time checks
     from certain applications are performed using libvsd.  */
  vsd_priv_load (&list, vc->server_root);
  if (*privilege == '+')
    {
      /* Check admin user has the privilege.  */
      if (strcmp (privilege + 1, "chrtftp") != 0)
	{
	  if (admin_check
	      && vsd_check_privilege (list, "admin", privilege + 1))
	    {
	      io_buffer_store (vc, "-ERR User admin does not have %s privilege",
			       privilege + 1);
	      return 0;
	    }
	}
      else
	{
	  /* Add ftp privilege if user hasn't already got it.  */
	  if (vsd_check_privilege (list, user_name, "ftp"))
	    {
	      if (admin_check
		  && vsd_check_privilege (list, "admin", "ftp"))
		{
		  io_buffer_store (vc, "-ERR User admin does not have ftp privilege");
		  return 0;
		}
	      status = vsd_priv_add_member (list, "ftp", user_name);
	    }
	}

      if (admin_check && strcmp (privilege + 1, "login") == 0)
	{
	  io_buffer_store (vc, "-ERR You cannot grant the login privilege");
	  return 0;
	}

      /* Telnet login accounts can only be assigned if the admin user
         has the `login' privilege.  */
      if (strcmp (privilege + 1, "telnet") == 0
	  && vsd_check_privilege (list, "admin", "login"))
	{
	  io_buffer_store (vc, "-ERR You need the login privilege to assign telnet");
	  return 0;
	}

      status = vsd_priv_add_member (list, privilege + 1, user_name);
    }
  else
    {
      status = vsd_priv_delete_member (list, privilege + 1, user_name);

      /* Automatically remove the chrtftp privilege.  */
      if (strcmp (privilege + 1, "ftp") == 0)
	status = vsd_priv_delete_member (list, "chrtftp", user_name);
    }

  vsd_priv_save (list, vc->server_root);
  vsd_priv_free (list);

  if (status == 0)
    return 1;

  io_buffer_store (vc, "-ERR Privilege modification for user %s failed: %m",
		   user_name);

  return 0;
}

void priv_modify (struct connection *vc, int argc, char *argv[])
{
  struct vsd_file_line *pw_file;
  struct passwd pw;

  /* Check user exists.  */
  vsd_passwd_load (&pw_file, vc->server_root);
  if (vsd_getpwnam (pw_file, argv[0], &pw))
    io_buffer_store (vc, "-ERR User %s does not exist", argv[0]);
  else
    common_set_rights (vc, argv[0], argv[1], 1);

  vsd_passwd_free (pw_file);
}

/* Return list of available privileges.  */
void priv_avail (struct connection *vc, int argc, char *argv[])
{
  struct vsd_file_line *priv_file;

  if (vsd_priv_load (&priv_file, vc->server_root))
    io_buffer_store (vc, "-ERR Cannot load privileges database: %m");
  else
    {
      char **privv;
      int privc, x;

      /* Get a vector of privileges available on the virtual server.  */
      privv = vsd_priv_list_privs (priv_file, &privc);
      for (x = 0; x <= privc; x++)
	io_buffer_store (vc, "%s\n", privv[x]);
      vsd_argv_free (privv, privc);
      vsd_priv_free (priv_file);
    }
}

void priv_list (struct connection *vc, int argc, char *argv[])
{
  FILE *pw_stream;
  char **group_membv, **priv_membv;
  int x, group_membc, priv_membc, min_uid, max_uid;
  struct vsd_file_line *priv_file, *passwd_file, *ent;
  struct passwd pw;

  vsd_passwd_load (&passwd_file, vc->server_root);
  if (argc == 1)
    /* Check user exists.  */
    if (vsd_getpwnam (passwd_file, argv[0], &pw))
      {
	io_buffer_store (vc, "-ERR User %s does not exist", argv[0]);
	vsd_passwd_free (passwd_file);
	return;
      }

  vsd_priv_load (&priv_file, vc->server_root);
  vsd_get_uidgid (vc->server_root, &min_uid, &max_uid, NULL, NULL);
  ent = passwd_file;
  do
    {
      ent = vsd_getpwent (ent, &pw);

      if (pw.pw_uid >= min_uid && pw.pw_uid <= max_uid)
	{
	  /* If asking for a specific name then skip any entries that
	     don't match.  */
	  if (argc == 1 && strcmp (pw.pw_name, argv[0]) != 0)
	    continue;
	  
	  group_membv = vsd_priv_get_group_members (vc->server_root,
						    pw.pw_name, &group_membc);
	  priv_membv = vsd_priv_get_members (priv_file, pw.pw_name,
					     &priv_membc);
	  io_buffer_store (vc, "user=%s uid=%d rights=",
			   pw.pw_name, pw.pw_uid);
	  if (! priv_membc && ! group_membc)
	    io_buffer_store (vc, "[none]\n");
	  else
	    {
	      if (priv_membc)
		{
		  for (x = 0; x < priv_membc; x++)
		    if (x == priv_membc - 1 && ! group_membc)
		      io_buffer_store (vc, priv_membv[x]);
		    else
		      io_buffer_store (vc, "%s;", priv_membv[x]);
		  
		  vsd_argv_free (priv_membv, priv_membc);
		}
	      if (group_membc)
		{
		  for (x = 0; x < group_membc; x++)
		    if (x == group_membc - 1)
		      io_buffer_store (vc, group_membv[x]);
		    else
		      io_buffer_store (vc, "%s;", group_membv[x]);
		  
		  vsd_argv_free (group_membv, group_membc);
		}
	      io_buffer_store (vc, "\n");
	    }
	}
    } while (ent != NULL);

  vsd_priv_free (priv_file);
  vsd_passwd_free (passwd_file);
}
