/* User password utility.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/syslog.h>
#include "libvsd.h"

int main (int argc, char *argv[])
{
  int c;
  int status, admin = 0;
  char *username = NULL;
  char *old_password, *new_password = NULL, *new_password2;
  struct passwd *pw;

  openlog ("passwd", LOG_PID | LOG_NOWAIT, LOG_AUTH);

  pw = getpwuid (getuid ());
  /* admin and root users are allowed to change passwords without
     previously verifying the old password.  */
  if (strcmp (pw->pw_name, "admin") == 0 || strcmp (pw->pw_name, "root") == 0)
    admin = 1;

  while ((c = getopt (argc, argv, "-u:p:")) != -1)
    switch (c)
      {
      case 'p':
	new_password = optarg;
	break;
      case 'u':
	username = optarg;
	break;
      case '?':
	fprintf (stderr, "Unknown option `-%c'.\n", optopt);
	return 1;
      default:
	username = optarg;
	break;
      }

  if (! username)
    username = pw->pw_name;

  if (! admin)
    old_password = strdup (getpass ("Old password: "));

  if (! admin || (admin && ! new_password))
    {
      new_password = strdup (getpass ("New password: "));
      new_password2 = getpass ("Re-enter new password: ");

      if (strcmp (new_password, new_password2))
	{
	  fprintf (stderr, "They don't match; try again.\n");
	  return 1;
	}
    }

  status = vsd_chpass ("/", username, admin, old_password, new_password);
  if (status == 4)
    {
      printf ("passwd: supplied password incorrect\n");
      syslog (LOG_NOTICE, "supplied password for user %s incorrect", username);
    }
  else if (status == 0)
    syslog (LOG_NOTICE, "password changed for user %s", username);
  else if (status == 6)
    {
      printf ("passwd: user %s does not exist", username);
      syslog (LOG_NOTICE, "user %s does not exist", username);
    }
  else if (status == 7)
    {
      printf ("passwd: cannot change password for user %s\n", username);
      syslog (LOG_NOTICE, "attempted to change password for user %s",
	      username);
    }
  else
    {
      printf ("passwd: internal error: status = %d\n", status);
      syslog (LOG_NOTICE, "internal error: status = %d, user = %s\n",
	      status, username);
    }

  return status;
}
