/* Change entries in the user password file.
   Copyright (c) 1999, 2000 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* TODO:
   Add shadow password support.  */

#include <errno.h>
/* We require these to import the crypt and strdup function declarations.  */
#define __USE_XOPEN
#define __USE_XOPEN_EXTENDED
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include "libvsd.h"

int vsd_passwd_load (struct vsd_file_line **list, const char *server_root)
{
  return vsd_load_file (list, server_root, "/etc/passwd");
}

int vsd_passwd_save (struct vsd_file_line *list, const char *server_root)
{
  return vsd_save_file (list, server_root, "/etc/passwd");
}

void vsd_passwd_free (struct vsd_file_line *list)
{
  vsd_free_file (list);
}

/* Return 1 if a match for `gid' is found in `line'. Return 0 if not.  */
static int match (struct vsd_file_line *line, int uid)
{
  int id;

  for (; line; line = line->next)
    {
      char *p = line->line;
      int i = 0;
      /* Search for the UID.  */
      while (*p && i != 2)
	if (*p++ == ':')
	  i++;

      /* Convert to a number.  */
      id = strtol (p, NULL, 10);
      if (id == uid)
	return 1;
    }

  return 0;
}

/* If `uid' is not 0 then check it is unique.  If it is unique then return
   `uid', otherwise return 0.
   If `uid' is 0 then find a unique user id and return that value. If
   there are no spare entries then return 1.  */
static uid_t unique_uid (struct vsd_file_line *file, gid_t uid,
			 uid_t min, uid_t max)
{
  int unique = (uid) ? 1 : 0;

  /* Search through the passwd file.  Start with the lowest group id and
     keep searching until we either exceed `max' or find something unique.  */
  if (unique)
    return (match (file, uid)) ? 0 : uid;

  for (uid = min; uid <= max; uid ++)
    if (! match (file, uid))
      return uid;
  return 1;
}

/* Delete a `user_name' from the user password file.
   Return 0 on success.
   Return 2 if the parameters are invalid.
   Return 6 if the user does not exist.
   Return 7 if the uid is out of range.
   Return 10 on file error.
   Return 13 if the username contains invalid characters.  */
int vsd_userdel (const char *server_root, const char *user_name)
{
  struct vsd_file_line *file, *line;
  struct passwd pw;
  uid_t min, max;

  if (vsd_valid_username (user_name))
    return 13;

  if (vsd_passwd_load (&file, server_root))
    return 10;

  /* Match the line and the user.  */
  line = vsd_file_find_entry (file, user_name);
  if (line == NULL)
    {
      vsd_passwd_free (file);
      return (errno == ENOENT) ? 6 : 1;
    }

  vsd_passwd_decode (line, &pw);
  /* Ensure the user account that we are deleting is within the
     virtual server range.  */
  vsd_get_uidgid (server_root, &min, &max, NULL, NULL);
  if (pw.pw_uid < min || pw.pw_uid > max)
    {
      vsd_passwd_free (file);
      return 7;
    }

  /* Delete the user.  */
  file = vsd_file_delete_line (file, line);
  /* Save.  */
  vsd_passwd_save (file, server_root);
  vsd_passwd_free (file);

  /* Delete the user from all groups.  */
  vsd_group_load (&file, server_root);
  vsd_group_remove_user (file, user_name);
  vsd_group_save (file, server_root);
  vsd_group_free (file);
  return 0;
}

/* Create a new user, called `user_name'.
   Return 0 on success.
   Return 2 if the parameters are invalid.
   Return 4 if the uid is already in use.
   Return 6 if the specified group does not exist.
   Return 9 if the user name is already in use.
   Return 10 on file error.
   Return 14 if there are no more uids.
   Return 13 if the username contains invalid characters.  */
int vsd_useradd (const char *server_root,
		 const char *user_name, const char *homedir,
		 const char *initial_group, const char *additional_groups,
		 const char *shell, uid_t *uid, const char *gecos)
{
  struct vsd_file_line *file, *line, *grpfile;
  uid_t min, max;
  dstring_t ds;
  struct group gr;

  if (vsd_valid_username (user_name))
    return 13;

  if (vsd_passwd_load (&file, server_root))
    return 10;

  /* Check `user_name' is unique.  */
  line = vsd_file_find_entry (file, user_name);
  if (line)
    {
      vsd_passwd_free (file);
      return 9;
    }
  else if (errno == EINVAL)
    {
      vsd_passwd_free (file);
      return 1;
    }

  /* Check uid range and validity.  */
  vsd_get_uidgid (server_root, &min, &max, NULL, NULL);
  if (*uid < min || *uid > max)
    {
      /* uid is outside the range available to the virtual server
         or the user wants us to allocate a uid.  */
      *uid = unique_uid (file, 0, min, max);
      if (*uid == 1)
	return 14;
    }
  else
    if (unique_uid (file, *uid, min, max) != *uid)
      {
	/* User specified a uid and it is not unique.  */
	vsd_passwd_free (file);
	return 4;
      }

  /* Check `initial_group' exists.  */
  vsd_group_load (&grpfile, server_root);
  line = vsd_groupgetnam (grpfile, initial_group);
  if (line == NULL)
    {
      vsd_passwd_free (file);
      vsd_group_free (grpfile);
      return 6;
    }
  vsd_group_decode (line, &gr);

  ds = dstring_new (64);
  dstring_fmt_append (ds, "%s:*:%d:%d:%s:%s:%s",
		      user_name, *uid, gr.gr_gid,
		      (gecos) ? gecos : user_name,
		      (homedir) ? homedir : "",
		      (shell) ? shell : "");

  vsd_group_struct_free (&gr);
  /* Add user to the group.  */
  vsd_group_add_user (line, user_name);

  /* A comma seperated list of groups that user is to be a member of.  */
  if (additional_groups)
    {
      /* Make a temporary copy of additional_groups.  */
      char *t, *s;

      s = strdup (additional_groups);
      do
	{
	  t = strsep (&s, ",");
	  if (t != NULL)
	    {
	      struct vsd_file_line *groupline = vsd_groupgetnam (grpfile, t);
	      if (groupline == NULL)
		{
		  /* The user specified a group that doesn't exist.  */
		  vsd_passwd_free (file);
		  vsd_group_free (grpfile);
		  return 6;
		}
	      vsd_group_add_user (groupline, user_name);
	    }
	} while (t != NULL);
      free (s);
    }

  vsd_group_save (grpfile, server_root);
  vsd_group_free (grpfile);

  /* Add user to passwd file.  */
  vsd_file_insert_line (vsd_file_tail (file), dstring_ptr (ds));
  vsd_passwd_save (file, server_root);
  vsd_passwd_free (file);
  dstring_free (ds);
  return 0;
}

struct passwd *vsd_passwd_decode (struct vsd_file_line *line,
				  struct passwd *pw)
{
  char *p = line->line;
  char buf[12];

  /* Get user name.  */
  pw->pw_name = (char *) malloc (string_lenc (p, ':') + 1);
  string_cpyc (pw->pw_name, p, ':');
  p += string_lenc (p, ':') + 1;

  /* Get password.  */
  pw->pw_passwd = (char *) malloc (string_lenc (p, ':') + 1);
  string_cpyc (pw->pw_passwd, p, ':');
  p += string_lenc (p, ':') + 1;

  /* Get uid.  */
  string_cpyc (buf, p, ':');
  pw->pw_uid = (uid_t) atoi (buf);
  p += string_lenc (p, ':') + 1;

  /* Get gid.  */
  string_cpyc (buf, p, ':');
  pw->pw_gid = (gid_t) atoi (buf);
  p += string_lenc (p, ':') + 1;

  pw->pw_gecos = (char *) malloc (string_lenc (p, ':') + 1);
  string_cpyc (pw->pw_gecos, p, ':');
  p += string_lenc (p, ':') + 1;

  pw->pw_dir = (char *) malloc (string_lenc (p, ':') + 1);
  string_cpyc (pw->pw_dir, p, ':');
  p += string_lenc (p, ':') + 1;

  pw->pw_shell = (char *) malloc (strlen (p) + 1);
  strcpy (pw->pw_shell, p);

  return pw;
}

/* Modify an user account, called `user_name'.
   Return 0 on success.
   Return 2 if the parameters are invalid.
   Return 4 if the uid is already in use.
   Return 6 if the specified group does not exist.
   Return 7 if the user does not exist.
   Return 9 if the user name is already in use.
   Return 10 on file error.
   Return 11 if the uid is out of range.
   Return 14 if there are no more uids.
   Return 13 if the username contains invalid characters.  */
int vsd_usermod (const char *server_root, const char *user_name,
		 const char *login_name, const char *homedir,
		 const char *initial_group, const char *additional_groups,
		 const char *shell, uid_t uid, const char *gecos)
{
  struct vsd_file_line *file, *userline, *groupline, *grpfile;
  uid_t min, max;
  dstring_t ds;
  struct group gr;
  struct passwd pw;

  if (vsd_valid_username (user_name))
    return 13;

  if (vsd_passwd_load (&file, server_root))
    return 10;

  /* Check `user_name' is unique.  */
  userline = vsd_file_find_entry (file, user_name);
  if (! userline && (errno == EINVAL || errno == ENOENT))
    {
      vsd_passwd_free (file);
      return (errno == ENOENT) ? 7 : 1;
    }

  vsd_passwd_decode (userline, &pw);
  if (uid != -1)
    {
      /* Check uid range and validity.  */
      vsd_get_uidgid (server_root, &min, &max, NULL, NULL);
      if (uid < min || uid > max)
	{
	  /* uid is outside the range available to the virtual server
	     or the user wants us to allocate a uid.  */
	  uid = unique_uid (file, 0, min, max);
	  if (uid == 1)
	    return 14;
	}
      else
	if (unique_uid (file, uid, min, max) != uid)
	  {
	    /* User specified a uid and it is not unique.  */
	    vsd_passwd_free (file);
	    return 4;
	  }
    }
  else
    {
      uid = pw.pw_uid;
      /* Ensure the user account that we are deleting is within the
	 virtual server range.  */
      vsd_get_uidgid (server_root, &min, &max, NULL, NULL);
      if (uid < min || uid > max)
	{
	  vsd_passwd_free (file);
	  return 7;
	}
    }

  /* Do /etc/group manipulation.  */
  vsd_group_load (&grpfile, server_root);

  /* If `initial_group' is set, then we need to remove the user from
     all other groups and then re-add it to the new group list.  */
  if (initial_group || additional_groups)
    {
      vsd_group_remove_user (grpfile, user_name);
    }

  if (initial_group)
    {
      groupline = vsd_groupgetnam (grpfile, initial_group);
      if (groupline == NULL)
	{
	  /* The user specified an initial group that doesn't exist.  */
	  vsd_passwd_free (file);
	  vsd_group_free (grpfile);
	  return 6;
	}
      /* Add user to initial_group.  */
      vsd_group_add_user (groupline, user_name);
      vsd_group_decode (groupline, &gr);
    }

  /* A comma seperated list of groups that user is to be a member of.  */
  if (additional_groups)
    {
      /* Make a temporary copy of additional_groups.  */
      char *t, *s;

      s = strdup (additional_groups);
      do
	{
	  t = strsep (&s, ",");
	  if (t != NULL)
	    {
	      groupline = vsd_groupgetnam (grpfile, t);
	      if (groupline == NULL)
		{
		  /* The user specified a group that doesn't exist.  */
		  vsd_passwd_free (file);
		  vsd_group_free (grpfile);
		  return 6;
		}
	      vsd_group_add_user (groupline, user_name);
	    }
	} while (t != NULL);
      free (s);
    }
      

  if (login_name)
    vsd_group_rename_user (grpfile, user_name, login_name);

  /* Recreate the passwd entry.  */
  ds = dstring_new (64);
  dstring_fmt_append (ds, "%s:%s:%d:%d:%s:%s:%s",
		      (login_name) ? login_name : user_name,
		      pw.pw_passwd,
		      uid, (initial_group) ? gr.gr_gid : pw.pw_gid,
		      (gecos) ? gecos : pw.pw_gecos,
		      (homedir) ? homedir : pw.pw_dir,
		      (shell) ? shell : pw.pw_shell);

  if (initial_group && ! login_name)
    {
      vsd_group_struct_free (&gr);
      /* Add user to the group.  */
      vsd_group_add_user (groupline, user_name);
    }

  if (login_name || initial_group)
    vsd_group_save (grpfile, server_root);
  vsd_group_free (grpfile);

  /* Delete old user information.  */
  file = vsd_file_delete_line (file, userline);
  /* Add user to passwd file.  */
  vsd_file_insert_line (vsd_file_tail (file), dstring_ptr (ds));
  vsd_passwd_save (file, server_root);
  vsd_passwd_free (file);
  dstring_free (ds);
  return 0;
}


/* Operations on an open password database.  */

/* Search for `name' in the password database. Return 0 on success,
   -1 on failure, or 1 if name does not exist.  Fill `pw' with
   the passwd entries for user `name', if found.  */
int vsd_getpwnam (struct vsd_file_line *file, const char *name,
		  struct passwd *pw)
{
  struct vsd_file_line *userline = vsd_file_find_entry (file, name);

  if (! userline && errno == ENOENT)
    return 1;

  if (! userline && errno == EINVAL)
    return -1;

  vsd_passwd_decode (userline, pw);
  return 0;
}

/* Search for `name' in the password database. Return 0 on success,
   -1 on failure, or 1 if name does not exist.  Fill `pw' with
   the passwd entries for user `name', if found.  */
int vsd_getpwuid (struct vsd_file_line *file, uid_t uid,
		  struct passwd *pw)
{
  struct vsd_file_line *ent = file;

  do
    {
      ent = vsd_getpwent (ent, pw);
      if (pw->pw_uid == uid)
	return 0;
    } while (ent != NULL);

  return 1;
}

/* Return the next user entry in the password database.  The appropriate
   fields in `pw' will be filled with the user entry information.
   Returns a pointer to the next password database line, or NULL if
   there are no further entries.  */
struct vsd_file_line *vsd_getpwent (struct vsd_file_line *ent,
				    struct passwd *pw)
{
  if (ent == NULL)
    return NULL;

  vsd_passwd_decode (ent, pw);
  return vsd_file_walk (ent);
}
