/* Dynamic string handling.
   Copyright (c) 1999 Idaya Ltd.
   Contributed by Nick Burrett <nick@dsvr.net>

   This file is part of the Virtual Server Administrator (FreeVSD)

   FreeVSD is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   FreeVSD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with FreeVSD; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include "libvsd.h"

/* Create a new dynamic string and reserve `size' bytes for it.  */
dstring_t dstring_new (int size)
{
  dstring_t ds = (dstring_t) malloc (sizeof (struct dstring));

  ds->buffer_size = size;
  ds->length = 0;
  ds->s = (char *) malloc (size);
  ds->s[0] = '\0';

  return ds;
}

/* Free memory allocated to a dynamic string.  */
void dstring_free (dstring_t ds)
{
  free (ds->s);
  free (ds);
}

/* Append a NULL terminated string, `s', to the dynamic string.  */
void dstring_append (dstring_t ds, const char *s)
{
  int len = strlen (s);

  if (ds->buffer_size - ds->length <= len)
    {
      ds->buffer_size += len + 32;
      ds->s = (char *) realloc (ds->s, ds->buffer_size);
    }

  strcpy (ds->s + ds->length, s);
  ds->length += len;
}

/* Append a NULL terminated formatted string, `fmt', to the dynamic
   string.  */
void dstring_fmt_append (dstring_t ds, const char *fmt, ...)
{
  va_list ap;
  int len, freespace;

  va_start (ap, fmt);
  freespace = ds->buffer_size - ds->length;
  len = vsnprintf (ds->s + ds->length, freespace, fmt, ap);
  if (len >= freespace)
    {
      /* Not enough free space. Allocate some more.  */
      ds->buffer_size += len + 2 - freespace;
      ds->s = (char *) realloc (ds->s, ds->buffer_size);
    }

  len = vsprintf (ds->s + ds->length, fmt, ap);
  ds->length += len;
  va_end (ap);
}

/* Duplicate the dstring in a malloc buffer and return a pointer to it.  */
char *dstring_yield (dstring_t ds)
{
  return strdup (ds->s);
}

/* Return a pointer to the dynamic string.  */
char *dstring_ptr (dstring_t ds)
{
  return ds->s;
}



