#include "dkbfconfig.h"

static void
config_err (char *string)
{
  fprintf (stderr, "dkbf Fatal Configuration Error: %s\n", string);
  MPI_Abort (MPI_COMM_WORLD, 0);
  exit (0);
}
void 
prompt_to_continue(char *err_msg)
{
  int answer;
  sleep(1);
  fflush(stdout);
  fflush(stderr);
  printf("%s: Continue [Y|n]? ",err_msg);
  fflush(stdout);
  do{
    answer = getchar();
    if(answer == 0x0A) return;
    if((answer == 'Y') || (answer=='y')) return;
    if((answer == 'N') || (answer=='n'))
      {
	config_err("user requested termination");
      }
    while(getchar() != 0x0A);
    printf("Continue [Y|n]?");
    fflush(stdout);
  }while(1);
}
static void
init_config (CONFIG * Config)
{
  memset (Config->pwlist, '\0', sizeof (Config->pwlist));
  memset (Config->masterlog, '\0', sizeof (Config->masterlog));
  memset (Config->slavelog, '\0', sizeof (Config->slavelog));
  memset (Config->cracklog, '\0', sizeof (Config->cracklog));
  memset (Config->jobname, '\0', sizeof (Config->jobname));
  memset (Config->characterset, '\0', sizeof (Config->characterset));
  memset (Config->mustcontain, '\0', sizeof (Config->mustcontain));
  Config->algorithm = 1;
  Config->charsetnum = 3;
  Config->charsetlen = 36;
  Config->loop = 0;
  Config->logginglevel = 1;
  Config->pwdump = 1;
  Config->blockexpiration = 3600;
  Config->blocksize = 10000000;
  Config->slavepriority = 0;
  Config->sleepstagger = 1;
  Config->passwordfilter = 0;
  Config->minuppercase = 0;
  Config->maxuppercase = NOMAX;
  Config->minlowercase = 0;
  Config->maxlowercase = NOMAX;
  Config->minpasswdlen = 0;
  Config->maxpasswdlen = 14;
  Config->minnumeric = 0;
  Config->maxnumeric = NOMAX;
  Config->minvowels = 0;
  Config->maxvowels = NOMAX;
  Config->minconsonants = 0;
  Config->maxconsonants = NOMAX;
  Config->minspecchar = 0;
  Config->maxspecchar = NOMAX;
  Config->maxrepeat_count = NOMAX;
  Config->maxdupes_count = NOMAX;
  Config->passfilt_dll = 0;
  Config->slavelogging = 0;
  return;
}
void
get_charset (CONFIG * Config)
{
  switch (Config->charsetnum)
    {
    case 0:
      strcpy (Config->characterset, "AB");	
      break;
    case 1:
      strcpy (Config->characterset, "ABCDEFGHIJKLMNOPQRSTUVWXYZ");
      break;
    case 2:
      strcpy (Config->characterset, "abcdefghijklmnopqrstuvwxyz");
      break;
    case 3:
      strcpy (Config->characterset, "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789");
      break;
    case 4:
      strcpy (Config->characterset, "abcdefghijklmnopqrstuvwxyz0123456789");
      break;
    case 5:
      strcpy (Config->characterset,
	      "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
      break;
    case 6:
      strcpy (Config->characterset,
	      "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789" \
	      "~!@#$%^&*()-_=+\\|[{]};:'\"<,.> ");
      break;
    case 7:
      strcpy (Config->characterset,
	      "abcdefghijklmnopqrstuvwxyz0123456789`~!@#$%^&*()" \
	      "-_=+\\|[{]};:'\"<,.> ");
      break;
    case 8:
      strcpy (Config->characterset,
	      "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"\
	      "!@#$%^&*() ");
      break;
    case 9:
      strcpy (Config->characterset,
	      "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"\
	      "!@#$%^&*()-_=+[{]}\\|;:'\",<.>/? ");
      break;
    case 10:
      strcpy (Config->characterset, "ABCDEFGHIJ");	
      break;
    case 11:
      strcpy (Config->characterset, "0123456789");	
      break;
    }				
  return;
}

/* return number of letters that are duplicated;
   this is the function; similar to EliminateByRule
   but this is case sensitive and is not "INLINED" */
int 
count_dupes(char *brute)
{
  int   x     = 0;
  int   c     = 0;
  int   flag  = 0;
  int   len   = 0;
  int   dupes = 0;
  char  tchar;
  char  *string = NULL;

  len = strlen(brute) + 1;
  string = (char *)malloc(sizeof(char)*len);
  if(!string)
    {
      config_err ("Out of memory in function: count_dupes()\n");
    }
  memset(string, '\0', len);
  strcpy(string, brute);

  for (x = 0; x < len; x++)
    {
      if (string[x] != '\0')
	{
	  tchar = string[x];
	}
      else
	{
	  continue;
	}
      string[x] = '\0';
      for (c = x + 1; c < len; c++)
	{
	  if (string[c] == tchar)
	    {
	      if (flag)
		dupes++;
	      string[c] = '\0';
	      flag = 0;
	    }
	}
      flag = 1;
    }
  free(string);
  string = NULL;
  return dupes;
}
int
charset_has_lowercase (CONFIG * Config)
{
  int    x;

  for (x = 0; x < Config->charsetlen; x++)
    {
      if (islower (Config->characterset[x]))
	return 1;
    }

  return 0;
}
int 
passfilt_compliant(char *characterset)
{
  int len     = strlen(characterset);
  int x       = 0;
  int lower   = 0;
  int upper   = 0;
  int numeric = 0;
  int special = 0;

  for(x=0;x<len;x++)
    {
      if(!lower)
	{
	  if(islower(characterset[x])) 
	    {
	      lower++;
	      continue;
	    }
	}
      if(!upper)
	{
	  if(isupper(characterset[x]))
	    {
	      upper++;
	      continue;
	    }
	}	      
      if(!numeric)
	{
	  if(isdigit(characterset[x]))
	    { 
	      numeric++;
	      continue;
	    }
	}
      if(!special)
	{
	  if(!isalnum(characterset[x]))
	    {
	      special++;
	      continue;
	    }
	}
    }
  if(special + numeric + upper + lower > 2 ) return 1;
  return 0;
}
int
validate_config (CONFIG * Config)
{
  time_t    now;

  if (strlen (Config->pwlist) < 1)
    {
      config_err ("No Password list specified\n");
    }

  if (strlen (Config->masterlog) < 1)
    {
      fprintf (stderr, "No Masterlog file specified - using defaults\n");
      strcpy (Config->masterlog, "master.log");
    }

  if (strlen (Config->slavelog) < 1)
    {
      fprintf (stderr, "No Slavelog file specified - using defaults\n");
      strcpy (Config->slavelog, "slave.log");
    }

  if (strlen (Config->cracklog) < 1)
    {
      fprintf (stderr, "No Crack log file specfied - using defaults\n");
      strcpy (Config->cracklog, "crack.log");
    }

  if (strlen (Config->jobname) < 1)
    {
      fprintf (stderr, "No Jobname specified - using timestamp"\
	       "as filename prefix.\n");
      time (&now);
      sprintf (Config->jobname, "dkbfjob_%d", now);
    }
  if ((Config->charsetnum < 0 || Config->charsetnum > 11) &&
      strlen (Config->characterset) < 2)
    {
      config_err ("Character set number is out of range and no valid custom" \
		  " character set is specified\n");
    }
  if (strlen (Config->characterset) < 2 && Config->charsetnum == 0)
    {
      config_err ("No valid character set specified. 2 characters required\n");
    }
  if (strlen (Config->characterset) > 1 && Config->charsetnum != 0)
    {
      fprintf (stderr, "Custom character set to be used even though\n" \
	       "charsetnum was set in the ini file\n");
    }
  if (strlen (Config->characterset) < 1 && Config->charsetnum != 0)
    {
      get_charset (Config);
    }
  Config->charsetlen = strlen (Config->characterset);
  if (Config->charsetlen > MAX_CHARACTERSET_SIZE)
    {
      config_err ("Character set is too long\n");
    }
  if (Config->algorithm < LM_NT_DUMP || Config->algorithm > NT_SNIFF)
    {
      config_err ("Algorithm selected is not supported\n");
    }
  if (Config->algorithm == LM_NT_DUMP || 
      Config->algorithm == LM_SNIFF || 
      Config->algorithm == LM_NT_SNIFF)
    {
      if (charset_has_lowercase (Config))
	{
	  config_err
	    ("Selected algorithm does not allow for lowercase letters\n");
	}
    }
  if (Config->algorithm == NT_DUMP || Config->algorithm == NT_SNIFF)
    {
      if(!charset_has_lowercase(Config))
	{
	  fprintf(stderr,"The selected algorithm is case sensitive\n");
	  prompt_to_continue("No lower case letters in characterset.");
	}
    }
  if(count_dupes(Config->characterset))
    {
      config_err("Characterset contains duplicate characters\n");
    }
  if (Config->algorithm == LM_NT_DUMP || 
      Config->algorithm == NT_DUMP || 
      Config->algorithm == LM_DUMP )
    {
      Config->pwdump = 1;
    }
  else
    {
      Config->pwdump = 0;
    }
  if (Config->logginglevel < 0 || Config->logginglevel > 2)
    {
      fprintf (stderr,
	       "Logging not set within range of 0-2 - using defaults\n");
      Config->logginglevel = 1;
    }
  if (Config->blockexpiration < 1)
    {
      fprintf (stderr, "Block expiration set too low - using defaults\n");
      Config->blockexpiration = 10000;
    }
  if (Config->blocksize < 1)
    {
      fprintf (stderr, "Block size set too low - using defaults\n");
      Config->blocksize = 1000000;
    }
  if (Config->slavepriority < 0)
    {
      fprintf (stderr, "Slave process scheduling priority increased.\n" \
	       "This may require higher privileges or fail.\n" \
	       "Defaults will be used on fail.\n");
    }
  if (Config->slavepriority > 19)
    {
      fprintf (stderr,
	       "Slave process scheduling priority decreased too low." \
	       "Defaults will be used\n");
      Config->slavepriority = 19;
    }
  if (Config->sleepstagger != 0 && Config->sleepstagger != 1)
    {
      config_err ("Sleepstagger option not set within range 0-1\n");
    }
  if (Config->passwordfilter != 0 && Config->passwordfilter != 1)
    {
      config_err ("Password filtering must be either 0 or 1 regardless of " \
		  "algorithm\n");
    }
  if (Config->passwordfilter && 
      (Config->algorithm == LM_NT_DUMP || Config->algorithm == LM_DUMP))
    {
      fprintf (stderr, "Password filtering will be ignored for this " \
	       "algorithm\n");
      Config->passwordfilter=0;
    }
  if ((Config->passwordfilter) && 
      (Config->algorithm == NT_DUMP || 
       Config->algorithm == LM_NT_SNIFF || 
       Config->algorithm == NT_SNIFF ||
       Config->algorithm == LM_SNIFF))
    {
      if (Config->minpasswdlen > Config->maxpasswdlen)
	config_err ("Min and Max password length setting is invalid\n");

      if ((Config->maxpasswdlen < Config->minspecchar)  ||
	  (Config->maxpasswdlen < Config->minuppercase) ||
	  (Config->maxpasswdlen < Config->minlowercase) ||
	  (Config->maxpasswdlen < Config->minconsonants)||
	  (Config->maxpasswdlen < Config->minvowels)    ||
	  (Config->maxpasswdlen < Config->minnumeric))
	config_err("Max password length or Minimum setting is invalid\n");

      if (Config->minuppercase > Config->maxuppercase)
	config_err ("Min and Max uppercase characters setting is invalid\n");

      if ((Config->maxuppercase == 0) &&
	  ((Config->algorithm == LM_NT_SNIFF) || 
	   (Config->algorithm == LM_SNIFF)))
	config_err("Uppercase characters are required. Check maxuppercase\n");

      if (Config->minlowercase > Config->maxlowercase)
	config_err ("Min and Max lowercase characters setting is invalid\n");
      
      if (Config->minnumeric > Config->maxnumeric)
	config_err ("Min and Max numeric characters  setting is invalid\n");

      if (Config->minvowels > Config->maxvowels)
	config_err ("Min and Max vowels  setting is invalid\n");

      if (Config->minconsonants > Config->maxconsonants)
	config_err ("Min and Max consonants setting is invalid\n");

      if (Config->minspecchar > Config->maxspecchar)
	config_err ("Min and Max special characters setting is invalid\n");

      if (Config->maxrepeat_count < 0 && Config->maxrepeat_count > NOMAX)
	config_err ("Maximum repeat count setting is invalid\n");

      if (Config->maxdupes_count < 0 && Config->maxdupes_count > NOMAX)
	config_err ("Maximum duplicates count setting is invalid\n");

      if (Config->passfilt_dll != 0 && Config->passfilt_dll != 1)
	config_err ("Passfilt_dll setting is invalid\n");

      if ((Config->passfilt_dll) && 
	  ((Config->algorithm == LM_SNIFF) || 
	   (Config->algorithm == LM_NT_SNIFF)))
	{
	  config_err("Passfilt_dll cannot be applied to this algorithm");
	}
      else
	{
	  if(Config->minpasswdlen < 6)
	    {
	      config_err("Passfilt.dll settings require a minimum 6-character"\
			 " password length\n");
	    }
	  if(!passfilt_compliant(Config->characterset))
	    {
	      config_err("The chosen character set does not have 3 of the"\
			 " following character groups  required by passfilt.dll"\
			 " settings: uppercase, lowercase,"\
			 " numeric, or special.\n");
	    }	  
	}
    }
  else
    {
      fprintf (stderr,
	       "Password filtering does not apply.  Rules not validated\n");
      Config->passwordfilter = 0;
    }
  if(Config->algorithm == NT_DUMP || Config->algorithm == LM_NT_SNIFF ||
     Config->algorithm == LM_SNIFF || Config->algorithm == NT_SNIFF)
    {
      if(Config->maxpasswdlen > 14)
	{
	  Config->maxpasswdlen = 14;
	  fprintf(stderr, "Password lengths will be limited to 14 characters.\n");
	}
    }
  if(Config->algorithm == LM_NT_DUMP || Config->algorithm == LM_DUMP)
  {
    Config->maxpasswdlen = 7;
  }
  if (Config->slavelogging != 0 && Config->slavelogging != 1)
    {
      fprintf (stderr,
	       "Slave logging option invalid. No slave logging will occur\n");
      Config->slavelogging = 0;
    }
  return 0;
}
int
log_job_configurations (CONFIG * Config, FILE * masterlog,
			STATUS * jobstatus)
{
  logger (Config->logginglevel, 0, 1, masterlog, "Job name: %s\n",
	  Config->jobname);

  logger (Config->logginglevel, 0, 1, masterlog, "Total Keys %.0Lf\n",
	  jobstatus->ttl_keys);
  logger (Config->logginglevel, 0, 1, masterlog, "First Key %.0Lf\n",
	  jobstatus->first_key);
  logger (Config->logginglevel, 0, 1, masterlog, "Last Key %.0Lf\n",
	  jobstatus->last_key);

  logger (Config->logginglevel, 0, 1, masterlog, "Total Blocks %.0Lf\n",
	  jobstatus->ttl_keys / Config->blocksize);
  logger (Config->logginglevel, 0, 1, masterlog, "Total Compute Nodes: %d\n",
	  jobstatus->slaves);

  logger (Config->logginglevel, 0, 1, masterlog,
	  "Job has %d initial accounts\n", jobstatus->usercount);

  logger (Config->logginglevel, 0, 1, masterlog, "Password list: %s\n",
	  Config->pwlist);
  logger (Config->logginglevel, 0, 1, masterlog, "Master log: %s\n",
	  Config->masterlog);
  logger (Config->logginglevel, 0, 1, masterlog,
	  "Slave log (base name): %s\n", Config->slavelog);
  logger (Config->logginglevel, 0, 1, masterlog, "Cracked accounts log: %s\n",
	  Config->cracklog);
  logger (Config->logginglevel, 0, 1, masterlog, "Character set is: %s\n",
	  Config->characterset);
  logger (Config->logginglevel, 0, 1, masterlog, "Character set length: %d\n",
	  Config->charsetlen);
  logger (Config->logginglevel, 0, 1, masterlog, "Algorithm is: %d\n",
	  Config->algorithm);

  if (Config->pwdump)
    logger (Config->logginglevel, 0, 1, masterlog,
	    "Password file is from pwdump file\n");
  else
    logger (Config->logginglevel, 0, 1, masterlog,
	    "Password file is from sniffer log\n");

  logger (Config->logginglevel, 0, 1, masterlog,
	  "Blocks will expire in %ld seconds\n", Config->blockexpiration);
  logger (Config->logginglevel, 0, 1, masterlog,
	  "Block size is: %.0Lf keys\n", Config->blocksize);
  logger (Config->logginglevel, 0, 1, masterlog,
	  "Attempting to set slave process to %d priority\n",
	  Config->slavepriority);

  if (Config->sleepstagger)
    logger (Config->logginglevel, 0, 1, masterlog,
	    "Sleep staggering is on\n");
  else
    logger (Config->logginglevel, 0, 1, masterlog,
	    "Sleep staggering is off\n");

  if (Config->passwordfilter && 
      (Config->algorithm == NT_DUMP || Config->algorithm == LM_NT_SNIFF ||
       Config->algorithm == LM_SNIFF || Config->algorithm == NT_SNIFF))
    {
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Password filtering is enabled\n");
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Uppercase Characters: %d min - %d max\n",
	      Config->minuppercase, Config->maxuppercase);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Lowercase Characters: %d min - %d max\n",
	      Config->minlowercase, Config->maxlowercase);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Password Length: %d min - %d max\n",
	      Config->minpasswdlen, Config->maxpasswdlen);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Numeric Characters: %d min - %d max\n",
	      Config->minnumeric, Config->maxnumeric);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Vowel Characters: %d min - %d max\n",
	      Config->minvowels, Config->maxvowels);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Consonants Characters: %d min - %d max\n",
	      Config->minconsonants, Config->maxconsonants);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Special Characters: %d min - %d max\n",
	      Config->minspecchar, Config->maxspecchar);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Maximum repeat count: %d max\n", Config->maxrepeat_count);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Maximum duplicates count: %d max\n", Config->maxdupes_count);
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Maximum repeat count: %d max\n", Config->maxrepeat_count);
      if (Config->passfilt_dll)
	{
	  logger (Config->logginglevel, 0, 1, masterlog,
		  "Passfilt.dll rules will be applied\n");
	}
      else
	{
	  logger (Config->logginglevel, 0, 1, masterlog,
		  "Passfilt.dll rules will NOT be applied\n");
	}
    }
  else
    {
      logger (Config->logginglevel, 0, 1, masterlog,
	      "Password filtering does not apply.  Rules not validated\n");
    }
  fflush(stdout);
  fflush(masterlog);
  prompt_to_continue("Job options validated");
  return 0;
}
int
readini (char *inifile, CONFIG * Config)
{
  FILE    *fp;
  char    line[MAX_LINE];
  char    word[MAX_WORD];
  char    *p;
  int     len;
  int     i;
  int     x;

  if ((fp = fopen (inifile, "r")) == NULL)
    {
      config_err ("Failed to open ini file\n");
    }

  init_config (Config);

  while (fgets (line, MAX_LINE, fp) != NULL)
    {
      /* ignore comments */
      if (line[0] == '#')
	continue;
      /* ignore blank lines */
      if (line[0] == '\n')
	continue;

      /* remove comments on same line;
	 this has been disabled to allow the '#' char in the
	 custom characterset field  */
      /*
      p = strchr (line, '#');
      if (p != NULL)
	*p = '\0';
     */

      /* remove whitespace */
      len = strlen (line);
      for (x = 0, i = 0; x < len; x++)
	{
	  if (isspace (line[x]))
	    continue;
	  line[i++] = line[x];
	}
      line[i] = '\0';
      p = strchr (line, '=');
      if (!p)
	continue;

      /* read lvalue */
      x = p - &line[0];
      strncpy (word, line, x);
      word[x] = '\0';

      /* make lvalue lower case */
      i = x;
      while (--i >= 0)
	word[i] = tolower (word[i]);

      /* move rvalue to Config struct */
      if (strcmp (word, "pwlist") == 0)
	{
	  strcpy (Config->pwlist, (char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "masterlog") == 0)
	{
	  strcpy (Config->masterlog, (char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "slavelog") == 0)
	{
	  strcpy (Config->slavelog, (char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "cracklog") == 0)
	{
	  strcpy (Config->cracklog, (char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "jobname") == 0)
	{
	  strcpy (Config->jobname, (char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "customcharacterset") == 0)
	{
	  strcpy (Config->characterset, (char *) chomp (&line[x + 1]));
	  Config->charsetlen = strlen (Config->characterset);
	  continue;
	}
      if (strcmp (word, "mustcontain") == 0)
	{
	  strcpy (Config->mustcontain, (char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "algorithm") == 0)
	{
	  Config->algorithm = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "charsetnum") == 0)
	{
	  Config->charsetnum = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "loop") == 0)
	{
	  Config->loop = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "logginglevel") == 0)
	{
	  Config->logginglevel = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "blockexpiration") == 0)
	{
	  Config->blockexpiration = atol ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "blocksize") == 0)
	{
	  Config->blocksize = atol ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "slavepriority") == 0)
	{
	  Config->slavepriority = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "sleepstagger") == 0)
	{
	  Config->sleepstagger = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "passwordfilter") == 0)
	{
	  Config->passwordfilter = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minspecchar") == 0)
	{
	  Config->minspecchar = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxspecchar") == 0)
	{
	  Config->maxspecchar = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minuppercase") == 0)
	{
	  Config->minuppercase = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxuppercase") == 0)
	{
	  Config->maxuppercase = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minlowercase") == 0)
	{
	  Config->minlowercase = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxlowercase") == 0)
	{
	  Config->maxlowercase = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minpasswdlen") == 0)
	{
	  Config->minpasswdlen = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxpasswdlen") == 0)
	{
	  Config->maxpasswdlen = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minnumeric") == 0)
	{
	  Config->minnumeric = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxnumeric") == 0)
	{
	  Config->maxnumeric = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minvowels") == 0)
	{
	  Config->minvowels = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxvowels") == 0)
	{
	  Config->maxvowels = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "minconsonants") == 0)
	{
	  Config->minconsonants = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxconsonants") == 0)
	{
	  Config->maxconsonants = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxrepeat_count") == 0)
	{
	  Config->maxrepeat_count = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "maxdupes_count") == 0)
	{
	  Config->maxdupes_count = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "passfilt_dll") == 0)
	{
	  Config->passfilt_dll = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      if (strcmp (word, "slavelogging") == 0)
	{
	  Config->slavelogging = atoi ((char *) chomp (&line[x + 1]));
	  continue;
	}
      fprintf (stderr, "%s",word);
      prompt_to_continue("- is an unrecognized or unsupported option");
    }/* end while not EOF */
  validate_config (Config);
  return 0;
}
MPI_Datatype *
CommitConfig ()
{
  MPI_Datatype    *pConfig;
  MPI_Datatype    *lConfig;
  CONFIG          Config;
  int             base;
  int             x;

  int bl[CONFIG_NBLOCKS] = { 
    CONFIG_MAXWORDSIZE,	        /* pwlist */
    CONFIG_MAXWORDSIZE,		/* masterlog */
    CONFIG_MAXWORDSIZE,		/* slavelog */
    CONFIG_MAXWORDSIZE,		/* cracklog */
    CONFIG_MAXWORDSIZE,		/* jobname */
    CONFIG_MAXSTRINGSIZE,	/* characterset */
    CONFIG_MAXSTRINGSIZE,	/* mustcontain */
    CONFIG_ONEINTSIZE,		/* algorithm */
    CONFIG_ONEINTSIZE,		/* charsetnum */
    CONFIG_ONEINTSIZE,		/* charsetlen */
    CONFIG_ONEINTSIZE,		/* loop */
    CONFIG_ONEINTSIZE,		/* logginglevel */
    CONFIG_ONEINTSIZE,		/* pwdump */
    CONFIG_ONELONGSIZE,		/* blockexpiration */
    CONFIG_ONEDOUBLESIZE,	/* blocksize */
    CONFIG_ONEINTSIZE,		/* slavepriority */
    CONFIG_ONEINTSIZE,		/* sleepstagger */
    CONFIG_ONEINTSIZE,		/* passwordfilter */
    CONFIG_ONEINTSIZE,		/* minuppercase */
    CONFIG_ONEINTSIZE,		/* maxuppercase */
    CONFIG_ONEINTSIZE,		/* minlowercase */
    CONFIG_ONEINTSIZE,		/* maxlowercase */
    CONFIG_ONEINTSIZE,		/* minpasswdlen */
    CONFIG_ONEINTSIZE,		/* maxpasswdlen */
    CONFIG_ONEINTSIZE,		/* minnumeric */
    CONFIG_ONEINTSIZE,		/* maxnumeric */
    CONFIG_ONEINTSIZE,		/* minvowels */
    CONFIG_ONEINTSIZE,		/* maxvowels */
    CONFIG_ONEINTSIZE,		/* minconsonants */
    CONFIG_ONEINTSIZE,		/* maxconsonants */
    CONFIG_ONEINTSIZE,		/* minspecchar */
    CONFIG_ONEINTSIZE,		/* maxspecchar */
    CONFIG_ONEINTSIZE,		/* maxrepeat_count */
    CONFIG_ONEINTSIZE,		/* maxdupes_count */
    CONFIG_ONEINTSIZE,		/* passfilt_dll */
    CONFIG_ONEINTSIZE           /* slavelogging */
  };				

  MPI_Aint disp[CONFIG_NBLOCKS];

  MPI_Datatype types[CONFIG_NBLOCKS] = { 
    MPI_CHAR,	                /* pwlist */
    MPI_CHAR,			/* masterlog */
    MPI_CHAR,			/* slavelog */
    MPI_CHAR,			/* cracklog */
    MPI_CHAR,			/* jobname */
    MPI_CHAR,			/* characterset */
    MPI_CHAR,			/* mustcontain */
    MPI_INT,			/* algorithm */
    MPI_INT,			/* charsetnum */
    MPI_INT,			/* charsetlen */
    MPI_INT,			/* loop */
    MPI_INT,			/* logginglevel */
    MPI_INT,			/* pwdump */
    MPI_LONG,			/* blockexpiration */
    MPI_LONG_DOUBLE,		/* blocksize */
    MPI_INT,			/* slavepriority */
    MPI_INT,			/* sleepstagger */
    MPI_INT,			/* passwordfitler */
    MPI_INT,			/* minuppercase */
    MPI_INT,			/* maxuppercase */
    MPI_INT,			/* minlowercase */
    MPI_INT,			/* maxlowercase */
    MPI_INT,			/* minpasswdlen */
    MPI_INT,			/* maxpasswdlen */
    MPI_INT,			/* minnumeric */
    MPI_INT,			/* maxnumeric */
    MPI_INT,			/* minvowels */
    MPI_INT,			/* maxvowels */
    MPI_INT,			/* minconsonants */
    MPI_INT,			/* maxconsonants */
    MPI_INT,			/* minspecchar */
    MPI_INT,			/* maxspecchar */
    MPI_INT,			/* maxrepeat_count */
    MPI_INT,			/* maxdupes_count */
    MPI_INT,			/* passfilt_dll */
    MPI_INT,			/* slavelogging */
  };

  lConfig = (MPI_Datatype *) malloc (sizeof (MPI_Datatype));

  MPI_Address (&Config, disp);
  MPI_Address (&Config.masterlog, &disp[1]);
  MPI_Address (&Config.slavelog, &disp[2]);
  MPI_Address (&Config.cracklog, &disp[3]);
  MPI_Address (&Config.jobname, &disp[4]);
  MPI_Address (&Config.characterset, &disp[5]);
  MPI_Address (&Config.mustcontain, &disp[6]);
  MPI_Address (&Config.algorithm, &disp[7]);
  MPI_Address (&Config.charsetnum, &disp[8]);
  MPI_Address (&Config.charsetlen, &disp[9]);
  MPI_Address (&Config.loop, &disp[10]);
  MPI_Address (&Config.logginglevel, &disp[11]);
  MPI_Address (&Config.pwdump, &disp[12]);
  MPI_Address (&Config.blockexpiration, &disp[13]);
  MPI_Address (&Config.blocksize, &disp[14]);
  MPI_Address (&Config.slavepriority, &disp[15]);
  MPI_Address (&Config.sleepstagger, &disp[16]);
  MPI_Address (&Config.passwordfilter, &disp[17]);
  MPI_Address (&Config.minuppercase, &disp[18]);
  MPI_Address (&Config.maxuppercase, &disp[19]);
  MPI_Address (&Config.minlowercase, &disp[20]);
  MPI_Address (&Config.maxlowercase, &disp[21]);
  MPI_Address (&Config.minpasswdlen, &disp[22]);
  MPI_Address (&Config.maxpasswdlen, &disp[23]);
  MPI_Address (&Config.minnumeric, &disp[24]);
  MPI_Address (&Config.maxnumeric, &disp[25]);
  MPI_Address (&Config.minvowels, &disp[26]);
  MPI_Address (&Config.maxvowels, &disp[27]);
  MPI_Address (&Config.minconsonants, &disp[28]);
  MPI_Address (&Config.maxconsonants, &disp[29]);
  MPI_Address (&Config.minspecchar, &disp[30]);
  MPI_Address (&Config.maxspecchar, &disp[31]);
  MPI_Address (&Config.maxrepeat_count, &disp[32]);
  MPI_Address (&Config.maxdupes_count, &disp[33]);
  MPI_Address (&Config.passfilt_dll, &disp[34]);
  MPI_Address (&Config.slavelogging, &disp[35]);

  /*make offsets relative to begining of struct */
  base = disp[0];
  for (x = 0; x < CONFIG_NBLOCKS; x++)
    disp[x] -= base;

  /* define new MPI data type */
  MPI_Type_struct (CONFIG_NBLOCKS, bl, disp, types, lConfig);
  MPI_Type_commit (lConfig);

  pConfig = (MPI_Datatype *) malloc (sizeof (MPI_Datatype));

  if (pConfig == NULL)
    {
      fprintf (stderr, "malloc failed in CONFIG commit\n");
      MPI_Abort (MPI_COMM_WORLD, 0);
    }

  pConfig = lConfig;
  return pConfig;
}
int
free_rates (unsigned long *noderates)
{
  free (noderates);
  noderates = NULL;

  return 0;
}
int
clear_rates (STATUS * status)
{
  int x;

  for (x = 0; x < status->size; x++)
    {
      status->noderates[x] = 0;
    }

  return 0;
}
static int
show_rates (STATUS * status, FILE * masterlog, CONFIG *Config)
{
  int  x;

  status->ttl_rate = 0.0;

  for (x = 1; x < status->size; x++)
    {
      status->ttl_rate += status->noderates[x];
    }

  status->percent_done = status->keys_done / status->ttl_keys * 100;

  logger (Config->logginglevel, 1, 1, masterlog,
	  "RATES: Total: %.2f, Avg: %.2f, Done: %2.2f\%\n",
	  status->ttl_rate, status->ttl_rate / (status->size - 1),
	  status->percent_done);

  return 0;
}
int
track_keyrate (int node, BLOCK * s_block, STATUS * jobstatus, 
	       FILE * masterlog, CONFIG *Config)
{
  if ((node > jobstatus->size) || (node <= 0))
    return 1;

  jobstatus->noderates[node] = s_block->rate;

  jobstatus->keys_done += s_block->size;

  show_rates (jobstatus, masterlog, Config);

  return 0;
}
int 
set_first_last_key (STATUS * jobstatus, CONFIG *Config)
{
  jobstatus->last_key = get_ttlkeys (Config);
  jobstatus->first_key = get_first_key (Config);
  jobstatus->ttl_keys = jobstatus->last_key - jobstatus->first_key;
  return 0;
}
int
initialize_status (STATUS * jobstatus, CONFIG * Config, int size)
{
  time(&jobstatus->begin);
  jobstatus->noderates = NULL;
  jobstatus->percent_done = 0.0;
  jobstatus->size = size;
  jobstatus->ttl_rate = 0.0;
  jobstatus->ttl_keys = 0;
  jobstatus->first_key = 0;
  jobstatus->last_key = 0;
  jobstatus->keys_done = 0;
  jobstatus->slaves = size -1 ;
  jobstatus->usercount = 0;
  jobstatus->noderates =
    (unsigned long *) malloc (sizeof (unsigned long) * size);

  if (jobstatus->noderates == NULL)
    return 1;

  clear_rates (jobstatus);

  set_first_last_key (jobstatus, Config);

  return 0;
}
int 
dkbf_getopts(int argc, char **argv, CONFIG *Config)
{
  char   inifile[MAX_WORD];
  int    x;

  memset (inifile, '\0', MAX_WORD);
  for (x = 0; x < argc; x++)
    {
      if (argv[x][0] == '-' && argv[x][1] == 'i')
	{
	  strcpy (inifile, argv[++x]);
	  break;
	}
    }

  if (strlen (inifile) > 0)
    {
      if (fileexists (inifile))
	{
	  readini (inifile, Config);
	}
    }
  else
    {
      readini ("dkbf.ini", Config);
    }
  return 0;
}
/*
int main()
{
  CONFIG Config;
  readini("dkbf_debug.ini", &Config);
  printf("%s\n", Config.characterset);
  return(0);
}
*/
