/* files.c - File i/o functions
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* External functions and variables */
extern char *text;
GtkWidget *textarea;
extern stats *text_stats;
void apply_global_key_text(void);
char *apply_key_text(void);

/* Load default fts */
float *load_slft_std(char *filename) { 
    FILE *inf;
    int i;
    float *slft = malloc(((int) 'Z' + 1) * sizeof(float));
    
    inf = fopen(filename, "r");
    if (!inf) {
	fprintf(stderr, "Error opening slft file: %s\n", filename);
	exit(1);
    }
    for (i = 'A'; i <= 'Z'; i++) {
	if(!(fscanf(inf, "%f", &(slft[i])) == 1))
	    fprintf(stderr, "Error in slft file: %s\n", filename);
    }
    fclose(inf);
    return slft;
}

float *load_bift_std(char *filename) { 
    FILE *inf;
    int i, j;
    float *bift = malloc(((int) 'Z' + 1) * ((int) 'Z' + 1) * sizeof(float));
    
    inf = fopen(filename, "r");
    if (!inf) {
	fprintf(stderr, "Error opening bift file: %s\n", filename);
	exit(1);
    }
    for (i = 'A'; i <= 'Z'; i++) {
 	for (j = 'A'; j <= 'Z'; j++) {
	    if(!(fscanf(inf, "%f", &((bift + 26 * i)[j])) == 1))
		fprintf(stderr, "Error in bift file: %s\n", filename);
	}
    }
    fclose(inf);
    return bift;
}

float *load_trift_std(char *filename) { 
    FILE *inf;
    int i, j, k;
    float *trift = malloc(((int) 'Z' + 1) * ((int) 'Z' + 1) * ((int) 'Z' + 1) * sizeof(float));
    
    inf = fopen(filename, "r");
    if (!inf) {
	fprintf(stderr, "Error opening trift file: %s\n", filename);
	exit(1);
    }
    for (i = 'A'; i <= 'Z'; i++) {
 	for (j = 'A'; j <= 'Z'; j++) {
	    for (k = 'A'; k <= 'Z'; k++) {
		if(!(fscanf(inf, "%f", &((trift + 26 * 26 * i + 26 * j)[k])) == 1))
		    fprintf(stderr, "Error in trift file: %s\n", filename);
	    }
	}
    }
    fclose(inf);
    return trift;
}

/* Saving defaults fts */
void do_save_slft(float *slft) {
    FILE *outfile;
    int i;
    outfile = fopen(DEFAULT_SLFT, "w");
    if (!outfile) {
	fprintf(stderr, "Error: Cannot open data/slft.dat\n");
	exit(1);
    }
    for (i = (int) 'A'; i <= (int) 'Z'; i++) {
	fprintf(outfile, "%.8f\n", slft[i]); 
    }
    fclose(outfile);
}

void do_save_bift(float *bift) {
    FILE *outfile;
    int i, j;
    outfile = fopen(DEFAULT_BIFT, "w");
    if (!outfile) {
	fprintf(stderr, "Error: Cannot open data/bift.dat\n");
	exit(1);
    }
    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    fprintf(outfile, "%.8f\n", (bift + 26 * i)[j]); 
	}
    }
    fclose(outfile);
}

void do_save_trift(float *trift) {
    FILE *outfile;
    int i, j, k;
    outfile = fopen(DEFAULT_TRIFT, "w");
    if (!outfile) {
	fprintf(stderr, "Error: Cannot open data/trift.dat\n");
	exit(1);
    }
    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    for (k = 'A'; k <= 'Z'; k++) {
		fprintf(outfile, "%.8f\n", (trift + 26 * 26 * i + 26 * j)[k]); 
	    }
	}
    }
    fclose(outfile);
}

/* Load file into source */
int do_load (char *filename) {
    FILE *inf;
    int c;
    int i = 0;
    inf = fopen(filename, "r");
    if (!inf)
	return FALSE;
    
    free(text);
    text = malloc(MAXTEXT * sizeof(char));
    while ( (c = getc(inf)) != EOF)
	text[i++] = c;
    text[i] = '\0';
    apply_global_key_text();

    fclose(inf);
    return TRUE;
}

/* Save transformed text */
int do_save (char *filename) {
    FILE *otf;
    int i = 0;
    char *buf;
    otf = fopen(filename, "w");
    if (!otf)
	return FALSE;

    buf = GLOBAL_TRANSFORMED_TEXT;
    while (buf[i] != '\0')
	putc(buf[i++], otf);
    g_free(buf);

    fclose(otf);
    return TRUE;
}


