/* ui_init.c - GUI Initialisation
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* Global variables */
GtkWidget *textarea;
char fromchar = 'A';
char tochar = 'a';
GtkWidget *key_controls;

/* External functions and data */
extern void init_global_key(void);
extern GtkWidget *global_key_labels[26];
extern key global_key;
extern stats *text_stats;
extern float *slft_std;
extern float *bift_std;
extern float *trift_std;
extern char *text;
stats *make_stats(char *text);
float *load_slft_std(char *filename);
float *load_bift_std(char *filename);
float *load_trift_std(char *filename);

/* Callbacks */
void cb_exit_crank (GtkWidget *widget, gpointer gdata);
void cb_invert (GtkWidget *widget, gpointer gdata);
void cb_shift_L (GtkWidget *widget, gpointer gdata);
void cb_shift_R (GtkWidget *widget, gpointer gdata);
void cb_reverse (GtkWidget *widget, gpointer gdata);
void cb_complete (GtkWidget *widget, gpointer gdata);
void cb_clear (GtkWidget *widget, gpointer gdata);
void cb_change (GtkWidget *widget, gpointer gdata);
void cb_start_open (GtkWidget *widget, gpointer gdata);
void cb_start_save (GtkWidget *widget, gpointer gdata);
void cb_change_from_select(GtkWidget *widget, gpointer gdata);
void cb_change_to_select(GtkWidget *widget, gpointer gdata);
void cb_close_key_controls (GtkWidget *widget, gpointer gdata);
void cb_open_key_controls (GtkWidget *widget, gpointer gdata);
void cb_open_prepro_dialog(void);
void cb_open_stats_dialog (GtkWidget *widget, gpointer gdata);
void cb_break (GtkWidget *widget, gpointer gdata);

/* Make the menu bar */
static GtkItemFactoryEntry menu_items[] = {
      { "/_File",                NULL,         NULL, 0, "<Branch>" },
      { "/File/_Open",         "<control>O", cb_start_open, 0, NULL },
      { "/File/_Save View As", "<control>S", cb_start_save, 0, NULL },
      { "/File/_Exit",         "<control>Q", cb_exit_crank, 0, NULL },
};

static GtkWidget *make_main_menu(GtkWidget *window) {
    GtkWidget *mainmenu;
    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    gint nmenu_items = sizeof (menu_items) / sizeof(menu_items[0]);
    accel_group = gtk_accel_group_new();
    item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_group);
    gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);
    gtk_window_add_accel_group (GTK_WINDOW(window), accel_group);
    mainmenu = gtk_item_factory_get_widget(item_factory, "<main>");
    return mainmenu;
}

GtkWidget *make_key_display(void) {
    int i;
    GtkWidget *table;
    GtkWidget *label;
    char buf[2];
    table = gtk_table_new(2, 26, TRUE);
    for (i=0; i < 26; i++) {
        
        sprintf(buf, "%c", i + 'A');
	label = gtk_label_new(buf);
        gtk_misc_set_alignment(GTK_MISC(label), 0.5, 1.0);
        gtk_table_attach_defaults(GTK_TABLE(table), label, i, i + 1, 0, 1);
        gtk_widget_show(label);
        
        sprintf(buf, "%c", global_key[i + 'A']);
	label = gtk_label_new(buf);
        gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.0);
        gtk_table_attach_defaults(GTK_TABLE(table), label, i, i + 1, 1, 2);
        gtk_widget_show(label);
        global_key_labels[i] = label;
    }
    return table;
}

static void addButton(GtkWidget *table, const char *text, void callback(GtkWidget *widget, gpointer), int la, int ra, int ta, int ba) {
    GtkWidget *btn;
    btn = gtk_button_new_with_label(text);
    gtk_signal_connect(GTK_OBJECT(btn), "pressed", GTK_SIGNAL_FUNC(callback), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), btn, la, ra, ta, ba);
    gtk_widget_show(btn);
}

GtkWidget *make_key_controls(void) {
    GtkWidget *table;
    GtkWidget *option_menu;
    GtkWidget *o_menu;
    GtkWidget *menu_item;
    GtkWidget *keywindow;
    GSList *group = NULL;
    int i;
    char buf[2];

    table = gtk_table_new(3, 3, FALSE);
    
    addButton(table, "Invert", cb_invert,  0, 1, 0, 1);
    addButton(table, "Shift <<", cb_shift_L, 1, 2, 0, 1);
    addButton(table, "Shift >>", cb_shift_R, 2, 3, 0, 1);
    addButton(table, "Reverse", cb_reverse, 0, 1, 1, 2);
    addButton(table, "Complete", cb_complete, 1, 2, 1, 2);
    addButton(table, "Clear", cb_clear, 2, 3, 1, 2);
    addButton(table, "Change: ", cb_change, 0, 1, 2, 3);

    /* Change FROM menu */
    option_menu = gtk_option_menu_new();
    o_menu = gtk_menu_new();
    for (i = 0; i < 26; i++) {
	sprintf(buf, "%c", i + 'A');
        menu_item = gtk_radio_menu_item_new_with_label(group, buf);
	group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
        gtk_menu_append(GTK_MENU(o_menu), menu_item);
        gtk_signal_connect(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cb_change_from_select), GINT_TO_POINTER((i + 'A')));
        gtk_widget_show(menu_item);
    }
    gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), o_menu);
    gtk_widget_show(o_menu);
    gtk_widget_show(option_menu);
    gtk_table_attach_defaults(GTK_TABLE(table), option_menu, 1, 2, 2, 3);
       
    /* Change TO menu */
    group = NULL;
    option_menu = gtk_option_menu_new();
    o_menu = gtk_menu_new();
    for (i = 0; i < 26; i++) {
	sprintf(buf, "%c", i + 'a');
        menu_item = gtk_radio_menu_item_new_with_label(group, buf);
	group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
        gtk_menu_append(GTK_MENU(o_menu), menu_item);
        gtk_signal_connect(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cb_change_to_select), GINT_TO_POINTER((i + 'a')));
        gtk_widget_show(menu_item);
    }
    sprintf(buf, "Blank");
    menu_item = gtk_radio_menu_item_new_with_label(group, buf);
    group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
    gtk_menu_append(GTK_MENU(o_menu), menu_item);
    gtk_signal_connect(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cb_change_to_select), GINT_TO_POINTER((0)));
    gtk_widget_show(menu_item);
    gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), o_menu);
    gtk_widget_show(o_menu);
    gtk_widget_show(option_menu);
    gtk_table_attach_defaults(GTK_TABLE(table), option_menu, 2, 3, 2, 3);
   
    keywindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(keywindow), "Key Controls");
    gtk_window_set_policy(GTK_WINDOW(keywindow), FALSE, FALSE, TRUE);
    gtk_container_add(GTK_CONTAINER(keywindow), table);
    gtk_widget_show(table);

    return keywindow;
}

void build_gui(void) {
    GtkWidget *window;
    GtkWidget *vbox1;
    GtkWidget *menu;
    GtkWidget *hbox1;
    GtkWidget *hbox2;
    GtkWidget *vscrollbar;
    GtkWidget *keydisplay;
    GtkWidget *keybtn;
    GtkWidget *preprobtn;
    GtkWidget *statsbtn;
    GtkWidget *crackbtn;

    /* Main window */
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "Crank v" VERSION);
    gtk_widget_set_usize(window, 600, 200);
    gtk_signal_connect(GTK_OBJECT(window), "delete_event", GTK_SIGNAL_FUNC(cb_exit_crank), NULL);
 
    /* Main vertical box */
    vbox1 = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(window), vbox1); 
    gtk_widget_show(vbox1);
   
    /* Main menu */
    menu = make_main_menu(window);
    gtk_box_pack_start(GTK_BOX(vbox1), menu, FALSE, TRUE, 0);
    gtk_widget_show(menu);
   
    /* Key bar */
    hbox1 = gtk_hbox_new(FALSE, 0);
    keydisplay = make_key_display();
    gtk_box_pack_start(GTK_BOX(hbox1), keydisplay, FALSE, TRUE, 0);
    gtk_widget_show(keydisplay);

    /* Create key window (but leave hidden) */
    key_controls = make_key_controls();
    gtk_signal_connect(GTK_OBJECT(key_controls), "delete_event", GTK_SIGNAL_FUNC(cb_close_key_controls), NULL);

    /* Key Control Button */
    keybtn = gtk_button_new_with_label("Key Controls");
    gtk_box_pack_start(GTK_BOX(hbox1), keybtn, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(keybtn), "pressed", GTK_SIGNAL_FUNC(cb_open_key_controls), NULL);
    gtk_widget_show(keybtn);

    /* Pre Pro Button */
    preprobtn = gtk_button_new_with_label("Preprocess source");
    gtk_box_pack_start(GTK_BOX(hbox1), preprobtn, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(preprobtn), "pressed", GTK_SIGNAL_FUNC(cb_open_prepro_dialog), NULL);
    gtk_widget_show(preprobtn);

    /* Stats Button */
    statsbtn = gtk_button_new_with_label("Statistics");
    gtk_box_pack_start(GTK_BOX(hbox1), statsbtn, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(statsbtn), "pressed", GTK_SIGNAL_FUNC(cb_open_stats_dialog), NULL);
    gtk_widget_show(statsbtn);

    /* Crack Button */
    crackbtn = gtk_button_new_with_label("Break");
    gtk_box_pack_start(GTK_BOX(hbox1), crackbtn, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(crackbtn), "pressed", GTK_SIGNAL_FUNC(cb_break), NULL);
    gtk_widget_show(crackbtn);

    gtk_widget_show(hbox1);
    gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, TRUE, 0);

    /* Text area */
    hbox2 = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox1), hbox2, TRUE, TRUE, 0);
    gtk_widget_show(hbox2);
    textarea = gtk_text_new(NULL, NULL);
    gtk_text_set_editable(GTK_TEXT(textarea), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox2), textarea, TRUE, TRUE, 0);
    
    /* Initial text */
    text = malloc(MAXTEXT * sizeof(char)); 
    sprintf(text, START_TEXT);
    text_stats = make_stats(text);
    gtk_text_insert(GTK_TEXT(textarea), NULL, NULL, NULL, text,  strlen(text));

    vscrollbar = gtk_vscrollbar_new (GTK_TEXT(textarea)->vadj);
    gtk_box_pack_start(GTK_BOX(hbox2), vscrollbar, FALSE, FALSE, 0);

    gtk_widget_show (vscrollbar);
    gtk_widget_show(textarea);
    gtk_widget_show(window);
}
