#include "dkbfslave.h"

int
slave (CONFIG Config, int rank)
{
  FILE *slavelog = stderr;

  int x = 0;
  int usercount = 0;
  int request = 0;
  int first_round = 0;

  struct user_struct *masterarray = NULL;
  struct user_struct *head = NULL;

  char slavefilewrank[MAX_STRING];

  MPI_Status status;
  BLOCK *s_block = NULL;

  MPI_Datatype *pBlock;
  MPI_Datatype *pUserInfo;

  /* malloc memory for block info */
  s_block = malloc_block ();
  if (!s_block)
    announce_shutdown ();

  /* set up slave logging */
  if (Config.slavelogging)
    {

      /*create filename to avoid collision on SMP boxen */
      sprintf (slavefilewrank, "%d_%s", rank, Config.slavelog);
      slavelog = jobfopen (Config.jobname, slavefilewrank, "a");

      /* report on slave logging status */
      if (slavelog)
	{
	  fprintf (slavelog, "Slave %d logging to %s\n", rank,
		   slavefilewrank);
	}
      else
	{
	  perror ("lmslave.c - logging");
	  fprintf (stderr, "%d failed to open logs\n", rank);
	  slavelog = stderr;
	}
    }

  /* adjust slave scheduling and priority */
  set_slavepriority (&Config, slavelog, rank);

  pBlock = CommitBlock ();
  pUserInfo = CommitUserInfo ();

  while (1)
    {
      /* request next block */
      request = REQUEST_BLOCK;
      MPI_Send (&request, 1, MPI_INT, MASTER, MPI_MSG_TAG, MPI_COMM_WORLD);

      /* get how many existing accounts */
      MPI_Recv (&usercount, 1, MPI_INT, MASTER, MPI_ANY_TAG, MPI_COMM_WORLD,
		&status);

      /* stagger nodes by sleeping according to rank
	 this should only occur on the first round;
       */
      if(first_round && Config.sleepstagger)
	{
	  sleep (rank);
	  first_round++;
	}

      /* no more accounts left - exit */
      if (!usercount)
	break;

      /* create memory for accounts  */
      masterarray =
	(struct user_struct *) malloc (sizeof (struct user_struct) *
				       usercount);
      /* exit node if out of memory */
      if (!masterarray)
	break;

      /* clear memory for accounts */
      for (x = 0; x < usercount; x++)
	memset (&masterarray[x], '\0', sizeof (struct user_struct));

      /* get the user info and put in the array we just malloc'd */
      MPI_Recv (masterarray, usercount, *pUserInfo,
		MASTER, MPI_ANY_TAG, MPI_COMM_WORLD, &status);

      /* some intialization for each block */
      head = init_linked_list ();

      /* convert contingous  master array into a double linked list */
      cnvt_array_to_list (head, masterarray, usercount);

      /* get block size and block starting point */
      MPI_Recv (s_block, 1, *pBlock, MASTER,
		MPI_ANY_TAG, MPI_COMM_WORLD, &status);

      /* log block */
      if (Config.slavelogging)
	logger (Config.logginglevel, 3, 0, slavelog,
		"%d got block %.0Lf; SIZE %.0Lf; USERCOUNT: %d\n",
		rank, s_block->start, s_block->size, usercount);

      /* no more blocks left */
      if (s_block->start == -1)
	break;

      head = brute (s_block, usercount, head, slavelog, &Config);

      /* report results of block */
      request = REPORT_BLOCK;
      MPI_Send (&request, 1, MPI_INT, MASTER, MPI_MSG_TAG, MPI_COMM_WORLD);
      MPI_Send (s_block, 1, *pBlock, MASTER, MPI_MSG_TAG, MPI_COMM_WORLD);

      if (Config.slavelogging)
	{
	  logger (Config.logginglevel, 3, 0, slavelog,
		  "%d: completed block %.0Lf; SIZE %.0Lf\n",
		  rank, s_block->start, s_block->size);
	}

      /* clean up after every block */
      free (masterarray);
      masterarray = NULL;
      free_struct_list (head);
      head = NULL;
    }				/*end while */

  /*tell master slave is shutting down */
  announce_shutdown ();

  if (Config.slavelogging)
    {
      logger (Config.logginglevel, 2, 0, slavelog,
	      "slave %d: exiting  normally\n", rank);
    }

  /* free memory and clean up  */
  MPI_Type_free (pBlock);
  free (pBlock);
  pBlock = NULL;

  MPI_Type_free (pUserInfo);
  free (pUserInfo);
  pUserInfo = NULL;

  fflush (stdout);

  free (s_block);
  s_block = NULL;

  if (masterarray)
    {
      free (masterarray);
      masterarray = NULL;
    }
  if (head)
    {
      free_struct_list (head);
      head = NULL;
    }

  fclose (slavelog);

  return (0);
}
struct user_struct *
brute (BLOCK * s_block, int usercount, struct user_struct *head,
       FILE * slavelog, CONFIG * Config)
{
  long double stop = 0;
  register long double keynum = 0;
  char brute_str[MAXPASSWDLEN];

  memset(brute_str, '\0', MAXPASSWDLEN);

  /* compute stopping point for this block */
  stop = s_block->start + s_block->size;
  time (&s_block->begin_t);
  
  /* get the first key for this block */
  block_to_string (brute_str, s_block->start, Config);

  /* start cracking */
  for (keynum = s_block->start; keynum < stop; keynum++)
    {
      /* check for filtering and eliminate */
      if (Config->passwordfilter)
	{
	  if (EliminateByRule (Config, brute_str))
	    continue;
	}

      /* each routine is responsible for 
         - hashing the brute string
         - comparing the hash to all accounts/users
         - comparison routine returns 1 after finding a match
         and 0 if no matches; should be used to trigger account removal 
         - notifying the master of any matches
         - removing any fully broken accounts
         - return value is 0 if all accounts broken; 1 otherwise
       */
      switch (Config->algorithm)
	{
	case LM_NT_DUMP:
	case LM_DUMP:
	  head = lanman_nt_dump (head, slavelog, Config, brute_str);
	  break;
	case NT_DUMP:
	  /* works */
	  head = nt_dump (head, slavelog, Config, brute_str);
	  break;
	case LM_NT_SNIFF:
	case LM_SNIFF:
	  /* works */
	  head = lanman_nt_sniff(head, slavelog, Config, brute_str); 
	  break;
	case NT_SNIFF:
	  /* works */
	  head = nt_sniff (head, slavelog, Config, brute_str);
	  break;
	}
      if (!head) break;

      get_next_brute_string (brute_str, Config->characterset);
    }/* end for/cracking loop */

  benchmark_block (s_block);
  return head;
}
struct user_struct *
lanman_nt_dump (struct user_struct *head, FILE * slavelog,
		CONFIG * Config, char *brute_str)
{
  char half_hash[8];
  struct user_struct *index = head;

  /* convert key (brute string) to hashed string */
  half_lanman (half_hash, brute_str);

  /* compare hash string to all */
  if (brute_routine (head, half_hash, brute_str,Config) == 1)
    {
      if(Config->algorithm == LM_NT_DUMP)
	{
	  /* get NT password  - case sensitive Unicode */
	  nt_ify_list (index);
	}

      /* remove broken accounts */
      head = prune_list (index);
    }
  return head;
}
struct user_struct *
nt_dump (struct user_struct *head, FILE * slavelog,
	 CONFIG * Config, char *brute_str)
{
  struct user_struct *index = head;
  int uni_len;
  char hold[MAXPASSWDLEN*2];
  char p16[] = { 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00};

  memset(hold, '\0', MAXPASSWDLEN*2);

  /* convert to unicode and return correct
     unicode length for md4 */
  uni_len = PutUniCode (hold, brute_str);
  md4hash (hold, p16, uni_len);

  /* do comparison of hash (hold) with all  */
  if (compare_nthash (Config, index, p16, brute_str))
    {
      head = prune_list (index);
    }
  return head;
}
struct user_struct *
lanman_nt_sniff (struct user_struct *head, FILE * slavelog,
		 CONFIG * Config, char *brute_str)
{
  struct user_struct *index = head;

  if (lm_check_sniff (index, brute_str, Config) == 1)
    {
      if(Config->algorithm == LM_NT_SNIFF)
	{
	  /* get NT passwords */
	  nt_ify_list (index);	  
	}

      /* remove broken accounts */
      head = prune_list (index);
    }
  return head;
}
struct user_struct *
nt_sniff (struct user_struct *head, FILE * slavelog,
	  CONFIG * Config, char *brute_str)
{
  struct user_struct *index = head;
  int uni_len;
  char hold[NTPASSWDLEN * 2];
  char p16[16] = { 0x00, 0x00, 0x00, 0x00,
		   0x00, 0x00, 0x00, 0x00,
		   0x00, 0x00, 0x00, 0x00,
		   0x00, 0x00, 0x00, 0x00};

  /* convert to unicode and return correct
     unicode length for md4 */
  uni_len = PutUniCode (hold, brute_str);
  md4hash (hold, p16, uni_len);

  /* do comparison of hash (hold) with all in Ustruct */
  if (compare_nthash (Config, index, p16, brute_str))
    {
      head = prune_list (index);
    }
  return head;
}
int
compare_nthash (CONFIG * Config, struct user_struct *head,
		char *nthash, char *brute_str)
{
  int ret = 0;
  struct user_struct *index = head;
  char pre_ntresp[21];
  char response[24];

  if (Config->pwdump)
    {
      while (index != NULL)
	{
	  if (memcmp (index->nthashb, nthash, 16) == 0)
	    {
	      strcpy (index->ntpasswd, brute_str);
	      index->ntdone = 1;
	      index->lmdone = 1;
	      report_account (index);
	      index = index->next;
	      ret = 1;
	    }
	  else
	    index = index->next;
	}
    }
  else
    {
      while (index != NULL)
	{
	  memset (pre_ntresp, '\0', 21);
	  memcpy (pre_ntresp, nthash, 16);
	  E_P24 (pre_ntresp, index->server_chall, response);

	  if (memcmp (index->ntresp_b, response, 24) == 0)
	    {
	      memcpy (index->nthashb, nthash, 16);
	      strcpy (index->ntpasswd, brute_str);
	      index->ntdone = 1;
	      index->lmdone = 1;
	      report_account (index);
	      index = index->next;
	      ret = 1;
	    }
	  else
	    index = index->next;
	}
    }
  return ret;
}
int
benchmark_block (BLOCK * s_block)
{

  time (&s_block->end_t);

  /* add benchmark info to block structure */
  s_block->elapsed = s_block->end_t - s_block->begin_t;
  if (s_block->elapsed > 0)
    {
      s_block->rate = s_block->size / s_block->elapsed;
    }
  else
    {
      /* prevents division by 0 - esp. during debugging */
      s_block->rate = 0;
    }
  return 0;
}
int
announce_shutdown ()
{
  int request = 0;

  /* for whatever reason we are done */
  request = NODE_SHUTDOWN;
  //printf ("node shutting down\n");
  //fflush (stdout);
  /* send shutdown notice to master */
  MPI_Send (&request, 1, MPI_INT, MASTER, MPI_MSG_TAG, MPI_COMM_WORLD);

  return 0;
}
