/*******************************************************************
slurpie.c: distributes tasks among slurps (slurp.c).
   Copyright (C) 1999. HaPpY (katami@hotmail.com).
   This source code is protected under the GPL. Read the
   README file for details.
********************************************************************/

/////////////////////////////
#include "master.h"
#include "shared.h"

/////////////////////////////
static const u_char *CL_HELP_MENU =
   "%s\n"	// title
   "syntax: master <options> <passwd file>\n"
   "\noptions:\n"
   "-h                    this help screen\n"
   "-v                    turn verbose mode on\n"
   "-l                    turn logging off\n"
   "-p <file name>        pre-made dictionary comparison\n"
   "-g <a|A|7|?|x> n m    generated dictionary comparison\n"
   "                      a: consider lower case characters\n"
   "                      A: consider upper case characters\n"
   "                      7: consider digits\n"
   "                      ?: consider punctuation\n"
   "                      x: consider the extended character set\n"
   "                      n: minimum password length\n"
   "                      m: maximum password length\n"
   "\n"
   "example: slurpie -p words.txt -g a? 6 8 passwd.txt\n"
   "will compare the passwd file: passwd.txt with words.txt and a generated "
   "list of every combination of lower case letters and punctuation with "
   "a length ranging from 6 to 8. results are stored in passwd.txt.log.\n\n";


/////////////////////////////
enum
{
   TASK_FREE,
   TASK_ACTIVE,
   TASK_DONE
};


/////////////////////////////
typedef struct
{
   u_int cl,
      cl_gdict,
      gd_cset_size,
      minplen,
      maxplen;
   u_char dictfile_name[MAX_STR_LEN];
   FILE *pwdfile;
   FILE *logfile;
   PwdInfo curpwd;
   u_int gtask_level, ptask_level;
} Config;


/////////
typedef struct
{
   u_int isactive;
   u_char state[32];
} Task;

typedef struct
{
   u_char curstate[32];
   u_int curplen;
} GDict;

///
typedef struct
{
   u_int next,
      num_done;
} PDict;


/////////////////////////////
static Config cfg;
static GDict gd;
static PDict pd;


/////////////////////////////
void print( const u_char *str, ... );


///////////////////////////// message_handler
int load_pwdfile_field( FILE *fp, char *s )
{
   int i;
   for( i=0; (s[i]=fgetc(fp))!=EOF && s[i]!=':' && s[i]!='\n'; i++ );
   s[i]=0;
   return !feof(fp) && i;
}

int more_work( void )
{
   while( !feof(cfg.pwdfile) )
   {
      if( !load_pwdfile_field(cfg.pwdfile,cfg.curpwd.user))
         return 0;
      if( !load_pwdfile_field(cfg.pwdfile,cfg.curpwd.passwd))
         return 0;
      if( cfg.curpwd.passwd[0] && cfg.curpwd.passwd[0] != '*' )
      {
         while( !feof(cfg.pwdfile) && (fgetc(cfg.pwdfile))!='\n' );
         cfg.curpwd.isactive = 0;
         memset( &pd, 0, sizeof(PDict) );
         memset( &gd, 0, sizeof(GDict) );
         gd.curplen = cfg.minplen;
         return 1;
      }
      while( !feof(cfg.pwdfile) && (fgetc(cfg.pwdfile))!='\n' );
   };
   return 0;
}

//////
u_char *send_init( u_char *sendbuf, u_int *send_len )
{
   u_char *pstr=sendbuf;

   *(pstr++) = (u_char)MSG_INIT;		// packet id
   *(pstr++) = (u_char)cfg.cl;
   if( cfg.cl & CL_PDICT )
   {
     *(pstr++) = (u_char)cfg.ptask_level = *num_nodes*PTASK_LEVEL;
      pstr += strcpy_len( pstr, cfg.dictfile_name );
   }
   if( cfg.cl & CL_GDICT )
   {
      *(pstr++) = (u_char)cfg.gtask_level;
      *(pstr++) = (u_char)cfg.cl_gdict;
   }
   *(pstr++) = 0;
   *send_len = pstr-sendbuf;

   return sendbuf;
}


//////
u_char *send_work_pdict( u_char *sendbuf, u_int *send_len )
{
   u_char *pstr=sendbuf;

   *(pstr++) = MSG_DO_PDICT;
   *(pstr++) = pd.next++;
   pstr += strcpy_len(pstr,cfg.curpwd.user);
   pstr += strcpy_len(pstr,cfg.curpwd.passwd);
   *send_len = pstr-sendbuf;

   return sendbuf;
}


///////
u_char *send_work_gdict( u_char *sendbuf, u_int *send_len )
{
   u_char *pstr=sendbuf;
   static u_int i, j;

   *(pstr++) = MSG_DO_GDICT;
   *(pstr++) = gd.curplen;
   
   memcpy(pstr,gd.curstate,gd.curplen);
   pstr += gd.curplen;
   pstr += strcpy_len(pstr,cfg.curpwd.user);
   pstr += strcpy_len(pstr,cfg.curpwd.passwd);
   *send_len = pstr-sendbuf;

// get next state
   for( j=i=(cfg.gtask_level<gd.curplen)?cfg.gtask_level:gd.curplen; j<=i; j++ )
      if( ++gd.curstate[j]>=cfg.gd_cset_size )
      {
         gd.curstate[j]=0;
         i++;
      }

   if(i>=gd.curplen)
   {
      i=cfg.gtask_level;
      memset( gd.curstate, 0, 32 );
      gd.curplen++;
   }

   return sendbuf;
}


////////////////
u_char *msg_proc( u_char *node_msg, u_int *send_len )
{
   static u_int alldone = 0;
   static u_char sendbuf[MAX_PACKET_SIZE];

   if(alldone)
      return 0;

   if(!strcmp(FIRST_SEND,node_msg))
      return send_init(sendbuf,send_len);

   switch(node_msg[0])
   {
   case MSG_SUCCESS:
      print("password found for %s: %s\n",
         node_msg+1,
         node_msg+strcpy_len(sendbuf,node_msg) );
      if( !strcmp(cfg.curpwd.user,node_msg+1) )
         if( !more_work() )
            break;
      sendbuf[0] = MSG_INIT;
      return msg_proc(sendbuf,send_len);
   case MSG_INIT:
      if(!cfg.curpwd.isactive)
         print("cracking: %s %s\n", cfg.curpwd.user, cfg.curpwd.passwd );
      cfg.curpwd.isactive = 1;
      if( cfg.cl & CL_PDICT )
         return send_work_pdict(sendbuf,send_len);
      else if( cfg.cl & CL_GDICT )
         return send_work_gdict(sendbuf,send_len);
   	break;
   case MSG_DO_PDICT:
      if(++pd.num_done<cfg.ptask_level)
         return send_work_pdict(sendbuf,send_len);
      // finished pdict
      else if( cfg.cl & CL_GDICT )
         return send_work_gdict(sendbuf,send_len);
      if( !more_work() )
         break;
      sendbuf[0] = MSG_INIT;
      return msg_proc(sendbuf,send_len);
   case MSG_DO_GDICT:
   	if(gd.curplen>cfg.maxplen)
      {
      // finished gdict
         if( !more_work() )
            break;
         sendbuf[0] = MSG_INIT;
         return msg_proc(sendbuf,send_len);
      }
      else 
               return send_work_gdict(sendbuf,send_len);
   default:
      print("invalid packet received.\n");
      return 0;
   }

   print("done.\n");
   alldone = 1;
   return 0;
}


///////////////////////////// commline_handler
int cl_proc( int argc, u_char **argv )
{
   static u_int i;

   cfg.cl = 0;
   cfg.cl_gdict = 0;
   cfg.gd_cset_size = 0;

   if(argc>2)
   {
      if( !(cfg.pwdfile = fopen(argv[--argc], "r")) )
      {
         printf("error: could not load passwd file: %s. exiting.\n\n", argv[argc] );
         return 0;
      }
      if( !more_work() )
      {
         printf("error: invalid passwd file: %s. exiting.\n\n", argv[argc] );
         return 0;
      }
      for( i=1; i<argc; i++ )
      {
         if( argv[i][0] == '-' )
         {
            switch( argv[i][1] )
            {
            case 'v':
               cfg.cl |= CL_VERBOSE;
               break;
            case 'l':
               cfg.cl |= CL_NO_LOG;
               break;
            case 'p':
               cfg.cl |= CL_PDICT;
               strcpy(cfg.dictfile_name, argv[++i]);
               break;
            case 'g':
               cfg.cl |= CL_GDICT;
               cfg.gtask_level = GTASK_LEVEL;
               for( i++; *argv[i]; argv[i]++ )
               {
                  switch( *argv[i] )
                  {
                  case 'a':
                     cfg.cl_gdict |= CL_GDICT_LCASE;
                     cfg.gd_cset_size += 26;
                     break;
                  case 'A':
                     cfg.cl_gdict |= CL_GDICT_UCASE;
                     cfg.gd_cset_size += 26;
                     break;
                  case '7':
                     cfg.cl_gdict |= CL_GDICT_DIGITS;
                     cfg.gd_cset_size += 10;
                     break;
                  case '?':
                     cfg.cl_gdict |= CL_GDICT_PUNCTUATION;
                     cfg.gd_cset_size += 33;
                     break;
                  case 'x':
                     cfg.cl_gdict |= CL_GDICT_EXTENDED;
                     cfg.gd_cset_size += 160;
                     break;
                  default:
                     goto jCOMM_HELP;
                  }
               }
               gd.curplen = cfg.minplen = atoi(argv[++i]);
               cfg.maxplen = atoi(argv[++i]);
               if( !(cfg.minplen && cfg.maxplen) )
                  goto jCOMM_HELP;
               break;
            case 'h':
            default:
               goto jCOMM_HELP;
            }
         }
         else
            goto jCOMM_HELP;
      }

      if( !(cfg.cl & CL_NO_LOG) )
      {
         u_char tstr[MAX_STR_LEN];
         sprintf( tstr, "%s.log", argv[argc] );
         if(!(cfg.logfile = fopen(tstr, "w")))
            printf("error: could not make logfile: %s. skipping.\n", tstr );
      }
      if( (cfg.cl & CL_GDICT) || (cfg.cl & CL_PDICT) )
         return 1;
      puts("error: -p or -g required.\n");
      return 0;
   }

jCOMM_HELP:
   printf(CL_HELP_MENU,TITLE);
   return 0;
}

///////////////////////////// exit_handler
int exit_proc( void )
{
   if( cfg.logfile )
      fclose(cfg.logfile);
   if( cfg.pwdfile )
      fclose(cfg.pwdfile);
   return 0;
}


///////////////////////////// print_handler
void print( const u_char *str, ... )
{
   static u_char tbuf[MAX_STR_LEN];
   va_list parg;
   
   va_start(parg, str);
   vsprintf(tbuf,str,parg);
   va_end(parg);
   if( cfg.cl & CL_VERBOSE )
      printf( tbuf );
   if( cfg.logfile && !(cfg.cl & CL_NO_LOG) )
      fputs( tbuf, cfg.logfile );
}

