/***********************************************************
master.c: network core for controlling nodes (node.c).
   Copyright (C) 1999. HaPpY (katami@hotmail.com).
   This source code is protected under the GPL. Read the
   README file for details.
***********************************************************/

/////////////////////////////
#include "master.h"


/////////////////////////////
#define NODES_FILE					"hosts.dat"


/////////////////////////////
typedef struct
{
   u_int num_nodes;
   Terminal *node[MAX_NODES];
   fd_set net_fds;
   struct timeval timeout;
} GlbData;


/////////////////////////////
static GlbData global;
const u_int *num_nodes = &global.num_nodes;
extern u_char *message_handler( u_char *, u_int *send_len );
extern int commline_handler( int, u_char ** );
extern int exit_handler( void );
extern void print_handler( const u_char *, ... );


/////////////////////////////
int init( void );
int connect_node( int id );
void disconnect_node( int id );
void error( const u_char *str, ... );
void clean_up( int exit_status );


/////////////////////////////
int init( void )
{
   FILE *fp;
   static unsigned id;
   static u_char str[MAX_STR_LEN];
   
   if(!(fp=fopen(NODES_FILE,"r")))
      return 0;
   for( id=0; !feof(fp) && id<MAX_NODES; id++ )
   {
      global.node[id] = (Terminal *)malloc(sizeof(Terminal));
      fscanf(fp,"%s",str );
      strcpy( global.node[id]->ipstr, str );
      global.node[id]->addr.sin_addr.s_addr = inet_addr(str);
      fscanf(fp,"%s\n", str );
      global.node[id]->addr.sin_port = atoi(str);
      global.node[id]->addr.sin_family = AF_INET;
      global.node[id]->socket = socket( AF_INET, SOCK_STREAM, 0 );
   }
   fclose(fp);

   global.timeout.tv_sec = CONNECT_TIMEOUT;
   global.timeout.tv_usec = 0;
   FD_ZERO(&global.net_fds);

   return global.num_nodes = id;
}

int connect_node( int id )
{
   if( !connect(global.node[id]->socket,
      (struct sockaddr *)&global.node[id]->addr,
      sizeof(global.node[id]->addr)) )
   {
      FD_SET(global.node[id]->socket, &global.net_fds);
      return 1;
   }
   disconnect_node(id);
   return 0;
}

void disconnect_node( int id )
{
   static int i;

   close(global.node[id]->socket);
   if( global.node[id]->socket != NULL_SOCKET )
      FD_CLR(global.node[id]->socket, &global.net_fds);
   free(global.node[id]);
   global.num_nodes--;
   for(i=id; i<global.num_nodes; i++)
      global.node[i] = global.node[i+1];
}

void error( const u_char *str, ... )
{
   static u_char tbuf[MAX_STR_LEN];
   static int id;
   va_list parg;
   
   va_start(parg, str);
   vsprintf(tbuf,str,parg);
   va_end(parg);
   print_handler( tbuf );

   clean_up(1);
}

void clean_up( int exit_status )
{
   while( global.num_nodes!=0 )
      disconnect_node(0);
   exit(exit_handler()+exit_status);
}


/////////////////////////////
char *get_time_str( u_int seconds )
{
   static char timestr[13];
   u_short sec, min, hour, day;

   sec = seconds%60;
   min = (seconds/60)%60;
   hour = (seconds/3600)%24;
   day = (seconds/86400)%365; 
   sprintf(timestr,"%d:%d:%d:%d",day,hour,min,sec);

   return timestr;
}

int main( int argc, u_char **argv )
{
   static unsigned id, init_time;

   global.num_nodes = 0;
   if(!commline_handler(argc,argv))
      clean_up(1);
   if(!init())
      error("error: could not load %s\n", NODES_FILE );


   for( id=0; id<global.num_nodes; )
   {
      print_handler("connecting to: %s %d: ",
         global.node[id]->ipstr,
         global.node[id]->addr.sin_port );
      if(!connect_node(id))
         print_handler("connect failed.\n");
      else
      {
         print_handler("successful.\n");
         id++;
      }
   }

   if(global.num_nodes == 0)
      error("no active nodes. exiting.\n");

   print_handler("%d nodes connected.\n", global.num_nodes );
   init_time = time(0);
   while(global.num_nodes>0)
   {
      static struct timeval tv;
      static fd_set fds;
      static u_char netbuf[MAX_PACKET_SIZE];
      tv  = global.timeout;
      fds  = global.net_fds;

      if( select(FD_SETSIZE, &fds, (fd_set *)0, (fd_set *)0, &tv) < 1 )
         error("error: server timed out. exiting.\n");
      for( id=0; id<global.num_nodes; )
      {
         if( FD_ISSET(global.node[id]->socket, &fds) )
         {
            netbuf[0] = 0;
            if( read(global.node[id]->socket, netbuf, MAX_PACKET_SIZE) >= 0 )
            {
               static u_char *pstr;
               static u_int len;
            	if( pstr = message_handler(netbuf,&len) )
              	   write( global.node[id++]->socket, pstr, len );
               else
               {
                  print("closing connection: %s %d.\n",
                     global.node[id]->ipstr,
                     global.node[id]->addr.sin_port );
                  disconnect_node(id);
               }
            }
            netbuf[0]=0;
         }
         else
            id++;
      }
   }

   print_handler("elapsed time: %s\n", get_time_str(time(0)-init_time) );
   clean_up(0);
}
