/* ui_stats.c - stats displays
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* callbacks */
void cb_save_slft(GtkWidget *widget, gpointer gdata);
void cb_save_bift(GtkWidget *widget, gpointer gdata);
void cb_save_trift(GtkWidget *widget, gpointer gdata);

static char *slft_display_titles[] = {"Letter", "Frequency"};
static char *bift_display_titles[] = {"First Letter", "Second Letter", "Frequency"};
static char *trift_display_titles[] = {"First Letter", "Second Letter", "Third Letter", "Frequency"};

/* Make a tabular display of slft using a GtkClist */
GtkWidget *make_slft_display(float *slft) {
    GtkWidget *clist;
    int i;
    char *row_data[2];
    char letter[2], value[30];
    
    clist = gtk_clist_new_with_titles(2, slft_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    for (i = 'A'; i <= 'Z'; i++) {
	sprintf(letter, "%c", (char) i);
	sprintf(value, "%.8f", slft[i]);
	row_data[0] = letter; row_data[1] = value;
	gtk_clist_append(GTK_CLIST(clist), row_data);
    }
    return clist;
}

/* Make a tabular display of bift using a GtkClist */
GtkWidget *make_bift_display(float *bift) {
    GtkWidget *clist;
    int i, j;
    char *row_data[3];
    char letter1[2], letter2[2], value[30];

    clist = gtk_clist_new_with_titles(3, bift_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    
	    sprintf(letter1, "%c", (char) i);
	    sprintf(letter2, "%c", (char) j);
	    sprintf(value, "%.8f", (bift + 26 * i)[j]);
	    row_data[0] = letter1; row_data[1] = letter2;  row_data[2] = value;
	    gtk_clist_append(GTK_CLIST(clist), row_data);
	}
    }
    return clist;
}
/* Make a tabular display of trift using a GtkClist */
GtkWidget *make_trift_display(float *trift) {
    GtkWidget *clist;
    float fvalue;
    int i, j, k;
    char *row_data[4];
    char letter1[2], letter2[2], letter3[2], value[30];

    clist = gtk_clist_new_with_titles(4, trift_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    for (k = 'A'; k <= 'Z'; k++) {
		fvalue = (trift + 26 * 26 * i + 26 * j)[k];
		if (fvalue > 0.0) {
		    sprintf(letter1, "%c", (char) i);
		    sprintf(letter2, "%c", (char) j);
		    sprintf(letter3, "%c", (char) k);
		    sprintf(value, "%.8f", fvalue);
		    row_data[0] = letter1; row_data[1] = letter2;  row_data[2] = letter3; row_data[3] = value;          
		    gtk_clist_append(GTK_CLIST(clist), row_data);
		}
	    }
	}
    }
    return clist;
}

GtkWidget *make_stats_summary(stats *the_stats) {
    char buf[256];
    GtkWidget *table;
    GtkWidget *save_button_slft;
    GtkWidget *save_button_bift;
    GtkWidget *save_button_trift;

    GtkWidget *label_slft_error;
    GtkWidget *label_slft_error_value;
    GtkWidget *label_bift_error;
    GtkWidget *label_bift_error_value;
    GtkWidget *label_trift_error;
    GtkWidget *label_trift_error_value;
    GtkWidget *label_total_error;
    GtkWidget *label_total_error_value;
    GtkWidget *label_letter_count;
    GtkWidget *label_letter_count_value;
    
    table = gtk_table_new(2,7, FALSE);
    
    save_button_slft = gtk_button_new_with_label("Save as Default Unigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_slft), "pressed", GTK_SIGNAL_FUNC(cb_save_slft), NULL);
    gtk_table_attach(GTK_TABLE(table), save_button_slft, 0, 1, 0, 1, 0, 0, 0, 0);
    gtk_widget_show(save_button_slft);

    save_button_bift = gtk_button_new_with_label("Save as Default Bigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_bift), "pressed", GTK_SIGNAL_FUNC(cb_save_bift), NULL);
    gtk_table_attach(GTK_TABLE(table), save_button_bift, 0, 1, 1, 2, 0, 0, 0, 0);
    gtk_widget_show(save_button_bift);

    save_button_trift = gtk_button_new_with_label("Save as Default Trigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_trift), "pressed", GTK_SIGNAL_FUNC(cb_save_trift), NULL);
    gtk_table_attach(GTK_TABLE(table), save_button_trift, 0, 1, 2, 3, 0, 0, 0, 0);
    gtk_widget_show(save_button_trift);
    
    sprintf(buf, "Unigram Error: ");
    label_slft_error = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_slft_error, 0, 1, 3, 4, 0, 0, 0, 0);
    gtk_widget_show(label_slft_error);

    sprintf(buf, "%f", the_stats->slft_error);
    label_slft_error_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_slft_error_value, 1, 2, 3, 4, 0, 0, 0, 0);
    gtk_widget_show(label_slft_error_value);
    
    sprintf(buf, "Bigram Error: ");
    label_bift_error = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_bift_error, 0, 1, 4, 5, 0, 0, 0, 0);
    gtk_widget_show(label_bift_error);

    sprintf(buf, "%f", the_stats->bift_error);
    label_bift_error_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_bift_error_value, 1, 2, 4, 5, 0, 0, 0, 0);
    gtk_widget_show(label_bift_error_value);

    sprintf(buf, "Trigram Error: ");
    label_trift_error = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_trift_error, 0, 1, 5, 6, 0, 0, 0, 0);
    gtk_widget_show(label_trift_error);

    sprintf(buf, "%f", the_stats->trift_error);
    label_trift_error_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_trift_error_value, 1, 2, 5, 6, 0, 0, 0, 0);
    gtk_widget_show(label_trift_error_value);

    sprintf(buf, "Total Error: ");
    label_total_error = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_total_error, 0, 1, 6, 7, 0, 0, 0, 0);
    gtk_widget_show(label_total_error);

    sprintf(buf, "%f", the_stats->total_error);
    label_total_error_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_total_error_value, 1, 2, 6, 7, 0, 0, 0, 0);
    gtk_widget_show(label_total_error_value);

    sprintf(buf, "Letter Count: ");
    label_letter_count = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_letter_count, 0, 1, 7, 8, 0, 0, 0, 0);
    gtk_widget_show(label_letter_count);

    sprintf(buf, "%d", the_stats->letter_count);
    label_letter_count_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_letter_count_value, 1, 2, 7, 8, 0, 0, 0, 0);
    gtk_widget_show(label_letter_count_value);

    return table;
}
